/**********************************************************************
 * Copyright (c) 2005 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.local.common;

import java.net.InetAddress;

import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;

/**
 * Please see TCPDataServer for implmentation details.
 */
public class MultiplexedDataServer implements AgentListener, Constants {
	private DataProcessor _processor;
	private long magicNumber = 0xFEEDC0DE;
	private long terminationCode = 0xDEADC0DE;
	private InetAddress inetaddr = null;
	private byte[] _messageHeader = new byte[MESSAGE_HEADER_LENGTH];
	private long _currentBufferSize = MAX_MESSAGE_LENGTH;
	private short _currentFlusherBuffer = 0;
	private int _currentHeaderOffset = 0;
	private int _bytesWritten = 0;

	/* Different types of data */
	public static final byte BINARY_DATA			= 0;
	public static final byte UTF8_STRING_DATA		= 1;
	public static final byte UNICODE_STRING_DATA	= 2;

	public void startServer(DataProcessor processor) throws Exception {
		_processor = processor;
	}

	public boolean isProcessing() {
		return true; // always processing
	}

	public void stopServer() {
	}

	public void resumeServer() {
	}

	public void resumeServer(DataProcessor processor) {
	}

	public void shutdownServer() {
	    this._processor = null;
	}

	public void incommingData(byte[] b, InetAddress peer) {
		_processor.incommingData(b, b.length, peer);
	}

	public void incommingData(char[] c, InetAddress peer) {
		_processor.incommingData(c, c.length, peer);
	}

	public void agentActive(Agent agent) {
		try {
			inetaddr = agent.getProcess().getNode().getInetAddress();
		} catch (InactiveProcessException e) {
		}
	}

	public void agentInactive(Agent agent) { 
	}

	public void error(Agent agent, String errorId, String errorMessage) {
	}

	public void handleCommand(Agent agent, CommandElement command) {
		switch((int)command.getTag()) {
		case (int)Constants.RA_BINARY_CUSTOM_COMMAND:
			BinaryCustomCommand binaryCustomCommand = (BinaryCustomCommand)command;
			int binaryDataLength;
			byte[] binaryData;

			// The byte array data and length
			binaryData = binaryCustomCommand.getDataBinary();
			binaryDataLength = binaryData.length;
			
			if (binaryDataLength >= sizeofLong) {
				// Read the magic number
				long code = Message.readRALongFromBuffer(binaryData, 0);

				if (code == magicNumber) {
					// Read the actual multiplexed data
					byte b[] = new byte[binaryDataLength - 4];
					System.arraycopy(binaryData, 4, b, 0, binaryDataLength - 4);
					//				String data = new String(binaryData, 4, binaryDataLength - 4);
					//				byte b[] = data.getBytes();
					processData(b, 0, b.length, inetaddr);
					//				try {
					//					incommingData(data.getBytes("UTF-8"), inetaddr);
					//				}
					//				catch(UnsupportedEncodingException e) {
					//					incommingData(data.getBytes(), inetaddr);
					//				}
				} else if (code == terminationCode) {
					/* Notify that the flusher is exiting */
					if (_processor instanceof DataServerListener) {
						((DataServerListener) _processor).dataServerExited();
					}
				}
			}
			break;
		}
	}

	protected int processData(byte[] data, int offset, int limit, InetAddress addr) {
		long messageLength;
		byte type;
		int current;
		byte[] _binaryForwardBuffer = new byte[MAX_MESSAGE_LENGTH];
		char[] _stringForwardBuffer=new char[MAX_MESSAGE_LENGTH];

		current = offset;

		/* Is there data to process */
		if(offset >= limit) {
			return limit;
		}

		/* Is this a new message? */
		if(_currentHeaderOffset < MESSAGE_HEADER_LENGTH) {
			/* Load the message header */
			current=this.loadMessageHeader(data, current, limit);

			/* Did we get the entire header, if not return */
			if(current == limit) {
				return current;
			}

			/* Resize and compress the forward buffer if nessesary */
			if(getMessageLength() >= _currentBufferSize) {
				type=getMessageType();

				if(type == BINARY_DATA || type == UTF8_STRING_DATA) {
					byte[] replacement = new byte[(int)getMessageLength()];
					/* Shift the available data to the front of the buffer */
					System.arraycopy(data, current, replacement, 0,(limit - current));
					_bytesWritten = limit-current;
					_binaryForwardBuffer = replacement;
				}
				else {
					char[] replacement = new char[(int)getMessageLength()];
					/* Shift the available data to the front of the buffer */
					for(int i = 0; i < limit - current + 1; i++) {
						replacement[i] = (char)data[i + current];
					}
					_bytesWritten = limit - current;
					_stringForwardBuffer = replacement;
				}
				return limit;
			}
		}

		/* Validate the message header, if we are in recovery
		   mode try and look at the next offset */
		if(!checkMessageMagic()) {
			System.out.println("Corrupt data");
			_currentHeaderOffset = 0;
			return processData(data, offset + 1, limit, addr);

		}


		/* How long is the current message */
		messageLength=getMessageLength();

		/* What is the message type */
		type=getMessageType();

		/* Process the entire buffer */
		while(current<limit) {

			if(type==BINARY_DATA || type==UTF8_STRING_DATA) {
				/* Copy as many bytes as possible into the forwarding buffer */
				while(current<limit && _bytesWritten<messageLength) {
					_binaryForwardBuffer[_bytesWritten++]=data[current++];
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
			else if(type==UNICODE_STRING_DATA){
				/* Copy as many bytes as possible into the forwarding buffer */
				while(offset<limit && _bytesWritten<messageLength) {
					_stringForwardBuffer[_bytesWritten>>1]=(char)((char)data[current++]
															| (char)(data[current++]<<8));
					_bytesWritten+=2;
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_stringForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
			else {
				/* Invalid message type */
				/* Copy as many bytes as possible into the forwarding buffer */
				while(offset<limit && _bytesWritten<messageLength) {
					_binaryForwardBuffer[_bytesWritten++]=data[current++];
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
		}
		return current;
	}

	protected int loadMessageHeader(byte[] data, int offset, int limit) {
		/* Load all we can into the header */
		while(offset<limit && _currentHeaderOffset < MESSAGE_HEADER_LENGTH) {
			_messageHeader[_currentHeaderOffset++] = data[offset++];
		}
		return offset;
	}

	/**
	 * Get the length of the current message
	 */
	protected long getMessageLength() {
		return Message.readRALongFromBuffer(_messageHeader, 5);
	}

	/**
	 * Get the message type.  There are currently three types
	    of messages.  BINARY_DATA, UTF8_STRING_DATA, UNICODE_STRING_DATA.
	 */
	protected byte getMessageType() {
		return _messageHeader[9];

	}

	/**
	 * Check the message magic number.  If the magic number is incorrect we need
	    to go into recovery mode
	 */
	protected boolean checkMessageMagic() {
		/* Check the message magic number */
		long messageKey=(long)(_messageHeader[0]<<24 & 0xff000000)
						| (long)(_messageHeader[1]<<16 & 0x00ff0000)
						| (long)_messageHeader[2]<<8 & 0x0000ff00
						| (long)_messageHeader[3];

		return true;
	}
}
