/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TCPDataServer.java,v 1.8 2005/04/28 16:27:51 slavescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.local.common;

import java.net.ServerSocket;
import java.net.Socket;
import java.net.InetAddress;
import java.net.SocketException;
import java.io.InputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.InterruptedException;
import java.util.Vector;

/**
 * Insert the type's description here.
 * Creation date: (9/15/00 12:57:03 PM)
 * @author:
 */
public class TCPDataServer {

	private ServerSocket 		_sock=null;
	private TCPDataProcessor	_server=null;
	private BufferFlusher   _flusher=null;
	private int				_port;
	private boolean _isDataServerRunning = true; // Bug 90153

	/* Buffers for getting incomming data off the
	   socket reader and onto a dispatch thread
	*/
	private static final short NUM_BUFFERS=32;
	private SingleBuffer[]	_bufferArray;
	private short			_currentFullBuffers=0;

	/* Different types of data */
	public static final byte BINARY_DATA			= 0;
	public static final byte UTF8_STRING_DATA		= 1;
	public static final byte UNICODE_STRING_DATA	= 2;

	private Vector _dataServerListeners=new Vector();

	private int totalBytes = 0; // Bug 90153

class SingleBuffer implements Constants {
	public InetAddress addr;
	public int length=0;
	public int size=MAX_MESSAGE_LENGTH;
	public byte[]  data=new byte[MAX_MESSAGE_LENGTH];

}

class TCPDataProcessor extends Thread implements Constants {
	ServerSocket  _socket;
	boolean _processing = true;
	private boolean _shutdown = false; /* 9707 */
	private short _currentFillerBuffer = 0;


	public void setSocket(ServerSocket sock) {
		_socket = sock;
	}

	/* 9707 */
	public void resumeProcessing() {
		synchronized(this) {
			_processing = true;
		}
	}

	/* 9707 */
	public void pauseProcessing() {
		synchronized(this) {
			_processing = false;
		}
	}

	/* 9707 */
	public boolean isProcessing() {
		synchronized(this) {
			return _processing;
		}
	}

	/* 9707 */
	public void shutdown() {
		_shutdown = true;
		pauseProcessing();
		try {
			_socket.close();
		} catch (IOException e) {
		}
		interrupt();
	}

	public void run() {
		_isDataServerRunning = true; // Bug 90153

		/* Run forever */
outer:	while(!_shutdown || (_currentFullBuffers != 0)) { /* 9707 */
			if(isProcessing()) {
				Socket incommingConnection = null;
				InputStream is = null;
				try {
					incommingConnection = _socket.accept();
					is = incommingConnection.getInputStream();
					incommingConnection.setSoTimeout(1000);
					//System.out.println("----Incomming connection");
				}
				catch(SocketException e) {
					/* The server socket is toast, stop processing */
					pauseProcessing();
					_isDataServerRunning = false; // Bug 90153
					continue;
				}
				catch(IOException e) {
					_isDataServerRunning = false; // Bug 90153
					continue;
				}
	
				while(true) {
						/* If all the buffers are full wait for the first one to be emptied */
						while(_currentFullBuffers == NUM_BUFFERS) {
							synchronized(_bufferArray[0]) {
								try {
									_bufferArray[0].wait();
								}
								catch(InterruptedException e) {
								}
							}
						}
	
						/* Fill the next buffer */
						_bufferArray[_currentFillerBuffer].addr = incommingConnection.getInetAddress();
						try {
							_bufferArray[_currentFillerBuffer].length = is.read(_bufferArray[_currentFillerBuffer].data);
							/*System.out.println("---Read "+_bufferArray[_currentFillerBuffer].length+" bytes");*/
							int _bytes = _bufferArray[_currentFillerBuffer].length; // Bug 90153
							totalBytes += _bytes;
							//System.out.println("---------------------Read "+_bytes+" into _bufferArray["+_currentFillerBuffer+"]");
							//System.out.println("---------------------Read "+totalBytes+" bytes in total");
							if (_bytes == -1) {
								_isDataServerRunning = false;
							}
						}
						catch(InterruptedIOException e) {
							/* Read timeout, don't want to wait too long */

							/* This is used to terminate the thread if the process is killed abnormally */
							if(_shutdown && (_currentFullBuffers == 0)) {
								pauseProcessing();
								_isDataServerRunning = false; // Bug 90153
								try {
									incommingConnection.close(); /* 9707 */
								} catch (IOException e1) {
									// Handle IOException
								}
								break outer;
							}
						}
						catch(IOException e) {
							/* Socket is toast, we are done */
							pauseProcessing();
							_isDataServerRunning = false; // Bug 90153
							try {
								incommingConnection.close(); /* 9707 */
							} catch (IOException e1) {
								// Handle IOException
							}
							break outer;
						}

						/* Is the connection closed? */
						if(_bufferArray[_currentFillerBuffer].length<0) {
							pauseProcessing();
							_isDataServerRunning = false; // Bug 90153
							try {
								/* Will hit here when detaching agent */
								incommingConnection.close();
							}
							catch(IOException e) {
							}
							break outer;
						}
						synchronized(_bufferArray[0]) {
							if(_bufferArray[_currentFillerBuffer].length>0) {
								/* Move on to the next buffer */
								_currentFillerBuffer++;
								if(_currentFillerBuffer == NUM_BUFFERS) {
									_currentFillerBuffer = 0;
								}
								_currentFullBuffers++;
	
								/* Is this the first buffer filled? */
								if(_currentFullBuffers==1) {
									_bufferArray[0].notifyAll();
								}
							}
						}
				}
			}
			else {
				try {
					/* Monitoring is stopped, keep this thread in sleep state */
					sleep(1000); /* 9707 */
				} catch (InterruptedException e) {
				}
			}
		}
	}

} /* end class TCPDataProcessor */


class BufferFlusher extends Thread implements Constants {
	private DataProcessor	_processor=null;
	private byte[]			_binaryForwardBuffer=new byte[MAX_MESSAGE_LENGTH];
	private char[]			_stringForwardBuffer=new char[MAX_MESSAGE_LENGTH];
	private byte[]			_messageHeader=new byte[MESSAGE_HEADER_LENGTH];
	private long				_currentBufferSize=MAX_MESSAGE_LENGTH;
	private short 			_currentFlusherBuffer=0;
	private int 			_currentHeaderOffset=0;
	private int				_bytesWritten=0;

	public void setProcessor(DataProcessor processor) {
		_processor=processor;
	}

	/** Load the _messageHeader from a buffer of data
	  * @param   data - the byte[] holding the raw data.
	  * @param offset - where the header starts in the raw data buffer.
	  * @param length - the length of the raw data buffer.
	  * @returns      - the new offset in the raw data buffer to continue
	  *                 reading from.
	  */
	protected int loadMessageHeader(byte[] data, int offset, int limit) {
		/* Load all we can into the header */
		while(offset<limit && _currentHeaderOffset<MESSAGE_HEADER_LENGTH) {
			_messageHeader[_currentHeaderOffset++]=data[offset++];
		}
		return offset;
	}

	/** Get the length of the current message
	  */
	protected long getMessageLength() {
		return Message.readRALongFromBuffer(_messageHeader, 5);
	}

	/** Get the message type.  There are currently three types
	    of messages.  BINARY_DATA, UTF8_STRING_DATA, UNICODE_STRING_DATA.
	  */
	protected byte getMessageType() {
		return _messageHeader[9];

	}

	/** Check the message magic number.  If the magic number is incorrect we need
	    to go into recovery mode
	  */
	protected boolean checkMessageMagic() {
		/* Check the message magic number */
		long messageKey=(long)(_messageHeader[0]<<24 & 0xff000000)
						| (long)(_messageHeader[1]<<16 & 0x00ff0000)
						| (long)_messageHeader[2]<<8 & 0x0000ff00
						| (long)_messageHeader[3];

		/* If the magic is incorect we need to go into recovery mode */

		/* TO_DO: RE-enable and test
		if(messageKey != RA_MAGIC ) {
			return false;
		}
		*/


		return true;
	}

	/** Recursively process a buffer of raw data.
	  */
	protected int processData(byte[] data, int offset, int limit, InetAddress addr) {
		long messageLength;
		byte type;
		int current;

		current=offset;

		/* Is there data to process */
		if(offset>=limit) {
			return limit;
		}

		/* Is this a new message? */
		if(_currentHeaderOffset<MESSAGE_HEADER_LENGTH) {
			/* Load the message header */
			current=this.loadMessageHeader(data, current, limit);

			/* Did we get the entire header, if not return */
			if(current==limit) {
				return current;
			}


			/* Resize and compress the forward buffer if nessesary */
			if(getMessageLength() >= _currentBufferSize) {
				type=getMessageType();

				if(type==BINARY_DATA || type==UTF8_STRING_DATA) {
					byte[] replacement=new byte[(int)getMessageLength()];
					/* Shift the available data to the front of the buffer */
					System.arraycopy(data, current, replacement, 0,(limit - current));
					_bytesWritten=limit-current;
					_binaryForwardBuffer=replacement;
				}
				else {
					char[] replacement=new char[(int)getMessageLength()];
					/* Shift the available data to the front of the buffer */
					for(int i=0; i<limit-current+1; i++) {
						try {
							replacement[i]=(char)data[i+current];
						} catch (Exception e) {
							System.out.println("BufferFlusher.processData(): replacement[i]=(char)data[i+current];");
							System.out.println("BufferFlusher.processData(): repalcement.length="+replacement.length+", data.length="+data.length+", i="+i+", current="+current);
							e.printStackTrace(System.out);
							throw new RuntimeException(e);
						}
					}
					_bytesWritten=limit-current;
					_stringForwardBuffer=replacement;
				}
				return limit;
			}
		}

		/* Validate the message header, if we are in recovery
		   mode try and look at the next offset */
		if(!checkMessageMagic()) {
			System.out.println("Corrupt data");
			_currentHeaderOffset=0;
			return processData(data, offset+1, limit, addr);

		}


		/* How long is the current message */
		messageLength=getMessageLength();

		/* What is the message type */
		type=getMessageType();

		/* Process the entire buffer */
		while(current<limit) {

			if(type==BINARY_DATA || type==UTF8_STRING_DATA) {
				/* Copy as many bytes as possible into the forwarding buffer */
				while(current<limit && _bytesWritten<messageLength) {
					_binaryForwardBuffer[_bytesWritten++]=data[current++];
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
			else if(type==UNICODE_STRING_DATA){
				/* Copy as many bytes as possible into the forwarding buffer */
				while(offset<limit && _bytesWritten<messageLength) {
					_stringForwardBuffer[_bytesWritten>>1]=(char)((char)data[current++]
															| (char)(data[current++]<<8));
					_bytesWritten+=2;
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_stringForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
			else {
				/* Invalid message type */
				/* Copy as many bytes as possible into the forwarding buffer */
				while(offset<limit && _bytesWritten<messageLength) {
					_binaryForwardBuffer[_bytesWritten++]=data[current++];
				}
				/* Are we at the end of the message? If so forward to the handler */
				if(_bytesWritten==messageLength) {
					_processor.incommingData(_binaryForwardBuffer, _bytesWritten, addr);
					_bytesWritten=0;
					_currentHeaderOffset=0;
					/* Continue processing this data buffer */
					current=this.processData(data, current, limit, addr);
				}
			}
		}
		return current;
	}

	public void run() {
//outer:	while(isProcessing() || (_currentFullBuffers != 0) || _isFinished == false) { /* 237169 make sure buffer is empty before exiting */
outer:	while(isProcessing() || (_currentFullBuffers != 0)) { // Bug 90153
			/* If there are no current buffers to empty wait */
			if(_currentFullBuffers==0) {
				_processor.waitingForData();
				do {
					synchronized(_bufferArray[0]) {
						try {
							_bufferArray[0].wait(1000);
						}
						catch(InterruptedException e) {
							return;
						}
					}
					if(!isProcessing() && _currentFullBuffers==0) {
						break outer;
					}
				}
				while(_currentFullBuffers==0);
			}

			/* Empty the current buffer */
			if(_bufferArray[_currentFlusherBuffer].length>0) {
				//System.out.println("---- Flushing "+ _bufferArray[_currentFlusherBuffer].length+" bytes from _bufferArray["+_currentFlusherBuffer+"]");
				processData(_bufferArray[_currentFlusherBuffer].data,
							0,
							_bufferArray[_currentFlusherBuffer].length,
							_bufferArray[_currentFlusherBuffer].addr);
				/* Mark the buffer as empty */
				_bufferArray[_currentFlusherBuffer].length=0;

			}

			synchronized(_bufferArray[0]) {

				_currentFullBuffers--;

				/* Increment the flusher to the next buffer */
				_currentFlusherBuffer++;
				if(_currentFlusherBuffer==NUM_BUFFERS) {
					_currentFlusherBuffer=0;
				}

				/* If the buffers were half full before this flush notify the
				   filler it can continue. Generally this could be NUMBUFFERS
				   but not as efficient as more thread switches happen
				*/
				if(_currentFullBuffers==0) {
					_bufferArray[0].notifyAll();
				}
			}
		}


		/* Notify that the flusher is exiting */
		//System.out.println("Notify that the flusher is exiting!!!");
		//System.out.println("isProcessing() = "+isProcessing()+", _currentFullBuffers = " +_currentFullBuffers);
		if(_processor instanceof DataServerListener) {
			((DataServerListener)_processor).dataServerExited();
		}
	}

	public void shutdown() {
		//interrupt(); // Bug 90153
	}

} /* end class RingBufferFiller */

/**
 * TCPDataServer constructor comment.
 */
public TCPDataServer() {
	super();
}
	public int getPort() {
		return _port;
	}
	public InetAddress getServerAddress() {
		return _sock.getInetAddress();
	}
	public void startServer(DataProcessor processor, int port) throws SocketException, IOException {
		/* Create the buffering mechanism */
		_bufferArray=new SingleBuffer[NUM_BUFFERS];
		for(int i=0; i<NUM_BUFFERS; i++)
			_bufferArray[i]=new SingleBuffer();

		/* Start the server */
		_sock=new ServerSocket(port);

		_port=_sock.getLocalPort();

		_server=new TCPDataProcessor();
		_server.setSocket(_sock);
		_server.setName("TCPDataFiller");

		/* Set up the data flusher */
		_flusher=new BufferFlusher();
		_flusher.setProcessor(processor);
		_flusher.setName("TCPDataFlusher");

		/* Don't block exit of the VM */
		_server.setDaemon(true);
		_flusher.setDaemon(true);

		/* RKD:
		   Because these two threads intensely contend for the same resources
		   we need to increase the priority of the flusher, otherwise I have
		   noticed that the flusher doesn't run often enough and the data sits
		   in the buffers far too long.  A relitive increase of 3 seems to much
		   better */

		_flusher.setPriority(_server.getPriority()+3);

		_server.start();
		_flusher.start();
	}

	public void startServer(DataProcessor processor) throws SocketException, IOException {
		startServer(processor, 0);
	}

	public boolean isProcessing() {
		return _isDataServerRunning; // Bug 90153
//		return _server.isAlive();
	}

	public void stopServer() {
		_server.pauseProcessing(); /* 9707 */
	}

	/* 9707 */
	public void resumeServer() {
		_server.resumeProcessing();
	}

	/* 9707 */
	public void resumeServer(DataProcessor processor) {
		_flusher.setProcessor(processor);
		_server.resumeProcessing();
	}

	/* 9707 */
	public void shutdownServer() {
		//System.out.println("TCPDataServer is shutted down!");
		_server.shutdown();
		_flusher.shutdown();
	}
}
