/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileManagerExtendedTest.java,v 1.4 2005/06/07 19:54:09 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.test.execution.local;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;
import java.util.Random;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.hyades.execution.core.DaemonConnectException;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.local.NodeImpl;

/**
 * File transfer test to exercise the new file manager extended implementation
 * via the file manager extended class and interface. This class is usable
 * outside of Eclipse as a standalone test. Some test files are generated first
 * and then these test files are operated on using the new file manager
 * implementation. The test files are generated with exponentially increasing
 * lengths to give a good sample of various file lengths. File name plurality,
 * progress monitor support and various additional operational semantic
 * fine-tuning is tested with this class. Additional javadocs can be found in
 * the IFileManagerExtended interface definition. Timing of file transfer
 * operations will also be conducted to demonstrate that file transfer
 * performance has increased about 50 times with the minimum transfer time taken
 * no longer starting at approximately 3s. In a typical test, a typical Java
 * class takes 50ms to transfer to the remote host and a large 16MB file takes
 * about 3s. File transfer in the opposite direction, to the local host, is
 * comparable.
 * 
 * @author Scott E. Schneider
 */
public class FileManagerExtendedTest {

    private static final int FILE_COUNT = 10;

    private static final int FILE_SAMPLE_SIZE = 10;

    private static final String HEADER = "File transfer test with a file of size: ";

    private static final String HOST = "127.0.0.1";

    private static final int ITERATIONS = 10000;

    private static final int OFFSET = 4096;
    
    private static final int COMMAND_INTERVAL_SLEEP = 1000;

    /**
     * Number of concurrent threads that will run an instance of this test, no
     * duplicate filenames will be used
     */
    private static final int CONCURRENT_CLIENT_THREADS = 1;

    /**
     * The number or worker threads that loop over the test body (not creating
     * new files with different names) for each concurrent client thread as
     * defined by CONCURRENT_CLIENT_THREADS
     */
    private static final int CONCURRENT_CLIENT_WORKER_THREADS = 1;

    /**
     * Delay to use before a new instance of this test on a concurrent thread is
     * started, used to stagger to operations a bit
     */
    private static final int THREAD_START_DELAY = 1250;

    public static void main(String[] args) {
        for (int i = 0; i < FileManagerExtendedTest.CONCURRENT_CLIENT_THREADS; i++) {
            new Thread() {
                public void run() {
                    new FileManagerExtendedTest().execute();
                }
            }.start();
            synchronized (Thread.currentThread()) {
                try {
                    Thread.currentThread().wait(FileManagerExtendedTest.THREAD_START_DELAY);
                } catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        }
    }

    private File createAndPopulateFile(String name, String extension, int size) throws IOException {
        File temp = File.createTempFile(name, extension);
        RandomAccessFile file = new RandomAccessFile(temp, "rw");
        FileChannel channel = file.getChannel();
        ByteBuffer byteBuffer = ByteBuffer.allocateDirect(size);
        byteBuffer.clear();
        String string = FileManagerExtendedTest.HEADER + size;
        byteBuffer.put(string.getBytes("UTF-8"));
        byteBuffer.put(this.getFixedLengthRandomBytes(size - string.length()));
        byteBuffer.flip();
        channel.position(0);
        channel.write(byteBuffer);
        channel.close();
        file.close();
        return temp;
    }

    private FileIdentifierList createDestinationIdentifiers(FileIdentifierList sourceIdentifiers) {
        FileIdentifierList destinationIdentifiers = FileIdentifierList.create();
        for (Iterator iterator = sourceIdentifiers.iterator(); iterator.hasNext();) {
            String fileIdentifier = (String) iterator.next();
            int extension = fileIdentifier.lastIndexOf(".");
            destinationIdentifiers.add(fileIdentifier.substring(0, extension).concat("-destination")
                    + fileIdentifier.substring(extension));
        }
        return destinationIdentifiers;
    }

    public void execute() {

        // Create a node pointing to local machine
        NodeImpl node = new NodeImpl(FileManagerExtendedTest.HOST);

        // Establish a session by connecting to the node
        ISession session = null;
        try {
            session = node.connect("10002", null, FileManagerExtendedTest.ITERATIONS);
        } catch (UnknownHostException e) {
        } catch (UnknownDaemonException e) {
        } catch (DaemonConnectException e) {
        }

        // Retrieve the file manager implementation from the file manager
        IFileManager fileManager = ((NodeImpl) node).getFileManager();

        // Assert that the new file manager extended implementation is available
        Assert.isTrue(fileManager instanceof IFileManagerExtended);
        final IFileManagerExtended manager = (IFileManagerExtended) fileManager;

        try {

            // Generate test files used for the file manager operations
            final FileIdentifierList sourceFileIdentifiers = this.generateSourceTestFiles();
            final FileIdentifierList destinationFileIdentifiers = this.createDestinationIdentifiers(sourceFileIdentifiers);

            // Use client worker threads to test overlapping file transfer
            for (int i = 0; i < FileManagerExtendedTest.CONCURRENT_CLIENT_WORKER_THREADS; i++) {
                new Thread() {
                    public void run() {

                        try {

                            // Assert that the server is available
                            Assert.isTrue(manager.isServerAvailable());
                            
                            Thread.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

                            // Upload files to server (without using any options)
                            manager.putFile(sourceFileIdentifiers, destinationFileIdentifiers);
                            
                            Thread.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

                            // Download files to client side (without using any options)
                            manager.getFile(sourceFileIdentifiers, destinationFileIdentifiers);
                            
                            Thread.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

                            // Delete files from the remote host (without using any options)
                            manager.deleteFile(destinationFileIdentifiers);
                            
                            Thread.sleep(FileManagerExtendedTest.COMMAND_INTERVAL_SLEEP);

                        } catch (Throwable t) {
                            t.printStackTrace();
                        }

                    }
                }.start();
            }

        } catch (IOException e) {

            // Dump stack trace if issues arise
            e.printStackTrace();

        }

        // Be a responsible TPTP-citizen and release the session
        session.release();

    }

    private FileIdentifierList generateSourceTestFiles() throws IOException {

        // Initialize new file identifier list to represent the file identities
        FileIdentifierList generatedFiles = FileIdentifierList.create();

        // First create test files to use in file transfer testing
        File[] files = new File[FileManagerExtendedTest.FILE_COUNT];
        for (int i = 0; i < FileManagerExtendedTest.FILE_COUNT; i++) {
            int size = (int) Math.pow(2, i) + FileManagerExtendedTest.OFFSET;
            for (int j = 0; j < FileManagerExtendedTest.FILE_SAMPLE_SIZE; j++) {
                files[i] = this.createAndPopulateFile("file_" + j + "-" + String.valueOf(size - FileManagerExtendedTest.OFFSET)
                        + "+" + String.valueOf(FileManagerExtendedTest.OFFSET) + "__", ".dummy", size);
                generatedFiles.add(files[i]);
            }
        }

        return generatedFiles;

    }

    private byte[] getFixedLengthRandomBytes(int count) {
        Random random = new Random(System.currentTimeMillis());
        byte[] bytes = new byte[count];
        random.nextBytes(bytes);
        return bytes;
    }

}