package org.eclipse.hyades.logging.events.cbe.impl.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConnectSituationImplTest.java,v 1.3 2005/02/16 22:20:40 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * JUnit testcase for the <code>ConnectSituationImpl</code> class.
 * 
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1
 */
public class ConnectSituationImplTest extends TestCase {

    /** Target instance for the tests. */
    private ConnectSituation sit;

    /**
     * Constructor for ConnectSituationImplTest.
     * @param testcase
     */
    public ConnectSituationImplTest(String testcase) {
        super(testcase);
    }

    /**
     * Tests whether <code>toString</code> doesn't blow up.
     */
    public void testToString() {
        assertNotNull(sit.toString());
    }

    /**
     * Tests presence of attributes in the public interface
     */
    public void testInterface() {
        sit.getReasoningScope();
        sit.getSituationDisposition();
        sit.getSuccessDisposition();
        // setters show on other tests

        assertTrue(sit instanceof SituationType);
    }

    /**
     * Tests the serialization for <code>AvailableSituation</code>.
     */
    public void testSerialization() throws Exception {
        Util.testEObjectSerialization(sit);
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        ConnectSituation sit2 = (ConnectSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit.setReasoningScope("r");
        sit.setSituationDisposition("s");
        sit.setSuccessDisposition("d");
        sit2 = (ConnectSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit2.setSuccessDisposition("difference");
        assertFalse(sit.equals(sit2));
    }

    /**
     * Tests validation for <code>AvailableSituation</code>.
     */
    public void testValidationMissingAttributes() throws ValidationException {
        Util.assertMissingAttributes(sit, "all");

        sit.setReasoningScope("r");
        sit.setSituationDisposition("s");
        sit.setSuccessDisposition("d");
        sit.validate();

        sit.setReasoningScope(null);
        Util.assertMissingAttributes(sit, "reasoningScope");

        sit.setReasoningScope("r");
        sit.setSituationDisposition(null);
        sit.setSuccessDisposition("d");
        Util.assertMissingAttributes(sit, "situationDisposition");

        sit.setReasoningScope("r");
        sit.setSituationDisposition("s");
        sit.setSuccessDisposition(null);
        Util.assertMissingAttributes(sit, "successDisposition");

    }

    /**
     * Tests validation for <code>AvailableSituation</code>.
     */
    public void testValidationInvalidLengths() {
        sit.setReasoningScope(Util.getPaddedString(64));
        sit.setSituationDisposition(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        try {
            sit.validate();
        }
        catch (ValidationException e) {
            fail("Strings have correct length");
        }

        sit.setReasoningScope(Util.getPaddedString(65));
        sit.setSituationDisposition(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "reasoningScope");

        sit.setReasoningScope(Util.getPaddedString(64));
        sit.setSituationDisposition(Util.getPaddedString(65));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "situationDisposition");

        sit.setReasoningScope(Util.getPaddedString(64));
        sit.setSituationDisposition(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(65));
        Util.assertInvalidBoundaries(sit, "successDisposition");
    }

    /**
     * 
     */
    public void testAdapter() {
        Util.assertAdapterNotification(sit);
    }

    /**
     * 
     */
    public void testUnsupportedFeature() {
        Util.assertUnsupportedFeature(sit);
    }

    /*
     * Protected methods.
     */

    /* (non-Javadoc)
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
        EventFactory eFx = EventFactory.eINSTANCE;
        sit = eFx.createConnectSituation();

    }
}
