package org.eclipse.hyades.logging.events.cbe.impl.tests;

import java.util.Arrays;

import junit.framework.TestCase;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MsgDataElementImplTest.java,v 1.6 2005/02/16 22:20:40 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * 
 * @author Andrea Chiavegatto
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class MsgDataElementImplTest extends TestCase {
	
    private MsgDataElement mde = null;
    
    /**
     * Constructor for MsgDataElementImplTest.
     * @param testcase
     */
    public MsgDataElementImplTest(String testcase) {
        super(testcase);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        mde = EventFactory.eINSTANCE.createMsgDataElement();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
    }

    public void testMultipleMsgCatalogTokensWithSameContents() {

        MsgCatalogToken msgCatalogToken = null;

        for (int counter = 0; counter < 5; counter++) {

            msgCatalogToken = EventFactory.eINSTANCE.createMsgCatalogToken();
            msgCatalogToken.setValue("msgCatalogToken");

            mde.addMsgCatalogToken(msgCatalogToken);
        }

        assertTrue(mde.getMsgCatalogTokens().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((MsgCatalogToken) (mde.getMsgCatalogTokens().get(counter))).getValue(), "msgCatalogToken");
        }
    }

    public void testMultipleMsgCatalogTokensWithSameObjects() {
       
        MsgCatalogToken msgCatalogToken = EventFactory.eINSTANCE.createMsgCatalogToken();
        msgCatalogToken.setValue("msgCatalogToken");

        for (int counter = 0; counter < 5; counter++) {
            mde.addMsgCatalogToken(msgCatalogToken);
        }

        //NOTE:  The resultant EList only contains one element since the same 
        //object instance cannot be duplicated in the same EList more than once since the 
        //lifetime of the contained object is controlled by the container object.
        assertTrue(mde.getMsgCatalogTokens().size() == 1);

        assertEquals(((MsgCatalogToken) (mde.getMsgCatalogTokens().get(0))).getValue(), "msgCatalogToken");
    }

    public void testMultipleMsgCatalogTokensArrayWithSameContents() {
       
        MsgCatalogToken[] msgCatalogTokens = new MsgCatalogToken[5];
        
        for (int counter = 0; counter < 5; counter++) {

            msgCatalogTokens[counter] = EventFactory.eINSTANCE.createMsgCatalogToken();
            msgCatalogTokens[counter].setValue("msgCatalogToken");
        }
        
        mde.setMsgCatalogTokens(msgCatalogTokens);

        assertTrue(mde.getMsgCatalogTokens().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((MsgCatalogToken) (mde.getMsgCatalogTokens().get(counter))).getValue(), "msgCatalogToken");
        }
    }

    public void testGetMsgIdType() {
        assertNull(mde.getMsgIdType());

        String input = "Testing getMsgIdType()";
        mde.setMsgIdType(input);

        assertEquals(input, mde.getMsgIdType());
    }

    public void testSetMsgIdType() {
        String input = "Testing setMsgIdType()";
        mde.setMsgIdType(input);
        assertEquals(input, mde.getMsgIdType());

        input = "setting another string";
        mde.setMsgIdType(input);
        assertEquals(input, mde.getMsgIdType());
    }

    public void testGetMsgCatalogType() {
        assertNull(mde.getMsgCatalogType());

        String input = "Testing getMsgCatalogType()";
        mde.setMsgCatalogType(input);

        assertEquals(input, mde.getMsgCatalogType());
    }

    public void testSetMsgCatalogType() {
        String input = "Testing setMsgCatalogType()";
        mde.setMsgCatalogType(input);
        assertEquals(input, mde.getMsgCatalogType());

        input = "setting another string";
        mde.setMsgCatalogType(input);
        assertEquals(input, mde.getMsgCatalogType());
    }

    public void testGetMsgCatalog() {
        assertNull(mde.getMsgCatalog());

        String input = "Testing getMsgCatalog()";
        mde.setMsgCatalog(input);

        assertEquals(input, mde.getMsgCatalog());
    }

    public void testSetMsgCatalog() {
        String input = "Testing setMsgCatalog()";
        mde.setMsgCatalog(input);
        assertEquals(input, mde.getMsgCatalog());

        input = "setting another string";
        mde.setMsgCatalog(input);
        assertEquals(input, mde.getMsgCatalog());
    }

    public void testGetMsgCatalogId() {
        assertNull(mde.getMsgCatalogId());

        String input = "Testing getMsgCatalogId()";
        mde.setMsgCatalogId(input);

        assertEquals(input, mde.getMsgCatalogId());
    }

    public void testSetMsgCatalogId() {
        String input = "Testing setMsgCatalogId()";
        mde.setMsgCatalogId(input);
        assertEquals(input, mde.getMsgCatalogId());

        input = "setting another string";
        mde.setMsgCatalogId(input);
        assertEquals(input, mde.getMsgCatalogId());
    }

    public void testGetMsgId() {
        assertNull(mde.getMsgId());

        String input = "Testing getMsgId()";
        mde.setMsgId(input);

        assertEquals(input, mde.getMsgId());
    }

    public void testSetMsgId() {
        String input = "Testing setMsgId()";
        mde.setMsgId(input);
        assertEquals(input, mde.getMsgId());

        input = "setting another string";
        mde.setMsgId(input);
        assertEquals(input, mde.getMsgId());
    }

    public void testGetMsgLocale() {
        assertNull(mde.getMsgLocale());

        String input = "Testing getMsgLocale()";
        mde.setMsgLocale(input);

        assertEquals(input, mde.getMsgLocale());
    }

    public void testSetMsgLocale() {
        String input = "Testing setMsgLocale()";
        mde.setMsgLocale(input);
        assertEquals(input, mde.getMsgLocale());

        input = "setting another string";
        mde.setMsgLocale(input);
        assertEquals(input, mde.getMsgLocale());
    }

    public void testGetMsgCatalogTokens() {
        assertTrue(mde.getMsgCatalogTokens().isEmpty());
        String[] values = { "child1", "child2" };
        mde.setMsgCatalogTokensAsStrings(values);
        assertEquals(2, mde.getMsgCatalogTokens().size());
    }

    public void testAddMsgCatalogToken() {
        // test get & add
        MsgCatalogToken mct1 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct1.setValue("test1");
        MsgCatalogToken mct2 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct2.setValue("test2");
        mde.addMsgCatalogToken(mct1);
        mde.addMsgCatalogToken(mct2);
        EList outValue = mde.getMsgCatalogTokens();
        assertEquals(2, outValue.size());
        assertTrue(outValue.contains(mct1));
        assertTrue(outValue.contains(mct2));
    }

    public void testAddMsgCatalogTokenAsString() {
        // test get & add
        String inValue = "test1";
        String inValue2 = "test2";
        mde.addMsgCatalogTokenAsString(inValue);
        mde.addMsgCatalogTokenAsString(inValue2);
        String[] outValue = mde.getMsgCatalogTokensAsStrings();
        assertEquals(2, outValue.length);
        assertTrue(Arrays.asList(outValue).contains(inValue));
        assertTrue(Arrays.asList(outValue).contains(inValue2));
    }

    public void testGetMsgCatalogTokensAsStrings() {
        // test uninitialized object
        assertEquals(0, mde.getMsgCatalogTokensAsStrings().length);

        // test get & set
        String[] inValue = { "test1", "test2", "test3", "test4" };
        mde.setMsgCatalogTokensAsStrings(inValue);
        String[] outValue = mde.getMsgCatalogTokensAsStrings();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));
    }

    public void testSetMsgCatalogTokensAsStrings() {
        // test get & set
        String[] inValue = { "test1", "test2", "test3" };
        mde.setMsgCatalogTokensAsStrings(inValue);
        String[] outValue = mde.getMsgCatalogTokensAsStrings();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // do it again to see if the value stored in the Extended Data Element object is replaced
        String[] inValue2 = { "test4", "test5" };
        mde.setMsgCatalogTokensAsStrings(inValue2);
        outValue = mde.getMsgCatalogTokensAsStrings();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue2)));
    }

    public void testSetMsgCatalogTokens() {
        // test get & set
        MsgCatalogToken mct1 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct1.setValue("test1");
        MsgCatalogToken mct2 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct2.setValue("test2");
        MsgCatalogToken[] inValue = { mct1, mct2 };
        mde.setMsgCatalogTokens(inValue);
        assertTrue(mde.getMsgCatalogTokens().containsAll(Arrays.asList(inValue)));

        // do it again to see if the value stored in the Extended Data Element object is replaced
        MsgCatalogToken mct3 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct3.setValue("another test1");
        MsgCatalogToken mct4 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct4.setValue("another test2");
        MsgCatalogToken[] inValue2 = { mct3, mct4 };
        mde.setMsgCatalogTokens(inValue2);
        assertTrue(mde.getMsgCatalogTokens().containsAll(Arrays.asList(inValue2)));
    }

    public void testSerialize() throws Exception {
        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("A");
        mde.setMsgCatalogId("B");
        mde.setMsgCatalogType("C");
        mde.setMsgId("D");
        mde.setMsgIdType("E");
        mde.setMsgLocale("L");
        MsgCatalogToken mct = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct.setValue("value1");
        mde.getMsgCatalogTokens().add(mct);
        mct = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct.setValue("value2");
        mde.getMsgCatalogTokens().add(mct);

        Util.testEObjectSerialization(mde);
    }

    /**
     * Tests whether the model has been generated in a way that 
     * allows duplicate values inside a list.
     */
    public void testDuplicateTokens() {
        EventFactory eFx = EventFactory.eINSTANCE;
        MsgDataElement mde = eFx.createMsgDataElement();
        mde.addMsgCatalogTokenAsString("1");
        mde.addMsgCatalogTokenAsString("1");
        assertEquals(2, mde.getMsgCatalogTokens().size());
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        MsgDataElement mde2 = (MsgDataElement) Util.clone(mde);
        assertEquals(mde, mde2);

        mde2.setMsgIdType("difference");
        assertFalse(mde.equals(mde2));
    }

    /**
     * Tests whether <code>toString</code> doesn't blow up.
     */
    public void testToString() {
        assertNotNull(mde.toString());
    }

    /**
     * Tests for missing required attributes.
     */
    public void testValidationMissingAttributes() throws ValidationException {
        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.validate();

        // msgId, msgIdType
        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgId("my msg id");
        mde.setMsgIdType("my msg id type");
        mde.validate();

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        //mde.setMsgId(Util.getPaddedString(256));
        mde.setMsgIdType(Util.getPaddedString(32));
        try {
            mde.validate();
            fail("missing msgId attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgId(Util.getPaddedString(256));
        //mde.setMsgIdType(Util.getPaddedString(32));
        try {
            mde.validate();
            fail("missing msgIdType attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        // msgCatalog, msgCatalogId, msgCatalogType
        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("my catalog");
        mde.setMsgCatalogId("my catalog id");
        mde.setMsgCatalogType("my catalog type");
        mde.validate();

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("my catalog");
        mde.setMsgCatalogId("my catalog id");
        //mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalogType attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("my catalog");
        //mde.setMsgCatalogId("my catalog id");
        mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalogId attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("my catalog");
        //mde.setMsgCatalogId("my catalog id");
        //mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalog... attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        //mde.setMsgCatalog("my catalog");
        mde.setMsgCatalogId("my catalog id");
        mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalof attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        //mde.setMsgCatalog("my catalog");
        //mde.setMsgCatalogId("my catalog id");
        mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalog... attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }

        mde = EventFactory.eINSTANCE.createMsgDataElement();
        //mde.setMsgCatalog("my catalog");
        mde.setMsgCatalogId("my catalog id");
        //mde.setMsgCatalogType("my catalog type");
        try {
            mde.validate();
            fail("missing msgCatalog... attribute");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0209E", exc);
        }
    }

    /**
     * Tests for invalid length attributes.
     */
    public void testValidationInvalidLengths() {
        // msgId and msgIdType go together
        mde.setMsgId(Util.getPaddedString(256));
        mde.setMsgIdType(Util.getPaddedString(32));
        Util.assertValidBoundaries(mde);
        mde.setMsgId(Util.getPaddedString(257));
        Util.assertInvalidBoundaries(mde, "msgId");
        mde.setMsgId(Util.getPaddedString(256));
        Util.assertValidBoundaries(mde);
        mde.setMsgIdType(Util.getPaddedString(33));
        Util.assertInvalidBoundaries(mde, "msgIdType");
        mde.setMsgIdType(Util.getPaddedString(32));
        Util.assertValidBoundaries(mde);

        // msgCatalog, msgCatalogType, and msgCatalogId go together
        mde.setMsgCatalogId(Util.getPaddedString(64));
        mde.setMsgCatalog(Util.getPaddedString(128));
        mde.setMsgCatalogType(Util.getPaddedString(32));
        Util.assertValidBoundaries(mde);
        mde.setMsgCatalogId(Util.getPaddedString(65));
        Util.assertInvalidBoundaries(mde, "msgCatalogId");
        mde.setMsgCatalogId(Util.getPaddedString(64));
        Util.assertValidBoundaries(mde);
        mde.setMsgCatalog(Util.getPaddedString(129));
        Util.assertInvalidBoundaries(mde, "msgCatalog");
        mde.setMsgCatalog(Util.getPaddedString(128));
        Util.assertValidBoundaries(mde);
        mde.setMsgCatalogType(Util.getPaddedString(33));
        Util.assertInvalidBoundaries(mde, "msgCatalogType");
        mde.setMsgCatalogType(Util.getPaddedString(32));
        Util.assertValidBoundaries(mde);

        // msgLocale
        mde.setMsgLocale(Util.getPaddedString(11));
        Util.assertValidBoundaries(mde);
        mde.setMsgLocale(Util.getPaddedString(12));
        Util.assertInvalidBoundaries(mde, "msgLocale");
        mde.setMsgLocale(Util.getPaddedString(11));
        Util.assertValidBoundaries(mde);

        //msgCatalogTokens
        MsgCatalogToken mct1 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct1.setValue(Util.getAlphaString(256));
        MsgCatalogToken mct2 = EventFactory.eINSTANCE.createMsgCatalogToken();
        mct2.setValue(Util.getAlphaString(256));
        MsgCatalogToken[] inValue = { mct1, mct2 };
        mde.setMsgCatalogTokens(inValue);
        Util.assertValidBoundaries(mde);

        mct2.setValue(Util.getAlphaString(257));
        mde.setMsgCatalogTokens(inValue);
        Util.assertInvalidBoundaries(mde, "msgCatalogTokens");
        mct2.setValue(Util.getAlphaString(256));
        mde.setMsgCatalogTokens(inValue);
    }
}
