package org.eclipse.hyades.logging.events.cbe.impl.tests;

import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StartSituationImplTest.java,v 1.3 2005/02/16 22:20:40 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * JUnit test case for the <code>StartSituationImpl</code> class.
 * 
 * @author Denilson Nastacio
 * @version 1.0.1
 * @since 1.0.1 
 */
public class StartSituationImplTest extends TestCase {

    /** Target instance for the tests. */
    private StartSituation sit;

    /**
     * Constructor for StartSituationImplTest.
     * @param testcase
     */
    public StartSituationImplTest(String testcase) {
        super(testcase);
    }

    /**
     * Tests whether <code>toString</code> doesn't blow up.
     */
    public void testToString() {
        assertNotNull(sit.toString());
    }

    /**
     * Tests presence of attributes in the public interface
     */
    public void testInterface() {
        sit.getSituationQualifier();
        sit.getSuccessDisposition();
        sit.getReasoningScope();
        // setters show on other tests

        assertTrue(sit instanceof SituationType);
    }

    /**
     * Tests the serialization for <code>StartSituation</code>.
     */
    public void testSerialization() {
        Util.testEObjectSerialization(sit);
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        StartSituation sit2 = (StartSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit.setSituationQualifier("q");
        sit.setSuccessDisposition("s");
        sit.setReasoningScope("r");
        sit2 = (StartSituation) Util.clone(sit);
        assertEquals(sit, sit2);

        sit2.setSuccessDisposition("difference");
        assertFalse(sit.equals(sit2));
    }

    /**
     * Tests validation for <code>StartSituation</code>.
     */
    public void testValidationMissingAttributes() throws ValidationException {
        Util.assertMissingAttributes(sit, "all");

        sit.setSuccessDisposition("s");
        sit.setSituationQualifier("q");
        sit.setReasoningScope("r");
        sit.validate();

        sit.setSuccessDisposition(null);
        Util.assertMissingAttributes(sit, "successDisposition");

        sit.setSuccessDisposition("s");
        sit.setSituationQualifier(null);
        sit.setReasoningScope("r");
        Util.assertMissingAttributes(sit, "situationQualifier");

        sit.setSuccessDisposition("s");
        sit.setSituationQualifier("q");
        sit.setReasoningScope(null);
        Util.assertMissingAttributes(sit, "reasoningScope");
    }

    /**
     * Tests validation for <code>StartSituation</code>.
     */
    public void testValidationInvalidLengths() {
        sit.setSituationQualifier(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(64));
        try {
            sit.validate();
        }
        catch (ValidationException e) {
            fail("Strings have correct length");
        }

        sit.setSituationQualifier(Util.getPaddedString(65));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "situationQualifier");

        sit.setSituationQualifier(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(65));
        sit.setReasoningScope(Util.getPaddedString(64));
        Util.assertInvalidBoundaries(sit, "successDisposition");

        sit.setSituationQualifier(Util.getPaddedString(64));
        sit.setSuccessDisposition(Util.getPaddedString(64));
        sit.setReasoningScope(Util.getPaddedString(65));
        Util.assertInvalidBoundaries(sit, "reasoningScope");
    }

    /**
     * 
     */
    public void testAdapter() {
        Util.assertAdapterNotification(sit);
    }

    /**
     * 
     */
    public void testUnsupportedFeature() {
        Util.assertUnsupportedFeature(sit);
    }

    /*
     * Protected methods.
     */

    /* (non-Javadoc)
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
        EventFactory eFx = EventFactory.eINSTANCE;
        sit = eFx.createStartSituation();

    }

}
