package org.eclipse.hyades.logging.events.cbe.impl.tests;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Util.java,v 1.8 2005/03/09 17:19:34 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Utilities for the unit testing of the Event implementation package.
 * 
 * @author Denilson Nastacio
 * @author Scott Brown
 * @version 1.0.1
 * @since 1.0.1
 */
public class Util {

    /**
     * Returns a string of numeric characters with the size specified as parameter.
     * <P>
     * Used for testing size limits on class variables
     */
    static String getPaddedString(int size) {
        StringBuffer result = new StringBuffer(size);
        final String paddedString = "0123456789";
        final int paddedStringSize = paddedString.length();
        for (int i = 0; i < size / paddedStringSize; i++) {
            result.append(paddedString);
        }
        result.append(paddedString.substring(0, size % paddedStringSize));
        return result.toString();
    }

    /**
     * Returns a string of alphabetic characters with the size specified as parameter.
     * <P>
     * Used for testing size limits on class variables
     */
    static String getAlphaString(int size) {
        StringBuffer result = new StringBuffer(size);
        final String paddedString = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        final int paddedStringSize = paddedString.length();
        for (int i = 0; i < size / paddedStringSize; i++) {
            result.append(paddedString);
        }
        result.append(paddedString.substring(0, size % paddedStringSize));
        return result.toString();
    }

    /**
     * Tests whether the object passed as parameter can be serialized and
     * deserialized.
     * 
     * @param o              object to be tested
     * 
     * @throws IOException
     */
    static void testSerialization(Object o) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream oo = new ObjectOutputStream(bos);
        oo.writeObject(o);

        try {
            ByteArrayInputStream bis = new ByteArrayInputStream(bos.toByteArray());
            ObjectInputStream oi = new ObjectInputStream(bis);
            Object o2 = oi.readObject();

            TestCase.assertEquals("objects do not match", o, o2);
        }
        catch (ClassNotFoundException e) {
            TestCase.fail(e.getClass() + " " + e.getMessage());
        }
    }

    /**
     * Creates a valid instance of a Event event  
     *
     * @return               a valid Event instance
     */
    public static CommonBaseEvent generateEvent() {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();

        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("CE123456789012345678901234567890");

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cit1.setLocation("My Location");
        cit1.setLocationType("Unknown");
        cit1.setComponent("My Component");
        cit1.setSubComponent("My sub component");
        cit1.setComponentIdType("My Component ID Type");
        cit1.setComponentType("My Component Type");
        result.setReporterComponentId(cit1);

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setLocation("My Location 2");
        cit2.setLocationType("Unknown");
        cit2.setComponent("My Component 2");
        cit2.setSubComponent("My sub component 2");
        cit2.setComponentIdType("My Component ID Type 2");
        cit2.setComponentType("My Component Type 2");
        result.setSourceComponentId(cit2);

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        return result;
    }

    /**
     * Creates a valid instance of a Event event  
     *
     * @return               a valid Event instance
     */
    static CommonBaseEvent generateEventWithoutRequiredPrimitives() {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setLocation("My Location 2");
        cit2.setLocationType("Unknown");
        cit2.setComponent("My Component 2");
        cit2.setSubComponent("My sub component 2");
        cit2.setComponentIdType("My Component ID Type 2");
        cit2.setComponentType("My Component Type 2");
        result.setSourceComponentId(cit2);

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("NOT AVAILABLE");
        sitType.setOperationDisposition("STARTABLE");
        sitType.setProcessingDisposition("MGMTTASK_PROCESS");
        sitType.setReasoningScope("EXTERNAL");
        sit.setCategoryName("AvailableSituation");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        return result;
    }

    /**
     * Creates a valid component identification for a Event event  
     *
     * @return               the component identification
     */
    static ComponentIdentification generateComponentIdentification() {
        ComponentIdentification result = EventFactory.eINSTANCE.createComponentIdentification();

        result.setApplication("A");
        result.setComponent("A");
        result.setComponentType("A");
        result.setComponentIdType("A");
        result.setExecutionEnvironment("A");
        result.setInstanceId("A");
        result.setLocation("A");
        result.setLocationType("FQHostname");
        result.setProcessId("A");
        result.setSubComponent("A");
        result.setThreadId("A");

        return result;
    }

    /**
     * Creates a valid situation information for a Event event  
     *
     * @return               the situation information
     */
    static MsgDataElement generateMsgDataElement() {
        MsgDataElement result = EventFactory.eINSTANCE.createMsgDataElement();

        result.getMsgCatalogTokens();
        result.setMsgCatalog("A");
        result.setMsgCatalogId("A");
        result.setMsgId("A");
        result.setMsgLocale("A");

        return result;
    }

    /**
     * Tests the serialization support for SituationType or any of its 
     * inherited classes.
     *  
     * @param c              EMF class for the object
     * @return               <code>true</code> if the serialization works.
     */
    static boolean testEObjectSerialization(EObject obj) {
        String STRING_PATTERN = "abc123";

        boolean result = true;

        EClass c = obj.eClass();
        try {
            testSerialization(obj);
            List attrs = c.getEAllAttributes();
            // Test with all attributes set
            for (int i = 0; i < attrs.size(); i++) {
                EAttribute attr = (EAttribute) attrs.get(i);
                if (attr.getEAttributeType().getName().equals("EString")) {
                    if (attr.getUpperBound() > 0) {
                        obj.eSet(attr, STRING_PATTERN);
                    }
                    else {
                        List strings = new ArrayList();
                        strings.add(STRING_PATTERN);
                        obj.eSet(attr, strings);
                    }
                }
                else {
                    // This 'assertion' only support the above types for now.
                    TestCase.fail("assertion type: " + attr.eClass());
                }
            } // for attributes
            testSerialization(obj);

            // test with only one attribute set (after null pointer
            // problems)  
            for (int i = 0; i < attrs.size(); i++) {
                for (int j = 0; j < attrs.size(); j++) {
                    EAttribute attr = (EAttribute) attrs.get(i);
                    if (i == j) {
                        if (attr.getEAttributeType().getName().equals("EString")) {
                            if (attr.getUpperBound() > 0) {
                                obj.eSet(attr, STRING_PATTERN);
                            }
                            else {
                                List strings = new ArrayList();
                                strings.add(STRING_PATTERN);
                                obj.eSet(attr, strings);
                            }
                        }
                        else {
                            // This 'assertion' only support the above types for now.
                            TestCase.fail("assertion type: " + attr.eClass());
                        }
                    }
                    else {
                        obj.eUnset(attr);
                    }
                    testSerialization(obj);
                } // for j
            } // for i
        }
        catch (IOException e) {
            TestCase.fail(e.getClass() + " " + e.getMessage());
        }

        return result;
    }

    /**
     * Fails the current fixture if attributes are missing.
     * 
     * @param obj            object to be tested
     * @param attrName       name of the required attribute that is missing.
     */
    static void assertMissingAttributes(Object obj, String attrName) {
        try {
            Method validationMethod = obj.getClass().getMethod("validate", null);
            validationMethod.invoke(obj, null);
            TestCase.fail("Should not accept, this attribute is missing: [" + attrName + "]");
        }
        catch (SecurityException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalArgumentException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (NoSuchMethodException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalAccessException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (InvocationTargetException e) {
            if ((e.getTargetException() instanceof ValidationException) == false) {
                TestCase.fail(e.getLocalizedMessage());
            }
            else {
                Util.assertExceptionId("IWAT0203E", (Exception) e.getTargetException());
            }
        }
    }

    /**
     * Clones an EObject
     */
	public static EObject clone(EObject obj) {
        EObject result = null;
        if (obj != null) {
            EClass c = obj.eClass();
            result = EventFactory.eINSTANCE.create(c);

            // Handling primitive attributes             
            List attrs = c.getEAllAttributes();
            for (int i = 0; i < attrs.size(); i++) {
                EAttribute a = (EAttribute) attrs.get(i);
                Object value = obj.eGet(a);
                result.eSet(a, value);
            } // for attrs

            // Handling complex structures 
            List refs = c.getEAllContainments();
            for (int i = 0; i < refs.size(); i++) {
                EReference r = (EReference) refs.get(i);
                Object value = obj.eGet(r);
                // Handling lists
                if (value instanceof List) {
                    List listValue = (List) value;
                    List newList = new ArrayList(listValue.size());
                    for (int j = 0; j < listValue.size(); j++) {
                        Object o = listValue.get(j);
                        if (o instanceof EObject) {
                            newList.add(Util.clone((EObject) o));
                        }
                        else {
                            newList.add(o);
                        }
                    }
                    result.eSet(r, newList);
                    // Handling complex attributes
                }
                else {
                    EObject eValue = Util.clone((EObject) value);
                    result.eSet(r, eValue);
                }
            } // for references
        } // if obj not null

        return result;
    }

    /**
     * Fails the validation for the object throws an exception.
     * 
     * @param obj            object to be tested
     */
    static void assertValidBoundaries(Object obj) {
        try {
            Method validationMethod = obj.getClass().getMethod("validate", null);
            validationMethod.invoke(obj, null);
        }
        catch (SecurityException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalArgumentException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (NoSuchMethodException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalAccessException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (InvocationTargetException e) {
            if (e.getTargetException() instanceof ValidationException) {
                TestCase.fail("String lengths are within the proper size");
            }
            else {
                TestCase.fail(e.getLocalizedMessage());
            }
        }
    }

    /**
     * Fails the current fixture if attributes exceed their
     * boundaries.
     * 
     * @param obj            object to be tested
     * @param attrName       name of the attribute exceeding its boundaries.
     */
    static void assertInvalidBoundaries(Object obj, String attrName) {
        try {
            Method validationMethod = obj.getClass().getMethod("validate", null);
            validationMethod.invoke(obj, null);
            TestCase.fail("Should not accept oversized attribute: [" + attrName + "]");
        }
        catch (SecurityException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalArgumentException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (NoSuchMethodException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (IllegalAccessException e) {
            TestCase.fail(e.getLocalizedMessage());
        }
        catch (InvocationTargetException e) {
            if ((e.getTargetException() instanceof ValidationException) == false) {
                TestCase.fail(e.getLocalizedMessage());
            }
            else {

                if(((Exception) e.getTargetException()).getCause() != null){
                    
                    Util.assertExceptionId("IWAT0417E", (Exception) e.getTargetException());
                    Util.assertExceptionId("IWAT0207E", ((Exception) e.getTargetException()).getCause());
                }
                else{
                    Util.assertExceptionId("IWAT0207E", (Exception) e.getTargetException());
                }
            }
        }
    }

    /**
     * Determines whether the exception message matches the proper message id.
     * 
     * @param msgId          message identifier, such as IWAT0187E
     * @param e              exception to be verified.
     */
    static void assertExceptionId(String msgId, Throwable t) {
        String exceptionMsgId = t.getLocalizedMessage().substring(0, 10).trim();
        TestCase.assertEquals(msgId, exceptionMsgId);

        // Testing for non-substituted arguments
        TestCase.assertEquals(-1, t.getLocalizedMessage().indexOf("{"));
    }

    /**
     * Asserts whether the notification mechanism works for the designated
     * class.
     * <P>
     * Required to determine whether the model has been generated with all
     * the proper notification flags set to true.
     * </P>
     * <P>
     * The <code>TestCase</code> is failed if the notification test fails.
     * 
     * @param target         the notifies to be test.
     */
    static void assertAdapterNotification(EObject target) {
        HashMap patternMap = new HashMap();
        patternMap.put("EString", "zyxabc");
        patternMap.put("ELong", new Long(812));
        patternMap.put("EShort", new Short((short) 1));

        MockEmfAdapter adapter = new MockEmfAdapter(target);
        target.eAdapters().add(adapter);
        EClass c = target.eClass();
        List l = c.getEAllAttributes();
        for (int i = 0; i < l.size(); i++) {
            EAttribute attr = (EAttribute) l.get(i);
            String attrTypeName = attr.getEAttributeType().getName();
            Object pattern = patternMap.get(attrTypeName);
            if (pattern != null) {
                if (attr.getUpperBound() > 0) {
                    target.eSet(attr, pattern);
                    TestCase.assertEquals(pattern, adapter.getLastNotification().getNewValue());
                }
                else {
                    List strings = new ArrayList();
                    strings.add(pattern);
                    target.eSet(attr, strings);
                    TestCase.assertEquals(pattern, adapter.getLastNotification().getNewValue());
                }
            }
            else {
                // This 'assertion' only support the above types for now.
                TestCase.fail("assertion type: " + attr.getEAttributeType().getName());
            }
        } // for all attributes
    }

    /**
     * Determines wheter the <code>eGet</code>, <code>eSet</code>,
     * <code>eIsSet</code>, and <code>eUnset</code> methods actually throw
     * exceptions in case of unknown features.
     *  
     * @param target         instance to be tested
     */
    static void assertUnsupportedFeature(EObject target) {
        // We have to pick a feature that does not exist on the target
        EventPackage ePck = EventPackage.eINSTANCE;
        EStructuralFeature f = ePck.getAssociatedEvent_ResolvedEvents();
        if (target.eClass().equals(ePck.getAssociatedEvent())) {
            f = ePck.getCommonBaseEvent_ElapsedTime();
        }
        try {
            target.eGet(f);
            TestCase.fail("Non-existent feature for eGet");
        }
        catch (IllegalArgumentException e) {
        }
        try {
            target.eSet(f, null);
            TestCase.fail("Non-existent feature for eSet");
        }
        catch (IllegalArgumentException e) {
        }
        try {
            target.eIsSet(f);
            TestCase.fail("Non-existent feature for eIsSet");
        }
        catch (IllegalArgumentException e) {
        }
        try {
            target.eUnset(f);
            TestCase.fail("Non-existent feature for eUnset");
        }
        catch (IllegalArgumentException e) {
        }
    }

}
