package org.eclipse.hyades.logging.events.cbe.util;

import java.awt.EventQueue;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.SimpleTimeZone;
import java.util.StringTokenizer;
import java.util.TimeZone;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.logging.core.LoggingCorePlugin;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.TemplateContentHandler;
import org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventHelpers.java,v 1.18 2005/04/19 14:55:43 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Miscellaneous routines to support functions such as validation, serialization
 * and comparison.
 * 
 * @author Denilson Nastacio
 * @author Jason Cornpropst
 * @author Scott Brown
 * @author Paul E. Slauenwhite
 * @author Adriana Viman
 * 
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventHelpers {

    /**
     * Static flag for quickly determining if a JRE 1.4.x and above
     * run-time environment.
     * <p>
     * By default, a JRE 1.4.x and above run-time environment is assumed.
     */
    private static boolean isJava14xRunTime = true;

    private static ResourceBundle resourceBundle = null;

    //NOTE: Set the locale of the date format to English since the 
    //XSD:dateTime format is used which is locale agnostic.
    private static SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss",Locale.ENGLISH);

    static {
        
        Calendar calendar = new GregorianCalendar(new SimpleTimeZone(0, "UTC"));
        calendar.setLenient(false);
        
        simpleDateFormat.setLenient(false);
        simpleDateFormat.setCalendar(calendar);
    }

    private static final String DEFAULT_FORMATTED_DATE_FORMAT = "MMMM d, yyyy h:mm:ss.SSS a z";
    
    /** 
     * CBE list elements where order is important during comparison. 
     */
    private static final List CBE_ORDERED_LISTS = Arrays.asList(new Object[] { EventPackage.eINSTANCE.getExtendedDataElement_Values().getName(), EventPackage.eINSTANCE.getMsgDataElement_MsgCatalogTokens().getName()});

    /**
     * CBE element (e.g. ExtendedDataElement#hexValue where the text case is not important during comparison.
     */
    private static final String CBE_CASE_INSENSITIVE_PROPERTY = EventPackage.eINSTANCE.getExtendedDataElement_HexValue().getName();

    private static final String RESOURCE_BUNDLE_NAME = "org.eclipse.hyades.logging.core.properties.plugin";

    private final static String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * Current thread lock for synchronization.
     * <p>
     * 
     * @deprecated As of 3.1, the event object is used to synchronize its operations.
     */
    public static final Object LOCK = new Object();

    /**
     * Generic serialization routine for EMF object.
     * 
     * @param o
     *            EMF object to be serialized
     * @param in
     *            Java IO input stream contained the serialized strem
     * 
     * @throws ClassNotFoundException
     *             if the class of an object inside the stream cannot be found
     *             in the classpath
     * @throws IOException
     *             if the input stream cannot be read for whatever reason
     */
    public static void serializableRead(EObject o, java.io.ObjectInputStream in) throws ClassNotFoundException, IOException {

        List features = o.eClass().getEAllStructuralFeatures();;
        
        for (int i = 0; i < features.size(); i++) {
          
            EStructuralFeature feature = (EStructuralFeature) features.get(i);
            Object value = in.readObject();
            o.eSet(feature, value);
        } 
    }

    /**
     * Generic serialization routine for EMF object.
     * 
     * @param o
     *            EMF object to be serialized
     * @param out
     *            Java IO stream to where the serialized object will be written.
     * 
     * @throws IOException
     *             if the object cannot be written to the output stream for
     *             whatever reason.
     */
    public static void serializableWrite(EObject o, java.io.ObjectOutputStream out) throws IOException {

        List features = o.eClass().getEAllStructuralFeatures();

        for (int i = 0; i < features.size(); i++) {

            Object sourceValue = o.eGet(((EStructuralFeature) (features.get(i))));
            
            //IIOP does not process ELists:
            if (sourceValue instanceof EList) {
          
                List newList = new ArrayList((List) sourceValue);
                out.writeObject(newList);
            } 
            else {
                out.writeObject(sourceValue);
            }
        } 
    }

    /**
     * Deep-comparison for two EMF objects.
     * <P>
     * If the the two EMF objects are of the same type, it traverses their
     * features recursively until a difference is found.
     * </P>
     * <P>
     * When comparing lists, this method ignores order.
     * </P>
     * <P>
     * This is a helper for implementing the <code>equals</code> method on EMF
     * objects, where <code>obj1</code> is the object where
     * <code>equals</code> was called and <code>obj2</code> is the parameter
     * for that call.
     * <P>
     * Therefore <code>obj1</code> can never be <code>null</code> and when
     * <code>obj2</code> is <code>null</code>, the result is
     * <code>false</code>.
     * </P>
     * 
     * @param obj1
     *            first element in the comparison
     * @param obj2
     *            second element in the comparison
     * 
     * @return <code>true</code> if the two objects are identical instancesor
     *         <code>false</code> otherwise.
     */
    public static boolean compareEObject(Object obj1, Object obj2) {

        // Not throwing IllegalArgumentException when obj1 is null because the
        // precondition is that this object should not be null/
        if (obj2 == null) { return false; }

        // At this point we can ensure that both pointers are not null.
        // Are they both EObjects?
        if (((obj1 instanceof EObject) == false) || ((obj2 instanceof EObject) == false)) { return false; }
        if (obj1 == obj2) { return true; }

        if (obj1.getClass().equals(obj2.getClass()) == false) { 
            return false; 
        }
        
        EObject source = (EObject) obj1;
        EObject target = (EObject) obj2;
        boolean same = true;
        List features = source.eClass().getEAllStructuralFeatures();
        
        for (int i = 0; i < features.size(); i++) {
        
            EStructuralFeature feature = (EStructuralFeature) features.get(i);

            Object sourceValue = source.eGet(feature);
            Object targetValue = target.eGet(feature);
            boolean sourceIsSet = source.eIsSet(feature);
            boolean targetIsSet = target.eIsSet(feature);

            // Are the source and target value both null or both not null
            if (((sourceValue != null) && (targetValue == null)) || ((sourceValue == null) && (targetValue != null)) || (sourceIsSet != targetIsSet)) {
                same = false;
                break;
            }

            // At this point, either target and source are null (equal)
            // or not null, in which case equality must be checked.
            if (sourceValue != null) {
                if (sourceValue instanceof List) {
                    if (CBE_ORDERED_LISTS.contains(feature.getName())) {
                        same = EventHelpers.compareEList((List) sourceValue, (List) targetValue, true);
                        // consider order
                    } else {
                        same = EventHelpers.compareEList((List) sourceValue, (List) targetValue, false);
                        // consider order
                    }
                } else {
                    if (feature.getName().equals(CBE_CASE_INSENSITIVE_PROPERTY)) {
                        same = ((String) sourceValue).equalsIgnoreCase((String) targetValue);
                    } else {
                        same = sourceValue.equals(targetValue);
                    }
                }
                if (same == false) {
                    break;
                }
            } // if source value not null
        } // for features

        return same;
    }

    /**
     * Converts the creationTime from a string in XML Schema dateTime format
     * (CCYY-MM-DDThh:mm:ss) to a long representing the milliseconds since
     * 1970-01-01T00:00:00
     * <p>
     * 
     * @param xsdDateTimeString
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @return The date in milliseconds since 1970-01-01T00:00:00Z. If the input
     *         cannot be converted or the date is older than 1970 then zero will
     *         be returned.
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid.
     * @see org.eclipse.hyades.logging.events.cbe.CommonBaseEvent#getCreationTimeAsLong()
     */
    public static long dateToLong(String xsdDateTimeString) {
        String dateString = null;
        String subSecondsString = null;
        String utcOffsetString = null;
        char utcOffsetDirection = '\0';
        Date date = null;
        long milliseconds = 0;
        int utcOffsetStringIndex = -1;
        int subSecondsStringIndex;

        //NOTE:  19 is the length of "yyyy-MM-ddTHH:mm:ss".
        if (xsdDateTimeString.length() < 19) { throw new IllegalArgumentException(xsdDateTimeString + " is not a valid xsd:dateTime"); }

        // Build a dateString that does not include any offset (i.e. anything
        // following Z, +, or -).
        // Set up utcOffsetDirection and utcOffsetString for later use
        //NOTE:  19 is the length of "yyyy-MM-ddTHH:mm:ss".
        if (xsdDateTimeString.indexOf('Z') >= 19) {
            utcOffsetStringIndex = xsdDateTimeString.indexOf('Z');

            // adjust dateTime to remove timezone information
            dateString = xsdDateTimeString.substring(0, utcOffsetStringIndex);

            utcOffsetDirection = 'Z';
            
        } 
        //NOTE:  19 is the length of "yyyy-MM-ddTHH:mm:ss".
        else if (xsdDateTimeString.indexOf('+') >= 19) {
            utcOffsetStringIndex = xsdDateTimeString.indexOf('+');

            // adjust dateTime to remove timezone information
            dateString = xsdDateTimeString.substring(0, utcOffsetStringIndex);

            // get the offset string
            utcOffsetDirection = xsdDateTimeString.charAt(utcOffsetStringIndex);
            utcOffsetString = xsdDateTimeString.substring(utcOffsetStringIndex + 1);
        } 
        //NOTE:  19 is the length of "yyyy-MM-ddTHH:mm:ss".
        else if (xsdDateTimeString.indexOf('-', 19) >= 19) {
            utcOffsetStringIndex = xsdDateTimeString.indexOf('-', 19);

            // adjust dateTime to remove timezone information
            dateString = xsdDateTimeString.substring(0, utcOffsetStringIndex);

            // get the offset string
            utcOffsetDirection = xsdDateTimeString.charAt(utcOffsetStringIndex);
            utcOffsetString = xsdDateTimeString.substring(utcOffsetStringIndex + 1);
        } else {
            dateString = xsdDateTimeString;
        }

        // Adjust the dateString to not include any subseconds and set the
        // subSecondsString
        subSecondsStringIndex = dateString.indexOf('.');

        if (subSecondsStringIndex > 0) {
            // adjust dateTime to remove timezone information
            if (utcOffsetStringIndex != -1) {
                dateString = xsdDateTimeString.substring(0, utcOffsetStringIndex);
            }

            // get the subSeconds string
            // make sure to add 1 to the offset so we can skip the .
            subSecondsString = dateString.substring(subSecondsStringIndex + 1);
        }

        // Make sure dateString is not null (it could be if, for example, the
        // input started with Z)
        // and then use the Java Calendar class to convert the input dateString
        if (dateString != null) {
            int myDash = xsdDateTimeString.indexOf('-');

            if (myDash == -1) {
                throw new IllegalArgumentException(xsdDateTimeString + " is not a valid xsd:dateTime. Cannot find '-'.");
            } else {
                String year = dateString.substring(0, myDash);

                if ((year.length() == 4) && (Integer.parseInt(year) > 1969)) {

                    try {
                        synchronized (simpleDateFormat) {
                            date = simpleDateFormat.parse(dateString);
                        }
                    } catch (ParseException e) {
                        throw new IllegalArgumentException(xsdDateTimeString + " is not a valid xsd:dateTime. Parse error is: " + e.getMessage());
                    }
                }
            }
        }

        // If the parse was successful (i.e. date is not null) then convert the
        // date to a long
        // and add in the subseconds. Note, in Java .5, .50, and .500 are not
        // equal but
        // in the xsd date/time they are equal.
        if (date != null) {
            // now get the milliseconds from the date
            milliseconds = date.getTime();

            // now convert microseconds string into a long and add it to
            // the milliseconds in the date
            if ((subSecondsString != null) && (subSecondsString.length() > 0)) {
                long subSeconds = Long.parseLong(subSecondsString);
                int subSecondsLength = subSecondsString.length();

                if (subSecondsLength == 1) {
                    subSeconds = subSeconds * 100;
                } else if (subSecondsLength == 2) {
                    subSeconds = subSeconds * 10;
                } else if (subSecondsLength == 3) {
                } else {
                    subSeconds = Long.parseLong(subSecondsString.substring(0, 3));
                }

                milliseconds = milliseconds + subSeconds;
            }

            // now depending on the offset adjust the time to UTC
            if ((utcOffsetDirection != '\0') && (utcOffsetString != null) && (utcOffsetString.length() == 5)) {
                long offsetHours = Long.parseLong(utcOffsetString.substring(0, 2));
                long offsetMinutes = Long.parseLong(utcOffsetString.substring(3));

                if (utcOffsetDirection == '+') {
                    milliseconds = milliseconds - (offsetHours * 60 * 60 * 1000) - (offsetMinutes * 60 * 1000);
                } else if (utcOffsetDirection == '-') {
                    milliseconds = milliseconds + (offsetHours * 60 * 60 * 1000) + (offsetMinutes * 60 * 1000);
                }
            } else if (utcOffsetString != null) {
                if (!utcOffsetString.equals("Z")) {
                    milliseconds = 0;
                }
            }
        }

        return milliseconds;
    }
    

    /**
     * Converts a long representing UTC in milliseconds to the XML Schema
     * datetime format (CCYY-MM-DDThh:mm:ssZ)
     * <p>
     * 
     * @param dateInMillis
     *            A long representing a coordinated universal time (UTC) time stamp in milliseconds.
     * @return The date in string format (CCYY-MM-DDThh:mm:ssZ)
     * @throws IllegalArgumentException
     *             If the long representing a coordinated universal time (UTC) time stamp in milliseconds is negative. 
     * @see org.eclipse.hyades.logging.events.cbe.CommonBaseEvent#setCreationTimeAsLong(long)
     */
    public static String longToDate(long dateInMillis) {

        if (dateInMillis < 0){
        	throw new IllegalArgumentException(dateInMillis + " cannot be negative.");
        }

        String dateTime = null;

        // create a the GMT timezone
        SimpleTimeZone gmt = new SimpleTimeZone(0, "UTC");

        // create a GregorianCalendar with the GMT time zone
        Calendar calendar = new GregorianCalendar(gmt);

        // set the date in the calendar
        calendar.setTime(new Date(dateInMillis));

        // get the interger representation for all the fields
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);

        // java has January as month 0 so need add 1 to the month
        month++;

        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int minute = calendar.get(Calendar.MINUTE);
        int second = calendar.get(Calendar.SECOND);
        int milliseconds = calendar.get(Calendar.MILLISECOND);

        // now create a string buffer to build the string
        // if the fields are not the correct size they must be padded with 0's
        StringBuffer stringBuffer = new StringBuffer(35);
        stringBuffer.append(year);
        stringBuffer.append('-');

        if (month < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(month);
        stringBuffer.append('-');

        if (day < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(day);
        stringBuffer.append('T');

        if (hour < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(hour);
        stringBuffer.append(':');

        if (minute < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(minute);
        stringBuffer.append(':');

        if (second < 10) {
            stringBuffer.append("0");
        }

        stringBuffer.append(second);

        stringBuffer.append(".");

        if (milliseconds < 10) {
            stringBuffer.append("00");
        } else if (milliseconds < 100) {
            stringBuffer.append("0");
        }

        stringBuffer.append(milliseconds);

        // are times are always UTC so append a Z
        stringBuffer.append("Z");
        dateTime = stringBuffer.toString();

        return dateTime;
    }

    /*
     * Private methods
     */

    /**
     * Compare two <code>EList</code> objects ignoring order
     * 
     * @param obj1           first list in the comparison
     * @param obj2           second list in the comparison
     * @param considerOrder  indicates whether the order of the elements is
     *                       important in the comparison.
     * 
     * @pre obj1!=null  
     * @pre obj2!=null  
     */
    private static boolean compareEList(List obj1, List obj2, boolean considerOrder) {

        // Since the lists in the CBE spec don't have a notion or order
        // or unicity, we have to make sure that not only an element in list1
        // is available in list2, but also that *n* identical elements in list1
        // have *n* corresponding elements in list2.
        // The algorithm walks list1 and nullifies any position containing
        // an corresponding element on list2, this avoids that the same element
        // in
        // list2 account for more than one element in list1. Of course we cannot
        // nullify elements in the source obj2, so we need a local copy of it.

        // Do the lists at least have the same length.
        if (obj1.size() != obj2.size()) { return false; }

        // indicates whether the traversal is going well
        boolean same = true;
        Object[] list2 = obj2.toArray(new Object[obj2.size()]);

        // We could have used list1.size() in the loop below, but that is a
        // method call and list2.length has the same value inside it
        for (int i = 0; i < list2.length; i++) {
            boolean found = false;
            Object o1 = obj1.get(i);
            if (considerOrder) {
                Object o2 = list2[i];
                found = o1.equals(o2);
            } else {
                for (int j = 0; j < list2.length; j++) {
                    Object o2 = list2[j];
                    if ((o2 != null) && (o1.equals(o2))) {
                        found = true;
                        list2[j] = null;
                        break;
                    }
                } // for list2
            }
            if (found == false) {
                same = false;
                break;
            }
        } // for list1

        return same;
    }

    public static String getString(String key) {

        //Load the resource bundle:
        if (resourceBundle == null) {

            //Attempt to load the Eclipse resource bundle:
            try {
                resourceBundle = LoggingCorePlugin.getDefault().getResourceBundle();
            }
            catch (Throwable t) {
                //Ignore since not an Eclipse run-time environment.
            }

            //Attempt to load the 'org.eclipse.hyades.logging.core.properties.plugin.properties' resource bundle:
            if (resourceBundle == null) {
                
                try {
                    resourceBundle = ResourceBundle.getBundle(RESOURCE_BUNDLE_NAME);
                }
                catch (MissingResourceException m) {
                    return key;
                }
            }
        }

        try {
            return (resourceBundle.getString(key.trim()).trim());
        }
        catch (Exception e) {
        }

        return key;
    }

    public static String getString(String key, Object argument) {
        return (getString(key, new Object[] { argument}));
    }

    public static String getString(String key, Object argumentA, Object argumentB) {
        return (getString(key, new Object[] { argumentA, argumentB}));
    }

    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC) {
        return (getString(key, new Object[] { argumentA, argumentB, argumentC}));
    }

    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC, Object argumentD) {
        return (getString(key, new Object[] { argumentA, argumentB, argumentC, argumentD}));
    }

    public static String getString(String key, Object[] arguments) {

        try {
            return (MessageFormat.format(getString(key), arguments));
        }
        catch (Exception e) {
            return key;
        }
    }

    /**
     * Maps data of the parameter <code>java.lang.Throwable</code> object to an
     * Extended Data Element.
     * <p>
     * This API provides a structured representation of the parameter <code>java.lang.Throwable</code>
     * object as an Extended Data Element with the following advantages:
     * <p>
     * <ul>
     * <li>Efficient specialized parsibility and de-/serialization.</li>
     * <li>Identifiable Extended Data Element <code>name</code>s (e.g. "Throwable" and "Cause")</li>
     * <li>Captures all nested causal <code>java.lang.Throwable</code>s.</li>
     * <li>Adheres to the 1024-character restriction imposed on the Extended Data Element <code>values</code>.</li>
     * <li></li>
     * </ul>
     * <p>
     * NOTE: This API invoked in JRE 1.3.x and below run-time environments parses the 
     * stack trace produced by the <code>java.lang.Throwable</code>'s <code>printStackTrace()</code>
     * API (e.g. no causal <code>java.lang.Throwable</code>(s)).  Alternatively, this API invoked in JRE 1.4.x 
     * and above run-time environments utilizes the <code>java.lang.Throwable</code>'s causal 
     * <code>java.lang.Throwable</code> and <code>StackTraceElement</code>(s) properties.  
	 * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element:
     * <p>
     * <ul>
     * <li>name = "Throwable"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
     * <code>java.lang.Throwable</code>'s localized message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt; <code>java.lang.Throwable</code>'s
     * stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>children[0] = &lt; <code>java.lang.Throwable</code>'s cause&gt;</li>
     * </ul>
     * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element if the 
     * parameter <code>java.lang.Throwable</code> object is <code>null</code>:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0] = <code>null</code></li>
     * </ul>
     * <p>
     * Causal <code>java.lang.Throwable</code>s are recursively converted to Extended
     * Data Elements using the following mapping:
     * <p>
     * <ul>
     * <li>name = "Cause"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt;cause's class name&gt;[: &lt;cause's localized
     * message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>[children[0] = &lt;cause's cause&gt;]</li>
     * </ul>
     * <p>
     * ...
     * <p>
     * NOTE: When the <code>java.lang.Throwable</code>'s class name and localized message
     * are greater than 1024 characters, the resultant class name and localized message string 
     * is segmented into a the first 1024-character elements of the <code>values</code> property.
     * As such, the <code>java.lang.Throwable</code>'s stack trace elements are transposed by the
     * number of 1024-character elements in the <code>values</code> property.
     * <p>
     * 
     * @param throwable
     *            The <code>java.lang.Throwable</code> object to be converted to an
     *            Extended Data Element.
     * @return The Extended Data Element representation of the parameter
     *         <code>java.lang.Throwable</code> object.
     */
    public static ExtendedDataElement convertToExtendedDataElement(Throwable throwable) {
        return (convertToExtendedDataElement(throwable,"Throwable"));
    }
    
    /**
     * Maps data of the parameter <code>java.lang.Throwable</code> object to an
     * Extended Data Element with the parameter <code>name</code>.
     * <p>
     * This API provides a structured representation of the parameter <code>java.lang.Throwable</code>
     * object as an Extended Data Element with the following advantages:
     * <p>
     * <ul>
     * <li>Efficient specialized parsibility and de-/serialization.</li>
     * <li>Identifiable Extended Data Element <code>name</code>s (e.g. &lt;name&gt; and "Cause")</li>
     * <li>Captures all nested causal <code>java.lang.Throwable</code>s.</li>
     * <li>Adheres to the 1024-character restriction imposed on the Extended Data Element <code>values</code>.</li>
     * <li></li>
     * </ul>
     * <p>
     * NOTE: This API invoked in JRE 1.3.x and below run-time environments parses the 
     * stack trace produced by the <code>java.lang.Throwable</code>'s <code>printStackTrace()</code>
     * API (e.g. no causal <code>java.lang.Throwable</code>(s)).  Alternatively, this API invoked in 
     * JRE 1.4.x and above run-time environments utilizes the <code>java.lang.Throwable</code>'s causal 
     * <code>java.lang.Throwable</code> and <code>StackTraceElement</code>(s) properties.  
	 * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
     * <code>java.lang.Throwable</code>'s localized message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt; <code>java.lang.Throwable</code>'s
     * stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>children[0] = &lt; <code>java.lang.Throwable</code>'s cause&gt;</li>
     * </ul>
     * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element if the 
     * parameter <code>java.lang.Throwable</code> object is <code>null</code>:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0] = <code>null</code></li>
     * </ul>
     * <p>
     * Causal <code>java.lang.Throwable</code>s are recursively converted to Extended
     * Data Elements using the following mapping:
     * <p>
     * <ul>
     * <li>name = "Cause"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt;cause's class name&gt;[: &lt;cause's localized
     * message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>[children[0] = &lt;cause's cause&gt;]</li>
     * </ul>
     * <p>
     * ...
     * <p>
     * NOTE: When the <code>java.lang.Throwable</code>'s class name and localized message
     * are greater than 1024 characters, the resultant class name and localized message string 
     * is segmented into a the first 1024-character elements of the <code>values</code> property.
     * As such, the <code>java.lang.Throwable</code>'s stack trace elements are transposed by the
     * number of 1024-character elements in the <code>values</code> property.
     * <p>
     * 
     * @param throwable
     *            The <code>java.lang.Throwable</code> object to be converted to an
     *            Extended Data Element.
     * @param name
     *            The <code>name</code> property of the Extended Data Element.
     * @return The Extended Data Element representation of the parameter
     *         <code>java.lang.Throwable</code> object.
     */
    public static ExtendedDataElement convertToExtendedDataElement(Throwable throwable, String name) {

        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        extendedDataElement.setName(name);
        extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

        if (throwable == null) {
            extendedDataElement.setValues(new String[] { null});
        }
        else{

            //Attempt to use the <code>java.lang.Throwable</code> APIs available in a JRE 1.4.x and above run-time environment:
            if (isJava14xRunTime) {

                try {

                    Object[] stackTraceElements = ((Object[])(throwable.getClass().getMethod("getStackTrace", null).invoke(throwable, null)));
                    
                    if (stackTraceElements.length > 0) {

                        //Ensure that the <code>java.lang.Throwable</code>'s class name and localized message
                        //are segmented if greater than 1024 characters:
                        String[] throwableStringArray = getExtendedDataElementValuesArray(throwable.toString());
                        
                        String[] values = new String[throwableStringArray.length + stackTraceElements.length];

                        System.arraycopy(throwableStringArray,0,values,0,throwableStringArray.length);
                        
                        //ASSUMPTION: Stack trace elements are typically less than 1024 characters.
                        //For performance reasons, do not segment stack trace elements.
                        for (int counter = 0; counter < stackTraceElements.length; counter++) {
                            values[counter + throwableStringArray.length] = stackTraceElements[counter].toString();
                        }

                        extendedDataElement.setValues(values);
                    } 
                    else {
                        extendedDataElement.setValues(getExtendedDataElementValuesArray(throwable.toString()));
                    }

                    Throwable cause  = ((Throwable)(throwable.getClass().getMethod("getCause", null).invoke(throwable, null)));

                    if (cause != null) {
                        extendedDataElement.addChild(convertToExtendedDataElement(cause, "Cause"));
                    }
                    
                    return extendedDataElement;
                } 
                catch (Throwable t) {
                    isJava14xRunTime = false;
                }
            }
            
            //Default to standard <code>java.lang.Throwable</code> APIs available in a JRE 1.3.x and below run-time environment:
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();                
            PrintWriter writer = new PrintWriter(outputStream, true);

            throwable.printStackTrace(writer);
                        
            if(!writer.checkError()){
                
                //NOTE:  The <code>java.lang.Throwable</code>'s stack trace is <i>guaranteed</i> to not be <code>null</code> or empty (e.g. class name) if successfully serialized.
                StringTokenizer stackTraceElements = new StringTokenizer(outputStream.toString().trim(),LINE_SEPARATOR);
                int tokensCount = stackTraceElements.countTokens();
                
                if(tokensCount > 1){

                    //Ensure that the <code>java.lang.Throwable</code>'s class name and localized message
                    //are segmented if greater than 1024 characters:
                    String[] throwableStringArray = getExtendedDataElementValuesArray(stackTraceElements.nextToken());
                    
                    tokensCount--;
                    
                    String[] values = new String[throwableStringArray.length + tokensCount];

                    System.arraycopy(throwableStringArray,0,values,0,throwableStringArray.length);

                    String stackTraceElement = null;                    
                    
                    //ASSUMPTION: Stack trace elements are typically less than 1024 characters.
                    //For performance reasons, do not segment stack trace elements.
                    for (int counter = 0; counter < tokensCount; counter++) {
                    
                        stackTraceElement = stackTraceElements.nextToken().trim();
                        
                        if(stackTraceElement.startsWith("at")){
                            values[counter + throwableStringArray.length] = stackTraceElement.substring(2).trim(); //2 is the length of "at".
                        }
						else{
						    values[counter + throwableStringArray.length] = stackTraceElement;
						}
                    }

                    extendedDataElement.setValues(values);
                }
                else{
                    extendedDataElement.setValues(getExtendedDataElementValuesArray(stackTraceElements.nextToken()));
                }
			}
            else{
                extendedDataElement.setValues(getExtendedDataElementValuesArray(throwable.toString()));
            }
            
            writer.close();            
        }

        return extendedDataElement;
    }

    /**
     * Convenience API to transpose a <code>values</code> string to a
     * <code>values</code> array, thereby ensuring the parameter
     * <code>values</code> string does not exceed the 1024 character limit, as
     * stated in the Common Base Event v1.0.1 specification.
     * <p>
     * If the parameter <code>values</code> property is larger than 1024
     * characters, the string is segmented into a String array of 1024-character
     * elements. However, if the parameter <code>values</code> property is
     * 1024 or less characters or <code>null</code>, the string is set
     * directly on the first element a String array.
     * <p>
     * 
     * @param values
     *            The values string to be transposed to a values array.
     */
    public static String[] getExtendedDataElementValuesArray(String values) {

        if (values == null) {
            return (new String[] { null});
        } else {

            int valuesLength = values.length();

            if (valuesLength > 1024) {

                String[] valuesArray = new String[((int) (Math.ceil(valuesLength / 1024.0)))];

                for (int counter = 0; counter < valuesArray.length; counter++) {
                    valuesArray[counter] = values.substring((counter * 1024), Math.min(((counter + 1) * 1024), valuesLength));
                }

                return valuesArray;
            } else {
                return (new String[] { values});
            }
        }
    }

    /**
     * Converts a long representing a coordinated universal time (UTC) date in 
     * milliseconds to a formatted string using the default date format pattern.  
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleTimeZone</code>.
     * <p>
     * The time zone of the returned string is coordinated universal time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param dateInMillis
     *            A long representing a coordinated universal time (UTC) time stamp in milliseconds.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the long representing a coordinated universal time (UTC) time stamp in milliseconds is negative.
     * @see java.util.SimpleTimeZone
     */
    public static String getFormattedDateString(long dateInMillis) throws IllegalArgumentException{
        return (getFormattedDateString(dateInMillis,DEFAULT_FORMATTED_DATE_FORMAT));   
    }
    
    /**
     * Converts a long representing a coordinated universal time (UTC) date in 
     * milliseconds to a formatted string using the parameter date format pattern.  
     * <p>
     * For more information on the syntax and meaning of the individual 
     * symbols in the parameter date format pattern, see the class comment header for 
     * <code>java.util.SimpleTimeZone</code>.
     * <p>
     * The time zone of the returned string is coordinated universal time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param dateInMillis
     *            A long representing a coordinated universal time (UTC) time stamp in milliseconds.
     * @param dateFormatPattern
     * 			  The date format pattern (see <code>java.util.SimpleTimeZone</code>).
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException
     *             If the long representing a coordinated universal time (UTC) time stamp in milliseconds is negative or 
     * 			   the date format pattern (see <code>java.util.SimpleTimeZone</code>) is <code>null</code> or invalid.
     * @see java.util.SimpleTimeZone
     */
    public static String getFormattedDateString(long dateInMillis, String dateFormatPattern) throws IllegalArgumentException{
    	        
        if (dateInMillis < 0){
        	throw new IllegalArgumentException(dateInMillis + " cannot be negative.");
        }

        //NOTE: Set the locale of the date format to English since the 
        //Gregorian calendar format is used which is locale agnostic.
        SimpleDateFormat dateFormat = new SimpleDateFormat(dateFormatPattern,Locale.ENGLISH);
        dateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "GMT")));
        
        return (dateFormat.format(new Date(dateInMillis)));
    }
    
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the default date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleTimeZone</code>.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is coordinated universal time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTimeString
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid.
     * @see java.util.SimpleTimeZone
     */
    public static String getFormattedDateString(String xsdDateTimeString) throws IllegalArgumentException{
        return (getFormattedDateString(xsdDateTimeString,DEFAULT_FORMATTED_DATE_FORMAT));
    }
        
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the parameter date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
     * For more information on the syntax and meaning of the individual 
     * symbols in the parameter date format pattern, see the class comment header for 
     * <code>java.util.SimpleTimeZone</code>.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is coordinated universal time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTimeString
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @param dateFormatPattern
     * 			  The date format pattern (see <code>java.util.SimpleTimeZone</code>).
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid or 
     * 			   the date format pattern (see <code>java.util.SimpleTimeZone</code>) is <code>null</code> or invalid.
     * @see java.util.SimpleTimeZone
     */
    public static String getFormattedDateString(String xsdDateTimeString, String dateFormatPattern) throws IllegalArgumentException{
   
        //Step 1: Convert to a long representing a UTC date in milliseconds:
        long utcDateInMillis = dateToLong(xsdDateTimeString);
                
        //Step 2: Resolve the time zone in the parameter string representing an XML Schema dateTime:
        //Assumption:  Since an IllegalArgumentException was not thrown from the call to dateToLong(xsdDateTimeString), the xsdDateTimeString is assumed to be 19 or more characters in length.
        TimeZone timeZone = null;
        char timeZoneSign = xsdDateTimeString.trim().charAt(xsdDateTimeString.trim().length() - 6);

        if((timeZoneSign == '+') || (timeZoneSign == '-')){
            timeZone = TimeZone.getTimeZone("GMT" + xsdDateTimeString.substring(xsdDateTimeString.lastIndexOf(timeZoneSign)).trim());
        }
        else{
            timeZone = new SimpleTimeZone(0, "GMT");
        }

        //Step 3: Create the formatted string:
        //NOTE: Set the locale of the date format to English since the 
        //XSD:dateTime format is used which is locale agnostic.
        SimpleDateFormat dateFormat = new SimpleDateFormat(dateFormatPattern,Locale.ENGLISH);
        dateFormat.setCalendar(new GregorianCalendar(timeZone));
        
        return (dateFormat.format(new Date(utcDateInMillis)));        
    }
    
    /**
     * Converts the parameter object to a <code>CommonBaseEvent</code> 
     * based on the Common Base Event v1.0.1 schema.
     * <p>
     * This method uses the following rules:
     * <p>
     * <ol>
     * <li>The <code>creationTime</code> Common Base Event property is set to the current time.</li>
     * <li>The <code>globalInstanceId</code> Common Base Event property is set to a unique GUID.</li>
     * <li>The <code>situation</code> Common Base Event property is set to an internal, log and report situation.</li>
     * <li>The <code>sourceComponentId</code> Common Base Event property is set to a default component ID for the calling logging facility.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * If the parameter object is an instance of a <code>CommonBaseEvent</code>, it is cast to a 
     * <code>CommonBaseEvent</code> and returned.
     * <p>
     * 
     * @param object
     *            The object to be converted to a Common Base Event.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     * @return The generated Common Base Event from the parameter object.
     */
    public static CommonBaseEvent convertObjectToCommonBaseEvent(Object object, int maxReferences) {
       
        //Determine if the complex object is a Common Base Event:
        if (object instanceof CommonBaseEvent) {    
            return ((CommonBaseEvent)(object));
        }
        
        CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();
        convertObjectToCommonBaseEvent(commonBaseEvent,object,maxReferences);
        
        return commonBaseEvent;
    }
    
    /**
     * Converts the parameter object to a <code>CommonBaseEvent</code> 
     * based on the Common Base Event v1.0.1 schema and sets the Common
     * Base Event properties on the parameter Common Base Event.
     * <p>
     * This method uses the following rules:
     * <p>
     * <ol>
     * <li>The <code>creationTime</code> Common Base Event property is set to the current time, if not previously set.</li>
     * <li>The <code>globalInstanceId</code> Common Base Event property is set to a unique GUID, if not previously set.</li>
     * <li>The <code>situation</code> Common Base Event property is set to an internal, log and report situation, if not previously set.</li>
     * <li>The <code>sourceComponentId</code> Common Base Event property is set to a default component ID for the calling logging facility, if not previously set.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * If the parameter object is an instance of a <code>CommonBaseEvent</code>, its properties are
     * set on the parameter <code>CommonBaseEvent</code> using the precedence rules defined 
     * <code>org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl</code>.
     * <p>
     * NOTE:  The parameter <code>CommonBaseEvent</code>'s existing <code>ContentHandler</code>,
     * if any, is preserved for future use.
     * <p>
     * 
     * @param commonBaseEvent
     *            The Common Base Event generated from converting the parameter object.
     * @param object
     *            The object to be converted to a Common Base Event.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     */
    public static void convertObjectToCommonBaseEvent(CommonBaseEvent commonBaseEvent, Object object, int maxReferences) {
       
        //Determine if the complex object is a Common Base Event:
        if (object instanceof CommonBaseEvent) {            
            
            TemplateContentHandler templateContentHandler = new TemplateContentHandlerImpl();
            templateContentHandler.setTemplateEvent(((CommonBaseEvent) (object)));
            
            ContentHandler currentContentHandler = commonBaseEvent.getContentHandler();
            
            commonBaseEvent.setContentHandler(templateContentHandler);
            
            try {
                commonBaseEvent.complete();                
            } 
            catch (CompletionException c) {
                //Ignore since content completion is based on 'best-effort'.
            }        
            
            commonBaseEvent.setContentHandler(currentContentHandler);            
        } 
        else {

            //Step 1: Set the default properties on the Common Base Event:

            //Set the global instance ID:
            if(commonBaseEvent.getGlobalInstanceId() == null){
                commonBaseEvent.setGlobalInstanceId(EventFactory.eINSTANCE.createGlobalInstanceId());
            }

            //Set the creation time to the current time:
            if(!commonBaseEvent.isSetCreationTime()){
                commonBaseEvent.setCreationTimeAsLong(System.currentTimeMillis());
            }

            //Set the situation to a default situation:
            if(commonBaseEvent.getSituation() == null){
                
	            Situation reportSituation = EventFactory.eINSTANCE.createSituation();
	            reportSituation.setReportSituation("INTERNAL", "LOG");
	
	            commonBaseEvent.setSituation(reportSituation);
            }

            //Set the source component ID to a default component ID:
            if(commonBaseEvent.getSourceComponentId() == null){
                commonBaseEvent.setSourceComponentId(null, "Logging", "Logger", "Logging_Application", "Application", Constants.LOCAL_HOST_IP_ADDRESS, ComponentIdentification.LOCATION_TYPE_IPV4);
            }

            //Step 2: Generate the Extended Data Element(s):
            commonBaseEvent.addExtendedDataElement(convertObjectToExtendedDataElement(object, null, maxReferences, 0));
        }
    }

    /**
     * Converts the parameter object to a <code>ExtendedDataElement</code>
     * using the following rules:
     * <p>
     * <ol>
     * <li>The package and class name of the object is the &apos;name&apos; property of the
     * <code>ExtendedDataElement</code>.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * 
     * @param object
     *            The complex object to be converted to an Extended Data Element.
     * @param instanceName
     *            The instance name of the complex object to be converted to an Extended Data Element.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     * @param currentReferenceCount
     *            The current number of referenced complex objects traversed.
     * @return The generated Extended Data Element of the parameter complex
     *         object.
     */
    private static ExtendedDataElement convertObjectToExtendedDataElement(Object object, String instanceName, int maxReferences, int currentReferenceCount) {

        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();

        if (object != null) {

            Class objectsClass = object.getClass();

            //Case: Simple type:
            if (isSimpleType(object)) {
                
                String objectToString = object.toString().trim();
                int valuesLength = objectToString.length();

                if (valuesLength > 1024) {

                    ArrayList valuesParts = new ArrayList();

                    for (int counter = 0; counter < valuesLength; counter += 1024) {
                        valuesParts.add(objectToString.substring(counter, Math.min((counter + 1024), valuesLength)));
                    }

                    extendedDataElement.setValuesAsStringArray(((String[]) (valuesParts.toArray(new String[valuesParts.size()]))));
                } 
                else {
                    extendedDataElement.setValuesAsString(objectToString);
                }
            }

            //Case: Array:
            else if (objectsClass.isArray()) {

                //Case: boolean array:
                if (object instanceof boolean[]) {

                    boolean[] array = ((boolean[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Boolean(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: char array:
                else if (object instanceof char[]) {

                    char[] array = ((char[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Character(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: byte array:
                else if (object instanceof byte[]) {

                    byte[] array = ((byte[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Byte(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: short array:
                else if (object instanceof short[]) {

                    short[] array = ((short[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Short(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: int array:
                else if (object instanceof int[]) {

                    int[] array = ((int[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Integer(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: long array:
                else if (object instanceof long[]) {

                    long[] array = ((long[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Long(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: float array:
                else if (object instanceof float[]) {

                    float[] array = ((float[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Float(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: double array:
                else if (object instanceof double[]) {

                    double[] array = ((double[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Double(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: Default (e.g. object arrays):
                else {

                    if(currentReferenceCount == maxReferences){            
                        extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                    }
                    else{

                        Object[] array = ((Object[]) (object));
    
                        for (int counter = 0; counter < array.length; counter++) {
                            extendedDataElement.addChild(convertObjectToExtendedDataElement(array[counter], null, maxReferences, (currentReferenceCount + 1)));
                        }
                    }
                }
            }

            //Case: Collection:
            else if (object instanceof Collection) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{
    
                    Iterator iterator = ((Collection) (object)).iterator();
    
                    while (iterator.hasNext()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(iterator.next(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Enumeration:
            else if (object instanceof Enumeration) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));                      
                }
                else{

                    Enumeration enumeration = ((Enumeration) (object));
    
                    while (enumeration.hasMoreElements()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(enumeration.nextElement(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Iterator:
            else if (object instanceof Iterator) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{

                    Iterator iterator = ((Iterator) (object));

                    while (iterator.hasNext()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(iterator.next(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Thread:
            else if (object instanceof Thread) {

                Thread thread = ((Thread) (object));

                extendedDataElement.addChild("Name", thread.getName());
                extendedDataElement.addChild("Priority", String.valueOf(thread.getPriority()));
                extendedDataElement.addChild("Group", thread.getThreadGroup().getName());
            }

            //Case: Throwable:
            //NOTE: Use the default Throwable --> Common Base Event mapping (see EventHelpers.convertToExtendedDataElement())
            //      to leverage existing tool.  As such, ignore the the maximum number of referenced causal Throwables that
            //      that are traversed.
            else if (object instanceof Throwable) {
                extendedDataElement = EventHelpers.convertToExtendedDataElement(((Throwable) (object)));
            }

            //Case: Map:
            else if (object instanceof Map) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{

                    Map map = ((Map) (object));
                    Iterator keys = map.keySet().iterator();
                    ExtendedDataElement entryExtendedDataElement = null;
                    ExtendedDataElement keyExtendedDataElement = null;
                    ExtendedDataElement valueExtendedDataElement = null;
                    int counter = 1;
    
                    while (keys.hasNext()) {
    
                        Object key = keys.next();
    
                        entryExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        entryExtendedDataElement.setName("Entry_".concat(String.valueOf(counter++)));
                        entryExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    
                        keyExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        keyExtendedDataElement.setName("Key");
                        keyExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
                        keyExtendedDataElement.addChild(convertObjectToExtendedDataElement(key, null, maxReferences, (currentReferenceCount + 1)));
    
                        entryExtendedDataElement.addChild(keyExtendedDataElement);
    
                        valueExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        valueExtendedDataElement.setName("Value");
                        valueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
                        valueExtendedDataElement.addChild(convertObjectToExtendedDataElement(map.get(key), null, maxReferences, (currentReferenceCount + 1)));
    
                        entryExtendedDataElement.addChild(valueExtendedDataElement);
    
                        extendedDataElement.addChild(entryExtendedDataElement);
                    }
                }
            }
            
            //Case: EventQueue:
            else if (object instanceof EventQueue) {
                //Ignore.
            }

            //Case: Class:
            else if (object instanceof Class) {

                Class theClass = ((Class) (object));

                extendedDataElement.addChild("Name", getClassName(theClass));

                if (theClass.isPrimitive()) {
                    extendedDataElement.addChild("Type", "primitive");
                } 
                else if (theClass.isArray()) {
                    extendedDataElement.addChild("Type", "array");
                } 
                else if (theClass.isInterface()) {
                    extendedDataElement.addChild("Type", "interface");
                } 
                else {
                    extendedDataElement.addChild("Type", "class");
                }

                Package thePackage = theClass.getPackage();

                if (thePackage != null) {
                    extendedDataElement.addChild("Package", thePackage.getName());
                }

                extendedDataElement.addChild("Modifers", Modifier.toString(theClass.getModifiers()));

                extendedDataElement.addChild("Superclass", getClassName(theClass.getSuperclass()));
            }

            //Case: Default:
            else {

                Method[] objectsMethods = objectsClass.getMethods();
                String methodName = null;
                Object returnObject = null;

                for (int counter = 0; counter < objectsMethods.length; counter++) {

                    methodName = objectsMethods[counter].getName().trim();

                    if ((methodName.length() > 3) && (methodName.startsWith("get")) && (objectsMethods[counter].getParameterTypes().length == 0)) {

                        try {
                            returnObject = objectsMethods[counter].invoke(object, null);
                        } 
                        catch (Throwable t) {
                            returnObject = null;
                        }

                        if((returnObject != null) && (!isSimpleType(returnObject)) && (currentReferenceCount == maxReferences)){            
                            extendedDataElement.addChild(createEndOfNestingExtendedDataElement(methodName.substring(3)));           
                        }
                        else{
                            extendedDataElement.addChild(convertObjectToExtendedDataElement(returnObject, methodName.substring(3), maxReferences, (currentReferenceCount + 1)));
                        }
                    }
                }
            }
        }

        extendedDataElement.setName(getClassName(object));

        if (extendedDataElement.getType() == null) {
            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
        }

        if ((instanceName != null) && (instanceName.trim().length() > 0)) {
            extendedDataElement.addChild("Instance_Name", instanceName);
        }

        return extendedDataElement;
    }

    private static ExtendedDataElement createEndOfNestingExtendedDataElement(String instanceName){
        
        ExtendedDataElement endExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        endExtendedDataElement.setName("End_Of_Nesting");
        endExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
        
        if ((instanceName != null) && (instanceName.trim().length() > 0)) {
            endExtendedDataElement.addChild("Instance_Name", instanceName);
        }
        
        return endExtendedDataElement;
    }
    
    /**
     * Returns the package and class name of parameter object.
     * <p>
     * 
     * @param object
     *            The object for which its package and class name is derived.
     * @return String The package and class name of parameter object.
     */
    private static String getClassName(Object object) {

        if (object == null) { 
            return "null"; 
        }

        Class theClass = object.getClass();
        String name = theClass.getName().trim();
        
        //Check if the class is an array:
        if (theClass.isArray()) {

            int index = (name.lastIndexOf('[') + 1);

            String encodingString = name.substring(index).trim();
            char encoding = encodingString.charAt(0);
            String type = "";

            if ((encoding == 'L') && (encodingString.endsWith(";"))) {
                type = encodingString.substring(1, (encodingString.length() - 1));
            } 
            else if (encoding == 'B') {
                type = "byte";
            } 
            else if (encoding == 'C') {
                type = "char";
            } 
            else if (encoding == 'D') {
                type = "double";
            } 
            else if (encoding == 'F') {
                type = "float";
            } 
            else if (encoding == 'I') {
                type = "int";
            } 
            else if (encoding == 'J') {
                type = "long";
            } 
            else if (encoding == 'S') {
                type = "short";
            } 
            else if (encoding == 'Z') {
                type = "boolean";
            }
            else{
                type = encodingString.substring(1, encodingString.length());
            }

            StringBuffer nameBuffer = new StringBuffer();
            nameBuffer.append(type);
            
            int depth = name.substring(0, index).trim().length();

            for (int counter = 0; counter < depth; counter++) {
                nameBuffer.append("[]");
            }
            
            return (nameBuffer.toString());
        }

        return name;
    }
    
    /**
     * Determines if the parameter non-null object is a simple type.
     * <p>
     * Simple types include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * 
     * @param theClass The non-null object for which its type is derived.
     * @return True if the parameter non-null object is a simple type, otherwise false.
     */
    private static boolean isSimpleType(Object object) {
        return ((object.getClass().isPrimitive()) || (object instanceof Boolean) || (object instanceof Character) || (object instanceof Byte) || (object instanceof Short) || (object instanceof Integer) || (object instanceof Long) || (object instanceof Float) || (object instanceof Double) || (object instanceof BigDecimal) || (object instanceof BigInteger) || (object instanceof String) || (object instanceof StringBuffer));
    }
}