package org.eclipse.hyades.logging.events.cbe.util.tests;

import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.tests.Util;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventHelpersTest.java,v 1.6 2005/02/16 22:20:41 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventHelpersTest extends TestCase {

    /**
     * Constructor for EventHelpersTest.
     * @param arg0
     */
    public EventHelpersTest(String testcase) {
        super(testcase);
    }

    /*
     * @see TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
    }

    /*
     * @see TestCase#tearDown()
     */
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * Test the convertToExtendedDataElement(Throwable) API in a simulated JDK 1.3.x or below run-time environment
     * and the default name.
     */
    public void testConvertToExtendedDataElementOnJDK13xDefaultName() {

        //TODO: Comment this line when running in a simulated JDK 1.3.x or below run-time environment:
        fail("Please re-run this test case with the EventHelpers.isJava14xRunTime flag set to false before compilation.");

        try {
           
            Throwable throwable = new Throwable("ThrowableMessage");

            //Test a routine execution:
            ExtendedDataElement extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            StackTraceElement stackTraceElements[] = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test with no stack trace:
            throwable.setStackTrace(new StackTraceElement[]{});

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
                        
            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test a 'null' Throwable:
            extendedDataElement = EventHelpers.convertToExtendedDataElement(null);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), null);
            
            assertEquals(extendedDataElement.getChildren().size(), 0);       
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }
    
    /**
     * Test the convertToExtendedDataElement(Throwable) API in a JDK 1.4.x or above run-time environment and
     * the default name.
     */
    public void testConvertToExtendedDataElementOnJDK14xDefaultName() {

        try {

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowableMessage" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowableMessage" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("ThrowableMessage", child);

            //Test a routine execution:
            ExtendedDataElement extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            StackTraceElement stackTraceElements[] = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            Throwable causeThrowable = throwable;
            ExtendedDataElement causeExtendedDataElement = extendedDataElement;

            for (int counter = 0; counter < THROWABLES_LEVELS; counter++) {

                causeThrowable = causeThrowable.getCause();
                
                assertNotNull(causeThrowable);
                    
                assertEquals(causeExtendedDataElement.getChildren().size(), 1);
                
                causeExtendedDataElement = ((ExtendedDataElement)(causeExtendedDataElement.getChildren().get(0)));
                
                assertEquals(causeExtendedDataElement.getName(), "Cause");

                assertEquals(causeExtendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

                stackTraceElements = causeThrowable.getStackTrace();
                
                assertEquals(causeExtendedDataElement.getValues().size(), (stackTraceElements.length + 1));
                
                assertEquals(((String)(causeExtendedDataElement.getValues().get(0))), causeThrowable.toString());
                
                for (int index = 0; index < stackTraceElements.length; index++) {
                    assertEquals(((String)(causeExtendedDataElement.getValues().get(index + 1))), stackTraceElements[index].toString());                
                }
            }
            
            assertNull(causeThrowable.getCause());
            
            assertEquals(causeExtendedDataElement.getChildren().size(), 0);
            
            //Test with no stack trace:
            throwable.setStackTrace(new StackTraceElement[]{});

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            causeThrowable = throwable;
            causeExtendedDataElement = extendedDataElement;

            for (int counter = 0; counter < THROWABLES_LEVELS; counter++) {

                causeThrowable = causeThrowable.getCause();
                
                assertNotNull(causeThrowable);
                    
                assertEquals(causeExtendedDataElement.getChildren().size(), 1);
                
                causeExtendedDataElement = ((ExtendedDataElement)(causeExtendedDataElement.getChildren().get(0)));
                
                assertEquals(causeExtendedDataElement.getName(), "Cause");

                assertEquals(causeExtendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

                stackTraceElements = causeThrowable.getStackTrace();
                
                assertEquals(causeExtendedDataElement.getValues().size(), (stackTraceElements.length + 1));
                
                assertEquals(((String)(causeExtendedDataElement.getValues().get(0))), causeThrowable.toString());
                
                for (int index = 0; index < stackTraceElements.length; index++) {
                    assertEquals(((String)(causeExtendedDataElement.getValues().get(index + 1))), stackTraceElements[index].toString());                
                }
            }
            
            assertNull(causeThrowable.getCause());
            
            assertEquals(causeExtendedDataElement.getChildren().size(), 0);
            
            //Test with no cause:
            throwable = new Throwable("ThrowableMessage");

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            stackTraceElements = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test a 'null' Throwable:
            extendedDataElement = EventHelpers.convertToExtendedDataElement(null);

            assertEquals(extendedDataElement.getName(), "Throwable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), null);
            
            assertEquals(extendedDataElement.getChildren().size(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }
    
    /**
     * Test the convertToExtendedDataElement(Throwable) API in a simulated JDK 1.3.x or below run-time environment
     * and a custom name.
     */
    public void testConvertToExtendedDataElementOnJDK13xCustomName() {

        //TODO: Comment this line when running in a simulated JDK 1.3.x or below run-time environment:
        fail("Please re-run this test case with the EventHelpers.isJava14xRunTime flag set to false before compilation.");

        try {
           
            Throwable throwable = new Throwable("ThrowableMessage");

            //Test a routine execution:
            ExtendedDataElement extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable,"CustomThrowable");

            assertEquals(extendedDataElement.getName(),"CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            StackTraceElement stackTraceElements[] = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test with no stack trace:
            throwable.setStackTrace(new StackTraceElement[]{});

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable,"CustomThrowable");

            assertEquals(extendedDataElement.getName(),"CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
                        
            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test a 'null' Throwable:
            extendedDataElement = EventHelpers.convertToExtendedDataElement(null,"CustomThrowable");

            assertEquals(extendedDataElement.getName(),"CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), null);
            
            assertEquals(extendedDataElement.getChildren().size(), 0);       
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }
    
    /**
     * Test the convertToExtendedDataElement(Throwable) API in a JDK 1.4.x or above run-time environment and
     * a custom name.
     */
    public void testConvertToExtendedDataElementOnJDK14xCustomName() {

        try {

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowableMessage" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowableMessage" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("ThrowableMessage", child);

            //Test a routine execution:
            ExtendedDataElement extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable,"CustomThrowable");

            assertEquals(extendedDataElement.getName(), "CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            StackTraceElement stackTraceElements[] = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            Throwable causeThrowable = throwable;
            ExtendedDataElement causeExtendedDataElement = extendedDataElement;

            for (int counter = 0; counter < THROWABLES_LEVELS; counter++) {

                causeThrowable = causeThrowable.getCause();
                
                assertNotNull(causeThrowable);
                    
                assertEquals(causeExtendedDataElement.getChildren().size(), 1);
                
                causeExtendedDataElement = ((ExtendedDataElement)(causeExtendedDataElement.getChildren().get(0)));
                
                assertEquals(causeExtendedDataElement.getName(), "Cause");

                assertEquals(causeExtendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

                stackTraceElements = causeThrowable.getStackTrace();
                
                assertEquals(causeExtendedDataElement.getValues().size(), (stackTraceElements.length + 1));
                
                assertEquals(((String)(causeExtendedDataElement.getValues().get(0))), causeThrowable.toString());
                
                for (int index = 0; index < stackTraceElements.length; index++) {
                    assertEquals(((String)(causeExtendedDataElement.getValues().get(index + 1))), stackTraceElements[index].toString());                
                }
            }
            
            assertNull(causeThrowable.getCause());
            
            assertEquals(causeExtendedDataElement.getChildren().size(), 0);
            
            //Test with no stack trace:
            throwable.setStackTrace(new StackTraceElement[]{});

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable,"CustomThrowable");

            assertEquals(extendedDataElement.getName(), "CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            causeThrowable = throwable;
            causeExtendedDataElement = extendedDataElement;

            for (int counter = 0; counter < THROWABLES_LEVELS; counter++) {

                causeThrowable = causeThrowable.getCause();
                
                assertNotNull(causeThrowable);
                    
                assertEquals(causeExtendedDataElement.getChildren().size(), 1);
                
                causeExtendedDataElement = ((ExtendedDataElement)(causeExtendedDataElement.getChildren().get(0)));
                
                assertEquals(causeExtendedDataElement.getName(), "Cause");

                assertEquals(causeExtendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

                stackTraceElements = causeThrowable.getStackTrace();
                
                assertEquals(causeExtendedDataElement.getValues().size(), (stackTraceElements.length + 1));
                
                assertEquals(((String)(causeExtendedDataElement.getValues().get(0))), causeThrowable.toString());
                
                for (int index = 0; index < stackTraceElements.length; index++) {
                    assertEquals(((String)(causeExtendedDataElement.getValues().get(index + 1))), stackTraceElements[index].toString());                
                }
            }
            
            assertNull(causeThrowable.getCause());
            
            assertEquals(causeExtendedDataElement.getChildren().size(), 0);
            
            //Test with no cause:
            throwable = new Throwable("ThrowableMessage");

            extendedDataElement = EventHelpers.convertToExtendedDataElement(throwable,"CustomThrowable");

            assertEquals(extendedDataElement.getName(),"CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            stackTraceElements = throwable.getStackTrace();
            
            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), throwable.toString());
            
            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String)(extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());                
            }

            assertEquals(extendedDataElement.getChildren().size(), 0);

            //Test a 'null' Throwable:
            extendedDataElement = EventHelpers.convertToExtendedDataElement(null,"CustomThrowable");

            assertEquals(extendedDataElement.getName(), "CustomThrowable");

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);
            
            assertEquals(((String)(extendedDataElement.getValues().get(0))), null);
            
            assertEquals(extendedDataElement.getChildren().size(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test the getExtendedDataElementValuesArray(String) API.
     */
    public void testGetExtendedDataElementValuesArray() {

        try {

            //Test a null string:
            String[] values = EventHelpers.getExtendedDataElementValuesArray(null);
            
            assertNotNull(values);
            
            assertEquals(values.length,1);
            
            assertEquals(values[0],null);
            
            //Test an empty string:            
            String valueString = "";
            
            values = EventHelpers.getExtendedDataElementValuesArray(valueString);
            
            assertNotNull(values);
            
            assertEquals(values.length,1);
            
            assertEquals(values[0],valueString);

            //Test a string with one character:            
            valueString = "s";
            
            values = EventHelpers.getExtendedDataElementValuesArray(valueString);
            
            assertNotNull(values);
            
            assertEquals(values.length,1);
            
            assertEquals(values[0],valueString);
            
            //Test a string of length 1024:
            for (int counter = 0; counter < 1023; counter++) {
                valueString += "s";
            }
            
            values = EventHelpers.getExtendedDataElementValuesArray(valueString);
            
            assertNotNull(values);
            
            assertEquals(values.length,1);
            
            assertEquals(values[0],valueString);
            
            //Test a string of length 1025:
            valueString += "s";
            
            values = EventHelpers.getExtendedDataElementValuesArray(valueString);
            
            assertNotNull(values);
            
            assertEquals(values.length,2);
            
            assertEquals(values[0],valueString.substring(0,1024));
            assertEquals(values[1],valueString.substring(1024));
            
            //Test a string of length 1025:
            valueString += valueString;
            
            values = EventHelpers.getExtendedDataElementValuesArray(valueString);
            
            assertNotNull(values);
            
            assertEquals(values.length,3);
            
            assertEquals(values[0],valueString.substring(0,1024));
            assertEquals(values[1],valueString.substring(1024,2048));            
            assertEquals(values[2],valueString.substring(2048));            
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }
    
    /**
     * Tests whether the comparison between two EMF objects.
     */
    public void testCompareEvent() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        assertTrue(cbe1.equals(cbe1));
        CommonBaseEvent cbe2 = (CommonBaseEvent) (Util.clone(cbe1));

        assertTrue(cbe1.equals(cbe2));
    }

    public void testConvertingLocaleSensitiveDates(){
                
        final long CURRENT_TIME_MILLIS = System.currentTimeMillis();

        Locale[] locales = Locale.getAvailableLocales();
        
        Locale defaultLocale = Locale.getDefault();
        
        ExtendedDataElement extendedDataElement = null;
        
        for (int counter = 0; counter < locales.length; counter++) {

            //Change the locale:
            Locale.setDefault(locales[counter]);

            extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();

            extendedDataElement.setValuesAsDateAsLong(CURRENT_TIME_MILLIS);            
            
            assertTrue(CURRENT_TIME_MILLIS == extendedDataElement.getValuesAsDateAsLong());
        }        
        
        //Reset the default locale:
        Locale.setDefault(defaultLocale);
    }
    
    /**
     * Tests whether order of elements is considered 
     * important for ExtendedDataElement.values
     */
    public void testCompareExtendedDataElementValues() {
        EventFactory eFx = EventFactory.eINSTANCE;

        ExtendedDataElement ede1 = eFx.createExtendedDataElement();
        ede1.getValues().add("A");
        ede1.getValues().add("B");

        ExtendedDataElement ede2 = eFx.createExtendedDataElement();
        ede2.getValues().add("A");
        ede2.getValues().add("B");

        assertEquals(ede1, ede2);

        ede1.getValues().clear();
        ede1.getValues().add("A");
        ede1.getValues().add("B");
        ede2.getValues().clear();
        ede2.getValues().add("B");
        ede2.getValues().add("A");

        assertTrue(ede1.equals(ede2) == false);
    }

    /**
     * Tests whether order of elements is considered 
     * important for MsgDataElement.msgCatalogTokens
     */
    public void testCompareMsgDataElementMsgCatalogTokens() {
        EventFactory eFx = EventFactory.eINSTANCE;

        MsgDataElement mde1 = eFx.createMsgDataElement();
        mde1.addMsgCatalogTokenAsString("A");
        mde1.addMsgCatalogTokenAsString("B");

        MsgDataElement mde2 = eFx.createMsgDataElement();
        mde2.addMsgCatalogTokenAsString("A");
        mde2.addMsgCatalogTokenAsString("B");

        assertEquals(mde1, mde2);

        mde1.getMsgCatalogTokens().clear();
        mde1.addMsgCatalogTokenAsString("A");
        mde1.addMsgCatalogTokenAsString("B");
        mde2.getMsgCatalogTokens().clear();
        mde2.addMsgCatalogTokenAsString("B");
        mde2.addMsgCatalogTokenAsString("A");

        assertTrue(mde1.equals(mde2) == false);
    }

    /**
     * Tests whether the comparison between two EMF objects fails when
     * the elements of internal lists have different numbers of copies
     * of the same element (AABA is not the same as AABB).
     */
    public void testCompareEObjectDuplicateListElements() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        // The idea is to get two lists (AABBB) and (AAABB)
        final int listDivide = 2;
        final int listElements = listDivide + 2;
        for (int i = 0; i < listElements; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            String contextId = "A";
            if (i >= listDivide) {
                contextId = "B";
            }
            cde1.setContextId(contextId);
            cbe1.getContextDataElements().add(cde1);

            if (i == listDivide) {
                contextId = "A";
            }
            ContextDataElement cde2 = EventFactory.eINSTANCE.createContextDataElement();
            cde2.setContextId(contextId);
            cbe2.getContextDataElements().add(cde2);
        }

        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects succeeds when
     * the elements of internal lists don't follow the same order.
     */
    public void testCompareEObjectDifferentListOrder() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        final int listElements = 5;
        for (int i = 0; i < listElements; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            cde1.setContextId(Integer.toString(i));
            cbe1.getContextDataElements().add(cde1);

            ContextDataElement cde2 = EventFactory.eINSTANCE.createContextDataElement();
            cde2.setContextId(Integer.toString(listElements - i - 1));
            cbe2.getContextDataElements().add(cde2);
        }

        assertTrue(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects fails when 
     * there are differences in their basic attributes.
     */
    public void testCompareEObjectDifferentAttribute() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        ComponentIdentification cit = EventFactory.eINSTANCE.createComponentIdentification();
        cbe2.setSourceComponentId(null);
        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects fails 
     * when the contents of the contained objects differ.
     */
    public void testCompareEObjectDifferentContainment() {
        EventPackage ePck = EventPackage.eINSTANCE;
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        // Testing whether different values will be caught.
        cbe1.getSourceComponentId().setApplication("something");
        cbe2.getSourceComponentId().setApplication("something different");
        assertFalse(cbe1.equals(cbe2));

        // Testing whether different pointer values will be caught.
        cbe1.getSourceComponentId().setApplication(null);
        cbe2.getSourceComponentId().setApplication("something different");
        assertFalse(cbe1.equals(cbe2));

        // Thread id is the last attribute inside the component id, may be
        // a boundary condition.
        cbe1.getSourceComponentId().setApplication("same");
        cbe2.getSourceComponentId().setApplication("same");
        cbe1.getSourceComponentId().setThreadId("id");
        cbe2.getSourceComponentId().setThreadId("id2");
        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison returns false when the object types
     * are not EMF types.
     */
    public void testCompareNotEObjects() {
        assertFalse(EventHelpers.compareEObject(new String(), new String()));
        CommonBaseEvent cbe = createReferenceEvent();
        assertFalse(cbe.equals(new String()));
        assertFalse(EventHelpers.compareEObject(new String(), cbe));
        String base = "AAA";
        assertFalse(EventHelpers.compareEObject(base, base));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareEObjectsNull() {
        CommonBaseEvent cbe = createReferenceEvent();
        assertFalse(EventHelpers.compareEObject(cbe, null));
        assertFalse(EventHelpers.compareEObject(null, cbe));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareEObjectsNotSet() {
        CommonBaseEvent cbe = createReferenceEvent();
        CommonBaseEvent cbe1 = createReferenceEvent();
        cbe.unsetRepeatCount();
        cbe1.setRepeatCount((short) 0);
        assertFalse(EventHelpers.compareEObject(cbe, cbe1));
        assertFalse(EventHelpers.compareEObject(cbe1, cbe));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareSituationTypes() {
        EventPackage ePck = EventPackage.eINSTANCE;
        EClass situationClass = ePck.getSituationType();
        EFactory eFx = ePck.getEFactoryInstance();

        List l = ePck.eContents();
        Iterator i = l.iterator();
        while (i.hasNext()) {
            EClass c = (EClass) i.next();
            if (c.isInstance(situationClass)) {
                EObject o1 = eFx.create(c);
                EObject o2 = eFx.create(c);
                assertTrue(o1.eClass().getName() + ".equals failed", o1.equals(o2));
            }
        }
    }

    /**
     * 
     */
    public void testDifferentClasses() {
        EventFactory efx = EventFactory.eINSTANCE;

        CommonBaseEvent cbe = efx.createCommonBaseEvent();
        ExtendedDataElement ebe = efx.createExtendedDataElement();
        EventHelpers.compareEObject(cbe, ebe);
    }

    /*
     * Private methods.
     */

    /**
     * Creates a reference event for all comparison.
     * <P>
     * The individual test fixture modify the attributes in order to match
     * the test scenario.
     */
    private CommonBaseEvent createReferenceEvent() {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();
        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("AAA");
        result.setElapsedTime(65);
        result.setExtensionName("Event_class");
        result.setLocalInstanceId("id");
        result.setMsg("msg");
        result.setPriority((short) 32);
        result.setRepeatCount((short) 12);
        result.setSequenceNumber(88);
        result.setSeverity((short) 67);
        result.setVersion("1.0.1");

        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("cat");
        mde.setMsgCatalogId("msgCatId");
        mde.setMsgCatalogType("type");
        mde.setMsgId("abc");
        mde.setMsgIdType("idType");
        mde.setMsgLocale("locale");
        mde.addMsgCatalogTokenAsString("token1");
        result.setMsgDataElement(mde);

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cit1.setApplication("App");
        cit1.setComponent("comp");
        cit1.setComponentIdType("idType");
        cit1.setComponentType("comp");
        cit1.setExecutionEnvironment("env");
        cit1.setInstanceId("instId");
        cit1.setLocation("location");
        cit1.setLocationType("type");
        cit1.setProcessId("process");
        cit1.setSubComponent("sub");
        cit1.setThreadId("threadid");
        result.setReporterComponentId(cit1);

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setApplication("App2");
        cit2.setComponent("comp2");
        cit2.setComponentIdType("idType2");
        cit2.setComponentType("comp2");
        cit2.setExecutionEnvironment("env2");
        cit2.setInstanceId("instId2");
        cit2.setLocation("location2");
        cit2.setLocationType("type2");
        cit2.setProcessId("process2");
        cit2.setSubComponent("sub2");
        cit2.setThreadId("threadid2");
        result.setSourceComponentId(cit2);

        result.getAny().add("BSD1");
        result.getAny().add("BSD2");

        for (int i = 0; i < 5; i++) {
            AssociatedEvent ae1 = EventFactory.eINSTANCE.createAssociatedEvent();
            ae1.setAssociationEngine("ABC" + i);
            AssociationEngine aei = EventFactory.eINSTANCE.createAssociationEngine();
            aei.setId("A" + i);
            aei.setName("B" + i);
            aei.setType("CFG" + i);
            ae1.setAssociationEngineInfo(aei);
            ae1.setResolvedEvents("tr YU IO" + i);
            result.getAssociatedEvents().add(ae1);
        }

        for (int i = 0; i < 5; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            cde1.setContextId("id" + i);
            cde1.setContextValue("bfg" + i);
            cde1.setName("rew" + i);
            cde1.setType("tr4" + i);
            result.getContextDataElements().add(cde1);
        }

        for (int i = 0; i < 5; i++) {
            ExtendedDataElement ede1 = EventFactory.eINSTANCE.createExtendedDataElement();
            ede1.setName("tre" + i);
            ede1.setType("string" + i);
            ede1.getValues().add("text" + i);
            result.getExtendedDataElements().add(ede1);
        }

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("av");
        sitType.setOperationDisposition("op");
        sitType.setProcessingDisposition("pro");
        sitType.setReasoningScope("reason");
        sit.setCategoryName("cat");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        return result;
    }

    /**
     * Tests whether the comparison for ExtendedDataElement.hexValue
     * is case insensitive.
     */
    public void testCompareExtendedDataElementHexValue() {
        ExtendedDataElement ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setHexValue("ABc123");
        ExtendedDataElement ede1 = EventFactory.eINSTANCE.createExtendedDataElement();
        ede1.setHexValue("abC123");

        assertEquals(ede, ede1);

    }
    
    public void testGetFormattedDateStringAPIWithLongParameter(){
    	try{    		    		
    		TimeZone.setDefault(new SimpleTimeZone(-18000000,"EST"));
    		
    		assertEquals(EventHelpers.getFormattedDateString(1009800000000L), "December 31, 2001 12:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00Z")), "December 31, 2001 11:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123Z")), "December 31, 2001 11:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00-05:00")), "January 1, 2002 4:00:00.000 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00+05:00")), "December 31, 2001 6:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123-05:00")), "January 1, 2002 4:00:00.123 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123+05:00")), "December 31, 2001 6:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00-04:00")), "January 1, 2002 3:00:00.000 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00+04:00")), "December 31, 2001 7:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123-04:00")), "January 1, 2002 3:00:00.123 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123+04:00")), "December 31, 2001 7:00:00.123 PM GMT");
    		
    		TimeZone.setDefault(new SimpleTimeZone(0,"UTC"));
    		
    		assertEquals(EventHelpers.getFormattedDateString(1009800000000L), "December 31, 2001 12:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00Z")), "December 31, 2001 11:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123Z")), "December 31, 2001 11:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00-05:00")), "January 1, 2002 4:00:00.000 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00+05:00")), "December 31, 2001 6:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123-05:00")), "January 1, 2002 4:00:00.123 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123+05:00")), "December 31, 2001 6:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00-04:00")), "January 1, 2002 3:00:00.000 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00+04:00")), "December 31, 2001 7:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123-04:00")), "January 1, 2002 3:00:00.123 AM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString(EventHelpers.dateToLong("2001-12-31T23:00:00.123+04:00")), "December 31, 2001 7:00:00.123 PM GMT");
    	} 
    	catch (AssertionFailedError a) {
    		throw (a);
    	} 
    	catch (Throwable t) {
    		assertTrue(t.toString(), false);
    	}
    }
    
    public void testGetFormattedDateStringAPIWithStringParameter(){
    	
    	try{
    		
    		TimeZone.setDefault(new SimpleTimeZone(-18000000,"EST"));
    		
    		assertEquals(EventHelpers.getFormattedDateString(1009800000000L), "December 31, 2001 12:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00Z"), "December 31, 2001 11:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123Z"), "December 31, 2001 11:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00-05:00"), "December 31, 2001 11:00:00.000 PM GMT-05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00+05:00"), "December 31, 2001 11:00:00.000 PM GMT+05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123-05:00"), "December 31, 2001 11:00:00.123 PM GMT-05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123+05:00"), "December 31, 2001 11:00:00.123 PM GMT+05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00-04:00"), "December 31, 2001 11:00:00.000 PM GMT-04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00+04:00"), "December 31, 2001 11:00:00.000 PM GMT+04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123-04:00"), "December 31, 2001 11:00:00.123 PM GMT-04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123+04:00"), "December 31, 2001 11:00:00.123 PM GMT+04:00");
    		
    		TimeZone.setDefault(new SimpleTimeZone(0,"UTC"));
    		
    		assertEquals(EventHelpers.getFormattedDateString(1009800000000L), "December 31, 2001 12:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00Z"), "December 31, 2001 11:00:00.000 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123Z"), "December 31, 2001 11:00:00.123 PM GMT");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00-05:00"), "December 31, 2001 11:00:00.000 PM GMT-05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00+05:00"), "December 31, 2001 11:00:00.000 PM GMT+05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123-05:00"), "December 31, 2001 11:00:00.123 PM GMT-05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123+05:00"), "December 31, 2001 11:00:00.123 PM GMT+05:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00-04:00"), "December 31, 2001 11:00:00.000 PM GMT-04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00+04:00"), "December 31, 2001 11:00:00.000 PM GMT+04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123-04:00"), "December 31, 2001 11:00:00.123 PM GMT-04:00");
    		
    		assertEquals(EventHelpers.getFormattedDateString("2001-12-31T23:00:00.123+04:00"), "December 31, 2001 11:00:00.123 PM GMT+04:00");
    	} 
    	catch (AssertionFailedError a) {
    		throw (a);
    	} 
    	catch (Throwable t) {
    		assertTrue(t.toString(), false);
    	}
    }
}
