/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecorderAgent.java,v 1.7 2005/04/20 16:31:54 jnevicos Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder.remote;

	
/**
 * This is the object which should be extended by an object which will provide the agent side functionality of a recorder
 * @author Ernest Jessee
 */
public abstract class RecorderAgent
{	
		
	/**
	 * The configuration parameters for this agent
	 */
	protected String configParameters = null;
	
	/**
	 * reference to the RecorderAgentController which instantiated this agent and will handle it's data messages
	 */
	protected RecorderAgentController controller;
	
	/**
	 * boolean indicating whether or not the agent is running and ready to record
	 */
	private  boolean isReady = false;
	
	
	/**
	 * commands to be handed by "handleCommand(...)";
	 */
	public static final int STOP_RECORDING=1;
	public static final int ABORT_RECORDING=2;
	public static final int INSERT_COMMENT=3;  //not implemented
	public static final int INSERT_SPLIT=4; //not implemented
	public static final int ANNOTATE_RECORDING = 5;

	
	/**
	 * This is the execution start point for the recorder agent.  At the point at which the runnable is ready to handle traffic, 
	 * setIsReady(true) should be called
	 * The thread started by this agent as the root of its operation should be returned for the death watcher.
	 */
	abstract public Thread run();
	
	/**
	 * called to initialize the agent
	 * @param String configParams - a semi-colon delimeted collection of comma delimated name-value pairs
	 * @return boolean
	 */
	abstract public boolean init(String configParams);
	
	/**
	 * called with a command should be handled (see commands above)
	 */
	abstract public boolean handleCommand(int commandID,String commandData)throws RecorderHandleCommandException;

	
	/**
	 * called to set the RecorderAgentController for this agent
	 * @param controller
	 * void
	 */
	public void setController(RecorderAgentController controller)
	{
		this.controller=controller;
	}
		
	/**
	 * called to determine if the agent is ready to perform
	 * @return boolean
	 */
	public synchronized boolean isReady()
	{
		return isReady;
	}
	
	/**
	 * should be called by the agent's run method when it is in a state to begin recording data
	 * @param isReady
	 * void
	 */
	public synchronized void setIsReady(boolean isReady)
	{
		this.isReady = isReady;
	}
	
	/**
	 * called by the agent and its "down-line" to send data across the Agent Controller data channel to the IRecorderDataProcessor on the other side
	 * @param byte[] data
	 * @param int iOffset
	 * @param int dataLength	 
	 */
	public synchronized void sendByteDataToDataProcessor(byte[] data, int iOffset, int dataLength)
	{
		controller.sendByteDataToDataProcessor(data, iOffset, dataLength);
	}

	/**
	 * called by the agent and its "down-line" to send a debug messate to the IRecorderDataProcessor on the other side
	 * @param String message
	 * void
	 */
	public synchronized void sendDebugMessageToDataProcessor(String message)
	{
		controller.sendDebugMessageToDataProcessor(message);	
	}

	
	/**
	 * called by the agent and its "down-line" to report an exception to IRecorderDataProcessor on the other side
	 * @param String explanation
	 * @param Throwable e
	 * void
	 */
	public void reportException(String explanation, Throwable e)
	{
		controller.reportException(explanation, e);
	}
	
	/**
	 * called by the agent and its "down-line" to send a contol message to the IRecorderDataProcessor on the other side.
	 * @param message
	 * void
	 */
	public void sendControlMessageToDataProcessor(String message)
	{
		controller.sendControlMessageToDataProcessor(message);
	}

}
