/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GlobalPacketQueue.java,v 1.5 2005/02/16 22:20:20 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.recorder.http.remote;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.MissingResourceException;

/**
 * @author mdunn
 * 
 */
public class GlobalPacketQueue extends Thread {
	
	static boolean NOTIFY_ON_ADD = false;
	
	/**
	 * Milliseconds to wait for inactivity.
	 */
	static long PACKET_WAIT_TOLERANCE = 500;
	
	
	static {
		try {
			String strValue = HttpRecResourceBundle.getInstance().getString("RECORDER_NOTIFY_ON_ADD");
			if (strValue.compareToIgnoreCase("TRUE") == 0) {
				NOTIFY_ON_ADD = true;
			}
		} catch (MissingResourceException e) {
			// Use default option
		}
		
		try {
			String strValue = HttpRecResourceBundle.getInstance().getString("RECORDER_PACKET_WAIT_TOLERANCE");
			PACKET_WAIT_TOLERANCE = Long.parseLong(strValue);
		} catch (Exception e) {
			// Use default option
		}
		
	}
	
	protected PacketWriter packetWriter;
	protected boolean shuttingDown = false;
	protected long timeOfLastPackedAdd = 0L;
	
	public GlobalPacketQueue (PacketWriter writer) {
		packetWriter = writer;
	}
	
	public void initiateShutdown () {
		shuttingDown = true;
		synchronized (this) {
			notify();
		}
	}
	
	public void run () {
		this.setName("PacketQueueWriter");
		//packetWriter.writeRecorderMessage(1,"IN GlobalPacketQueue.run(), PACKET_WAIT_TOLERANCE is: " + PACKET_WAIT_TOLERANCE);
		while (!shuttingDown) {
			synchronized (this) {
				if (packetQueue.size() == 0) {
					//If nothing on queue, wait for another thread to enqueue a packet
					try {
						wait();
					} catch (InterruptedException e) {
						//no problem
					}
				}
			}
			removeAndWrite ();
			Thread.yield();
		}
		removeAndWrite();
	}
	

	protected void removeAndWrite () {
		Collection packetsCollection = removeAll();
		if (packetsCollection != null) {
			Iterator iter = packetsCollection.iterator();
			while (iter.hasNext()) {
				byte[] bytePacket = (byte[])iter.next(); // get rid of refer
				
				// only write packet when no activity has 
				// occurred in a while
				waitForInactivity ();
				
				packetWriter.sendToDataProcessor (bytePacket);
				Thread.yield();
			}
		}
	}

	/**
	 * Sleep until no packet has been added for PACKET_WAIT_TOLERANCE
	 * from the previous packet add.
	 */
	protected void waitForInactivity () {
		long lastUpdate, curTime, nextCheckTime;
		while (true) {
			curTime = System.currentTimeMillis();
			lastUpdate = timeOfLastPackedAdd;
			nextCheckTime = lastUpdate + PACKET_WAIT_TOLERANCE;
			if (nextCheckTime > curTime) {
				try {
					Thread.sleep(nextCheckTime - curTime);
				} catch (InterruptedException e) {
					// doesn't matter
				}
			} else {
				break;
			}
		}
	}

	protected LinkedList packetQueue = new LinkedList();

	public void add (byte buf[], int offset, int len) {
		byte[] tmpBuff = new byte[len];
		System.arraycopy(buf,offset,tmpBuff,0,len);
		synchronized (this) {
			packetQueue.addLast(tmpBuff); 
			timeOfLastPackedAdd = System.currentTimeMillis();
			if (NOTIFY_ON_ADD) {
				notify();
			}
		}
	}
	
	public void add (String thisPacket) {
		byte []buf = thisPacket.getBytes();
		add (buf, 0, thisPacket.length());
	}
	/**
	 * Returns the LinkedList representing the queue and
	 * creates a new LinkedList to represent the queue.
	 * Returns null if no items are currently in the queue. 
	 * 
	 * @return LinkedList representing the queue.
	 */
	public Collection removeAll () {
		LinkedList workToDo = null;
		synchronized (this) {
			if (packetQueue.size() == 0) {
				workToDo = null;
			} else {
				workToDo = packetQueue;
				packetQueue = null;
				packetQueue = new LinkedList();
			}
		}
		return workToDo;
	}
}