/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecordArray.java,v 1.3 2005/02/16 22:20:27 qiyanli Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;


/**
 * This is the record for an array of variables.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class RecordArray {

    /**
     * an array of elements
     */
    private RecordVariable[] _elements;


    /**
     * Creates an array of elements using the <code>elementClass</code> and <code>elementValues</code>.
     * If <code>elementValues</code> is <code>null</code>, it will be treated as an empty array.
     * 
     * @param event         the name of the event
     * @param elementClass  the class of each element
     * @param elementValues an array of the values
     */
    public RecordArray(String event, Class elementClass, Object[] elementValues) {

        /* If any of the three input parameters is null, or if the parameter lists are of different lengths,
           the parameter list is assumed to be empty. */
        int listLength = elementValues == null ? 0 : elementValues.length;

        /* Create a parameter list with all the information. */
        _elements = new RecordVariable[listLength];
        for (int i = 0; i < listLength; i++) {
            _elements[i] = new RecordVariable(event, i, elementClass, elementValues[i]);
        }
    }


    /**
     * Creates a heterogeneous array using the <code>elementNames</code>, <code>elementClasses</code>,
     * and <code>elementValues</code>.  All three parameters must be of the same length, or all
     * parameters will be ignored.
     * 
     * @param event             the name of the event
     * @param elementIds        an array of the element IDs
     * @param elementClasses    the class of each element
     * @param elementValues     an array of the values
     */
    public RecordArray(String event, int[] elementIds, Class[] elementClasses, Object[] elementValues) {
        
        /* If any of the three input parameters is null, or if the parameter lists are of different lengths,
           the parameter list is assumed to be empty. */
        int listLength = (elementIds == null || elementClasses == null || elementValues == null ||
            elementIds.length != elementClasses.length || elementClasses.length != elementValues.length ?
            0 : elementIds.length);

        /* Create a parameter list with all the information. */
        _elements = new RecordVariable[listLength];
        for (int i = 0; i < listLength; i++) {
            _elements[i] = new RecordVariable(event, elementIds[i], elementClasses[i], elementValues[i]);
        }
    }


    /**
     * Converts this record into a string.
     * 
     * @return  this record in the string format
     */
    public String toString() {

        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < _elements.length; i++) {
            String elementString = _elements[i].toString();
            if (elementString != null) {
                buffer.append(_elements[i].toString() + '\n');
            }
        } 
        return buffer.toString();
    }
}
