/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileClientHandlerImplExtended.java,v 1.7 2005/05/16 23:49:31 jptoomey Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.collection.framework;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;

import org.eclipse.hyades.internal.execution.core.file.FileServerCommandFactory;
import org.eclipse.hyades.internal.execution.core.file.IFileServerCommand;
import org.eclipse.hyades.internal.execution.core.file.ISocketChannel;
import org.eclipse.hyades.internal.execution.file.FileServiceConstants;

/**
 * Handles a client request coming in to the file server, in the typical case
 * this runs on a new thread that is used for the purpose of servicing requests
 * and is not the same thread that the connection is accepted on
 * 
 * @author Scott E. Schneider
 */
class FileClientHandlerImplExtended implements IFileClientHandlerExtended, Runnable {

    /**
     * Channel to communicate with client
     */
    private final ISocketChannel clientChannel;

    /**
     * Creates a new file clietn handler implementation instance that will
     * likely run on a new thread just to handle the request
     * 
     * @param clientChannel
     *            the channel to communicate with the client
     */
    FileClientHandlerImplExtended(ISocketChannel clientChannel) {
        this.clientChannel = clientChannel;
    }

    /**
     * Service client file server request, need to bootstrap file server command
     * factory by first figuring out the command by the command identity sent in
     * the first 128 bytes of data. After this the file server command factory
     * can take the handling from here. The command that will eventually be
     * instantiated to handle the client request will encapsulate the process of
     * communication and request and response handling.
     */
    public void run() {

        // Allocate buffer to hold command identity length
        ByteBuffer identityLength = ByteBuffer.allocate(4);

        /*
         * Load command via file server command factory and execute, this should
         * be changed to use the simple communication abstract that abstract the
         * mechanism for communication (see AbstractFileServerCommand.State)
         */
        try {

            // Read command identity into buffer
            this.clientChannel.read(identityLength);
            identityLength.flip();

            // Read command identity data now
            int length = identityLength.getInt();

            /*
             * Determine if the legacy client protocol is in use, if so return
             * IO exception exception constant from legacy implementation, five
             * is a good number to pick, every legacy command id is less than
             * five and five is too short to be the command identity length
             * (typically a fully-qualified java class)
             */
            if (length < 5) {
                ByteBuffer fileNameLength = ByteBuffer.allocate(4);
                this.clientChannel.read(fileNameLength);
                fileNameLength.flip();
                ByteBuffer fileName = ByteBuffer.allocate(fileNameLength.getInt());
                this.clientChannel.read(fileName);
                ByteBuffer legacyException = ByteBuffer.allocate(1);
                legacyException.put((byte) FileServiceConstants.RA_IO_ERROR);
                legacyException.flip();
                this.clientChannel.write(legacyException);
                return; // return early if legacy
            }

            ByteBuffer identityData = ByteBuffer.allocate(length);
            this.clientChannel.read(identityData);
            identityData.flip();

            // Allocate a string for the command identity received
            String identity = null;

            /*
             * The following code is a fix carried over from the legacy file
             * client handler implementation, bugzilla_65922 -- z/OS triggered
             * change
             */
            try {

                // First try with UTF-8 encoding
                identity = new String(identityData.array(), 0, length, "UTF-8"); //$NON-NLS-1$

            } catch (UnsupportedEncodingException e) {

                // Any issues will trigger fallback to default encoding
                identity = new String(identityData.array(), 0, length);

            }

            // Clear up the buffers (not seemingly required)
            identityLength.clear();
            identityData.clear();

            // Instantiate named identity using filer server command factory
            IFileServerCommand command = FileServerCommandFactory.getInstance().createFileServerCommand(identity,
                    this.clientChannel);

            // Execute the command on the server-side
            command.execute();

        } catch (IOException e) {
        } finally {

            try {

                // Clean-up appropriately
                this.clientChannel.close();

            } catch (IOException e) {
            }

        }

    }
}