/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: FileConnectionHandlerExtended.java,v 1.5 2005/05/16 23:49:31 jptoomey Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.internal.collection.framework;

import java.io.IOException;
import java.net.Socket;
import java.nio.channels.SocketChannel;

import org.eclipse.hyades.internal.execution.core.file.ISocketChannel;
import org.eclipse.hyades.internal.execution.core.file.SocketChannelFactory;
import org.eclipse.hyades.internal.execution.file.IConnectionHandler;

/**
 * Extended implementation of the file connection handler that accepts socket
 * channels instead of plain sockets. The creator of this classes instances
 * expects the new connection handler interface rather then the legacy one.
 * 
 * @author Scott E. Schneider
 */
class FileConnectionHandlerExtended implements IConnectionHandler {

    /**
     * An adapter that adapts the legacy connection handler interface calls into
     * the new one.
     * 
     * SSL and java.nio channels are not integrated in Java 1.4 but are in Java
     * 5.0 -- in the future with Java 5.0, this adapter can be removed or
     * possibly reworked to better combine with this SSL and java.nio channel
     * integration
     * 
     * @see SocketChannel
     * 
     * @author Scott E. Schneider
     */
    static class Adapter implements org.eclipse.hyades.execution.security.IConnectionHandler {

        {
            this.handler = new FileConnectionHandlerExtended();
        }

        /**
         * The handler implementation, the connection accepted below gets
         * forwarded to the handler
         */
        private final FileConnectionHandlerExtended handler;

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.execution.security.IConnectionHandler#connectionAccepted(java.net.Socket)
         */
        public void connectionAccepted(Socket socket) {
            try {

                // Pass the accepted socket by first wrapping it with a channel
                this.handler.connectionAccepted(SocketChannelFactory.getInstance().create(socket));

            } catch (IOException e) {

                // Handle exception appropriately, channel could not be seeded
            }
        }

    }

    {
        this.workerGroup = new ThreadGroup("File Server Worker Thread Group");
    }

    private ThreadGroup workerGroup;

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.internal.execution.file.IConnectionHandler#connectionAccepted(
     *      org.eclipse.hyades.internal.execution.file.ISocketChannel)
     */
    public void connectionAccepted(ISocketChannel clientChannel) {

        // Connection is accepted, create a new thread to handle client request
        Thread worker = new Thread(this.workerGroup, this.createFileClientHandler(clientChannel));
        worker.setDaemon(true);
        worker.setName(clientChannel.toString());
        worker.start();

    }

    /**
     * Create runnable client request handler, this will service the request and
     * open up to allow more connections on the file server port
     * 
     * @return the runnable that will service the request
     */
    private Runnable createFileClientHandler(ISocketChannel clientChannel) {
        return new FileClientHandlerImplExtended(clientChannel);
    }

}