/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: SetConfig.java,v 1.31 2005/05/04 19:03:11 samwai Exp $

Contributors:
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.internal.config.generator;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Hashtable;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class SetConfig extends SetConfigSkeleton {
	private static boolean help = false;
	private static boolean examples = false;

	public static void main(String[] args) {
		SetConfig sc = new SetConfig();

		if(TestJvm.isMinimumVersion(ConfigUtility.getString("Config.Java.Version"), System.getProperty("java.version"))) {
			int size = 100;
			Hashtable hash = new Hashtable(size);;
			sc.populateHashtable(hash, args); // add all command line input to the hash table
			sc.processParameters(hash);
			if(help) {
				sc.printHelp();
				sc.printPluginHelp(hash);
				return;
			}
			else if(examples) {
				sc.printExamples();
				sc.printPluginExamples(hash);
				return;
			}
			sc.setFileName(HashUtility.getValue(hash, "RASERVER_HOME") + sr + "config" + sr + "serviceconfig.xml");
			sc.run(hash, silent, overwrite);
			sc.generatePluginConfigurations(hash);
		}
		else {
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_Version.Err") + ": " + ConfigUtility.getString("Config.Java.Version"));
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_Version.Msg"));
		}
	}

	public void init(Hashtable hash) {
		String home = HashUtility.getValue(hash, "RASERVER_HOME");
		setFileName(home + sr + "config" + sr + "serviceconfig.xml");
	}

	private void processParameters(Hashtable hash) {
		//
		// Try to use the $PWD/.. directory as the RASERVER_HOME if none
		// specified
		//
		if (HashUtility.getValue(hash, "RASERVER_HOME") == null) {
			File pwd = new File("..");
			try {
				HashUtility.setValue(hash, "RASERVER_HOME", pwd.getCanonicalPath());
			} catch (IOException e) {
				// cannot resolve ".."
			}
		}

		if (HashUtility.exist(hash, "-help") || HashUtility.exist(hash, "-h") || HashUtility.exist(hash, "-?")) {
			help = true;
		}
		if (HashUtility.exist(hash, "-example") || HashUtility.exist(hash, "-ex")) {
			examples = true;
		}
		if (HashUtility.exist(hash, "-debug")) {
			Logger.setDebug(true);
		}
		if (HashUtility.exist(hash, "-overwrite") || HashUtility.exist(hash, "-ovr")) {
			overwrite = true;
		}
		if (HashUtility.exist(hash, "-silent") || HashUtility.exist(hash, "-s")) {
			silent = true;
		}
		if (HashUtility.exist(hash, "-noverify") || HashUtility.exist(hash, "-nov")) {
			verify = false;
		}
	}

	public void askUser() {
		String[] hostList = null;
		final String jvm; //86262
		//
		// Get the required parameters
		//
		if (!isValidRaserverHome(configFile.getValue("RASERVER_HOME"))) {
			if(silent) {
				return; // cannot resolve RASERVER_HOME under silent mode, return
			}
			else {
				// Bug 87150
				String home;
				home = promptRaserverHome(configFile.getValue("RASERVER_HOME"));
				configFile.setValue("RASERVER_HOME", home);
				configFile.setFileName(home + sr + "config" + sr + "serviceconfig.xml");
			}
		}

		//
		// JAVA_PATH=
		//
		if(PlatformObject.name.equals("OS/400")) {
			configFile.setValue("JAVA_PATH", "/QSYS.LIB/Q5BVAJVM.PGM");
		}
		else {
			if(silent) {
				if ((configFile.getValue("JAVA_PATH") == null) || (configFile.getValue("JAVA_PATH").equals(""))){
					if(isValidJava(System.getProperty("java.home") + sr + "bin" + sr + "java" + PlatformObject.exeExt)) {
						configFile.setValue("JAVA_PATH", System.getProperty("java.home") + sr + "bin" + sr + "java" + PlatformObject.exeExt);
					}
					else {
						configFile.setValue("JAVA_PATH", "%JAVA_PATH%");
					}
				}
			}
			else {
				if (configFile.getValue("JAVA_PATH") == null) {
					configFile.setValue("JAVA_PATH", promptJavaPath(System.getProperty("java.home") + sr + "bin" + sr + "java" + PlatformObject.exeExt));
				}
				else {
					configFile.setValue("JAVA_PATH", promptJavaPath(configFile.getValue("JAVA_PATH")));
				}
			}
		}
		jvm = resolveJvmDll(configFile.getValue("JAVA_PATH"));//86262
		configFile.setValue("jvm", jvm);//86262
		//
		// ALLOW=
		//
		if(silent) {
			if (configFile.getValue("ALLOW") == null) {
				configFile.setValue("ALLOW", "LOCAL");
			}
		}
		else {
			if (configFile.getValue("ALLOW") == null) {
				configFile.setValue("ALLOW", promptAllow("LOCAL"));
			}
			else {
				configFile.setValue("ALLOW", promptAllow(configFile.getValue("ALLOW")));
			}
		}
		//
		// HOSTS=
		//
		if(configFile.getValue("ALLOW").toUpperCase().equals("CUSTOM")) {
			if(silent) {
				if (configFile.getValue("HOSTS") == null) {
					configFile.setValue("HOSTS", "localhost");
				}
			}
			else {
				if (configFile.getValue("HOSTS") == null) {
					configFile.setValue("HOSTS", promptHosts("localhost"));
				}
				else {
					configFile.setValue("HOSTS", promptHosts(configFile.getValue("HOSTS")));
				}
			}
		}
	}

	//
	// Retrieve the raserver home specified by the user
	//
	private String promptRaserverHome(String defaultValue) {
		String rc = null;
		System.out.println(ConfigUtility.getString("Config.Prompt.RASERVER_HOME.Tag"));
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
		System.out.print(" (");
		System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidRaserverHome(rc));
		return rc;
	}

	//
	// Retrieve the java home specified by the user
	//
	private String promptJavaPath(String defaultValue) {
		String rc = null;
		if(PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_PATH.Tag.Windows"));
		}
		else {
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_PATH.Tag.Unix"));
		}
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
		System.out.print(" (");
		System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidJava(rc));
		return rc;
	}

	//
	// Retrieve the allowed hosts specified by the user
	//
	private String promptAllow(String defaultValue) {
		String rc = null;
		System.out.println(ConfigUtility.getString("Config.Prompt.Allow.Tag"));
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
		System.out.print(" (");
		System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidAllow(rc));
		return rc;
	}

	//
	// Retrieve the host names specified by the user
	//
	private String promptHosts(String defaultValue) {
		System.out.println(ConfigUtility.getString("Config.Prompt.Hosts.Tag"));
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.Default") + "\"" + defaultValue + "\"");
		System.out.print(" (");
		System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
		return ConfigUtility.askUser(defaultValue);
	}

	//
	// Check if RASREVER_HOME is valid
	//
	private boolean isValidRaserverHome(String path) {
		if(!verify) {
			return true;
		}
		else {
			boolean rc = false;
			if (path == null) {
				System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
				return false;
			} else {
				File p = new File(path);
				// Make sure specified path is a directory and exist
				if (p.exists() && p.isDirectory()) {
					if(PlatformObject.name.equals("OS/400")) {
						// Check if SetConfig.sh exists
						File f = new File(p.getAbsolutePath() + File.separator + "bin" + File.separator + "SetConfig.sh");
						if (f.exists()) {
							rc = true;
						}
					}
					else {
						// Check if RAServer.exe or RAServer exists
						File f = new File(p.getAbsolutePath() + File.separator + "bin" + File.separator + "RAServer" + PlatformObject.exeExt);
						if (f.exists()) {
							rc = true;
						}
					}
				}
				if (!rc) {
					System.out.println(ConfigUtility.getString("Config.Prompt.InvalidValue"));
				}
				return rc;
			}
		}
	}

	//
	// Check if JAVA_PATH is valid
	//
	private boolean isValidJava(String path) {
		if(!verify) {
			return true;
		}
		else {
			if (path == null) {
				System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_PATH.Err"));
				return false;
			}
			else {
				File p = new File(path);
				// Make sure specified path is a directory and exist
				if (p.exists() && p.isFile()) {
					boolean rc = TestJvm.isSupportedJava(path, ConfigUtility.getString("Config.Java.Version"));
					if (!rc) {
						System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_Version.Err") + ": " + ConfigUtility.getString("Config.Java.Version"));
					}
					return rc;
				}
				else {
					System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_PATH.Err"));
					return false;
				}
			}
		}
	}

	//
	// Check if ALLOW is valid
	//
	private boolean isValidAllow(String allow) {
		if(!verify) {
			return true;
		}
		else {
			boolean rc = false;
			if(allow == null) {
				System.out.println(ConfigUtility.getString("Config.Prompt.Allow.Err"));
				return false;
			}
			else {
				if (allow.toLowerCase().equals("all") || allow.toLowerCase().equals("local") || allow.toLowerCase().equals("custom")) {
					rc = true;
				}
				if (!rc) {
					System.out.println(ConfigUtility.getString("Config.Prompt.Allow.Err"));
				}
				return rc;
			}
		}
	}

	//
	// Check if HOSTS is valid
	//
	private boolean isValidHosts(String hosts) {
		if(!verify) {
			return true;
		}
		else {
			return (hosts != null);
		}
	}

	//
	// Bug 61264
	// Check if path is writable
	//
	private boolean isWritable(String path) {
		File f = new File(path);
		if(f.exists()) {
			return f.canWrite();
		}
		else {
			return f.getParentFile().canWrite();
		}
	}


	//
	// Populate the hash table using the command line arguments
	//
	private void populateHashtable(Hashtable hash, String[] args) {
		if (args != null) {
			for (int i = 0; i < args.length; i++) {
				int eq = args[i].indexOf('=');
				if (eq == -1) { // a flags, e.g. -help
					HashUtility.setValue(hash, args[i], "");
				} else { // a parameter, e.g. name=value
					String name = trim(args[i].substring(0, eq));
					String value = trim(args[i].substring(eq + 1));
					if((name != null) && (value != null)) {
						HashUtility.setValue(hash, name, value);
					}
				}
			}
		}
	}

	//
	// Print the help menu
	//
	public void printHelp() {
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility.getString("Config.Cmd.Usage.Windows"));
		} else {
			System.out.println(ConfigUtility.getString("Config.Cmd.Usage.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Cmd.Flags"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Help.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Help.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Silent.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Silent.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.NoVerify.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.NoVerify.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Examples.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.Examples.Msg"));
		System.out.println();
		System.out.println(ConfigUtility.getString("Config.Cmd.Params"));
		System.out.println(ConfigUtility.getString("Config.Cmd.RASERVER_HOME.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.RASERVER_HOME.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.JAVA_PATH.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.JAVA_PATH.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.ALLOW.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.ALLOW.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.HOSTS.Tag"));
		System.out.println("\t" + ConfigUtility.getString("Config.Cmd.HOSTS.Msg"));
	}

	//
	// Print the help menu
	//
	public void printExamples() {
		System.out.println(ConfigUtility.getString("Config.Example.Tag"));
		System.out.println(ConfigUtility.getString("Config.Example.00.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.00.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.00.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.01.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.01.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.01.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.02.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.02.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.02.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.03.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.03.Msg.Windows"));
		} else {
			System.out.println("\t" + ConfigUtility.getString("Config.Example.03.Msg.Unix"));
		}
	}

	//
	// Generate the main config
	//
	public void generateConfiguration() {
		Document doc;
		Element acConfig;
		Element holder;

		Element acEnv;
		Element app;
		Element var;
		Element param;
		Element hosts;
		Element allow;
		Element plugin;
		String sr = PlatformObject.sr;

		//
		// Standard initializer 
		//
		doc = configFile.getDoc();
		if(doc == null) {
			return;
		}

		acConfig = doc.createElement(AgentControllerConfig.TAG);
		holder = configFile.getHolder();
		holder.appendChild(acConfig); // Add to the root holder

		//
		// There can be only one AgentControllerConfiguration element
		//
		AgentControllerConfig.setActiveConfiguration(acConfig, "default");
		AgentControllerConfig.setLoggingDetail(acConfig, "LOW");
		AgentControllerConfig.setLoggingLevel(acConfig, "INFORMATION");
		AgentControllerConfig.setPort(acConfig, "10002");
		AgentControllerConfig.setSecuredPort(acConfig, "10003");
		AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));
		AgentControllerConfig.setFilePort(acConfig, "10005"); // Bug 60082
		AgentControllerConfig.setProcessPolling(acConfig, "true"); // Bug 59316
		AgentControllerConfig.setDataMultiplexed(acConfig, "false"); // Bug 88179

		//
		// Set the version to current
		//
		AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));

		//
		// Set the Jvm location
		//
		AgentControllerConfig.setJvm(acConfig, configFile.getValue("jvm"));//86262

		//
		// Agent Controller Environment
		//
		acEnv = doc.createElement(AgentControllerEnvironment.TAG);
		AgentControllerEnvironment.setConfiguration(acEnv, "default");
		acConfig.appendChild(acEnv);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "JAVA_PATH");
		Variable.setValue(var, configFile.getValue("JAVA_PATH"));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "RASERVER_HOME");
		Variable.setValue(var, configFile.getValue("RASERVER_HOME"));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, PlatformObject.libEnv);
		Variable.setValue(var, "%RASERVER_HOME%" + sr + PlatformObject.libPath);
		Variable.setPosition(var, "prepend");
		acEnv.appendChild(var);

		/* Bug 69061 begins */
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "SYS_TEMP_DIR");
		Variable.setValue(var, PlatformObject.tempDir);
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "LOCAL_AGENT_TEMP_DIR"); // Bug 59544
		Variable.setValue(var, "%SYS_TEMP_DIR%");
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);
		/* Bug 69061 ends */

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "commons-logging.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hcframe.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexcore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexl.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hexr.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hl14.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcommons.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlevents.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		/* Bug 58037 begins */
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "hlcbe101.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "ecore.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "lib" + sr + "common.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);
		/* Bug 58037 ends */
		
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH");
		Variable.setValue(var, "%RASERVER_HOME%" + sr + "plugins" + sr + "org.eclipse.hyades.test" + sr + "lib" + sr + "runtime.jar");
		Variable.setPosition(var, "append");
		acEnv.appendChild(var);

		//
		// Application: java.exe
		//
		app = doc.createElement(Application.TAG);
		Application.setConfiguration(app, "default");
		Application.setExecutable(app, "java.exe");
		Application.setPath(app, "%JAVA_PATH%");
		Application.setLocation(app, "%SYS_TEMP_DIR%");
		acConfig.appendChild(app);

		//
		// Hosts
		//
		hosts = doc.createElement(Hosts.TAG);
		Hosts.setConfiguration(hosts, "default");
		acConfig.appendChild(hosts);

		if(configFile.getValue("ALLOW") != null) {
			if(configFile.getValue("ALLOW").toLowerCase().equals("custom")) {
				String[] allowedHostList = ConfigUtility.stringToArray(configFile.getValue("HOSTS"));
				if(allowedHostList.length > 0) {
					for(int i = 0; i < allowedHostList.length; i++) {
						allow = doc.createElement(Allow.TAG);
						hosts.appendChild(allow);
						if(allowedHostList[i].equalsIgnoreCase("all") || allowedHostList[i].equalsIgnoreCase("local")) {
							Allow.setHost(allow, allowedHostList[i].toUpperCase());
						}
						else {
							Allow.setHost(allow, allowedHostList[i]);
						}
					}
				}
				else {
					allow = doc.createElement(Allow.TAG);
					Allow.setHost(allow, "LOCAL");
					hosts.appendChild(allow);
				}
			}
			else {
				allow = doc.createElement(Allow.TAG);
				Allow.setHost(allow, configFile.getValue("ALLOW").toUpperCase());
				hosts.appendChild(allow);
			}
		}
		else {
			allow = doc.createElement(Allow.TAG);
			Allow.setHost(allow, "LOCAL");
			hosts.appendChild(allow);
		}

		//
		// Plugins
		//
		plugin = doc.createElement(Plugin.TAG);
		Plugin.setPath(plugin, "%RASERVER_HOME%" + sr + "plugins");
		acConfig.appendChild(plugin);

		configFile.saveToFile();
	}

	//
	// Generate the plugin config files
	//
	private void generatePluginConfigurations(Hashtable hash) {
		String home = configFile.getValue("RASERVER_HOME");
		if(home != null) {
			String pluginDirStr = home + sr + "plugins";
			File pluginDir = new File(pluginDirStr);
			if(pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				for(int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName();
					String jarName = pluginDirStr.replace('\\', '/') + "/" + pluginName + "/lib/config.jar";
					String clsName = pluginName + ".SetConfig";
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						URLClassLoader loader = URLClassLoader.newInstance(urls);
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Init the generator
						Method mInit= cls.getMethod("init", new Class[] { hash.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader", new Class[] { Class.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Invoke the run() method
						Method mRun = cls.getMethod("run", new Class[] { hash.getClass(), boolean.class, boolean.class });
						Logger.out("About to call run()");
						mRun.invoke(obj, new Object[] {hash, new Boolean(silent), new Boolean(overwrite)});
						Logger.out("Returned from run()");
					} catch (MalformedURLException e0) {
						Logger.err(ConfigUtility.getString("Config.Extension.JarNotFound") + " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.err(ConfigUtility.getString("Config.Extension.ClassNotFound") + " " + clsName);
						Logger.err(ConfigUtility.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.err(ConfigUtility.getString("Config.Extension.MethodNotFound") + " run()");
					} catch (InvocationTargetException e3) {
						Logger.err(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (IllegalAccessException e4) {
						Logger.err(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (InstantiationException e5) {
						Logger.err(ConfigUtility.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			}
			else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		}
		else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	//
	// Print the help from each plugin
	//
	private void printPluginHelp(Hashtable hash) {
		String home = HashUtility.getValue(hash, "RASERVER_HOME");
		if(home != null) {
			String pluginDirStr = home + sr + "plugins";
			File pluginDir = new File(pluginDirStr);
			if(pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				for(int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName();
					String jarName = pluginDirStr.replace('\\', '/') + "/" + pluginName + "/lib/config.jar";
					String clsName = pluginName + ".SetConfig";
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						URLClassLoader loader = new URLClassLoader(urls, ClassLoader.getSystemClassLoader());
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Init the generator
						Method mInit= cls.getMethod("init", new Class[] { hash.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader", new Class[] { Class.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Call the printHelp() method
						Method mHelp= cls.getMethod("printHelp", null);
						Logger.out("About to call printHelp()");
						mHelp.invoke(obj, null);
						Logger.out("Returned from printHelp()");
					} catch (MalformedURLException e0) {
						Logger.out(ConfigUtility.getString("Config.Extension.JarNotFound") + " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.out(ConfigUtility.getString("Config.Extension.ClassNotFound") + " " + clsName);
						Logger.out(ConfigUtility.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodNotFound") + " run()");
					} catch (InvocationTargetException e3) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (IllegalAccessException e4) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (InstantiationException e5) {
						Logger.out(ConfigUtility.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			}
			else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		}
		else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	//
	// Print the help from each plugin
	//
	private void printPluginExamples(Hashtable hash) {
		String home = HashUtility.getValue(hash, "RASERVER_HOME");
		if(home != null) {
			String pluginDirStr = home + sr + "plugins";
			File pluginDir = new File(pluginDirStr);
			if(pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				for(int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName();
					String jarName = pluginDirStr.replace('\\', '/') + "/" + pluginName + "/lib/config.jar";
					String clsName = pluginName + ".SetConfig";
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						URLClassLoader loader = new URLClassLoader(urls, ClassLoader.getSystemClassLoader());
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Init the generator
						Method mInit= cls.getMethod("init", new Class[] { hash.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader", new Class[] { Class.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Call the printExamples() method
						Method mExamples= cls.getMethod("printExamples", null);
						Logger.out("About to call printExamples()");
						mExamples.invoke(obj, null);
						Logger.out("Returned from printExamples()");
					} catch (MalformedURLException e0) {
						Logger.out(ConfigUtility.getString("Config.Extension.JarNotFound") + " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.out(ConfigUtility.getString("Config.Extension.ClassNotFound") + " " + clsName);
						Logger.out(ConfigUtility.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodNotFound") + " run()");
					} catch (InvocationTargetException e3) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (IllegalAccessException e4) {
						Logger.out(ConfigUtility.getString("Config.Extension.MethodInvocationError") + " run()");
					} catch (InstantiationException e5) {
						Logger.out(ConfigUtility.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			}
			else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		}
		else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	private String resolveJvmDll(String string) {
		if(PlatformObject.name.equals("OS/400")) {
			return new String("QSYS/QJVAJNI"); // Bug 58152
		}
		else {
			int index;

			index = string.lastIndexOf(sr + "java" + PlatformObject.exeExt);
			if(index == -1) {
				index = string.lastIndexOf(sr + "javaw" + PlatformObject.exeExt);
				if(index == -1) {
					return new String("%JAVA_PATH%");
				}
			}

			String str = string.substring(0, index);
			String rc = null;

			try {
				rc = ConfigUtility.getJvmLib(str);
			}
			catch(Exception e) {
				Logger.out(ConfigUtility.getString("Config.Jvm.Warning.CannotFindJvmLibrary") + ": " + PlatformObject.jvmLib);
			}

			if(rc == null) {
				return new String("%JAVA_PATH%");
			}
			else {
				return rc;
			}
		}
	}

}
