/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: ThreadDumpProcessorImpl.java,v 1.7 2005/03/02 21:12:31 paules Exp $

Contributors:
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.logging.jvm.threadanalysis;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.hyades.collection.threadanalyzer.DumpData;
import org.eclipse.hyades.collection.threadanalyzer.Monitor;
import org.eclipse.hyades.collection.threadanalyzer.StkEntry;
import org.eclipse.hyades.collection.threadanalyzer.StkEntryNative;
import org.eclipse.hyades.collection.threadanalyzer.Thd;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;


/**
 * @author duncan
 *
 * To change the template for this generated type comment go to
 * Window - Preferences - Java - Code Generation - Code and Comments
 */
public class ThreadDumpProcessorImpl implements ThreadDumpProcessor {

	private	EventFactory                _simpleEventFactory;
	private	ComponentIdentification	    _compId;
	private Situation					_situation;
	private AssociationEngine 			_associationEngine; 


	/* (non-Javadoc)
	 * @see com.ibm.ws.performance.threadanalyzer.getthreaddump.ThreadDumpProcessor#generateCBE(com.ibm.ws.performance.threadanalyzer.DumpData)
	 */
	public Vector generateCBEs(DumpData dd) {

		Vector             			cbes = new Vector();
		ExtendedDataElement 		xdeThdDump, xdeThreads, xdeMonitors;
		CommonBaseEvent 			cbeThdDump, cbeThreads, cbeMonitors;
        AssociatedEvent 			associatedThreadsEvent 	= _simpleEventFactory.createAssociatedEvent();

        _compId.setComponent(dd.getDumpingJvmName());
        
        cbeThdDump 	= createCBE("ThreadDump", null		, cbes);
		cbeThdDump.addExtendedDataElement("Type","Java");
		cbeThdDump.addExtendedDataElement("Description",dd.getDescription());
		cbeThdDump.addExtendedDataElement("DisplayName",dd.getDisplayName());
 
        cbeThreads = createCBE("Threads"	, cbeThdDump, cbes);
		xdeThreads = cbeThdDump.addExtendedDataElementWithIntValue("Threads",dd.getThreads().size());
		addThreads(dd, xdeThreads, cbeThreads, cbes);

		cbeMonitors	= createCBE("Monitors"	, cbeThdDump, cbes);
		xdeMonitors = cbeThdDump.addExtendedDataElementWithIntValue("Monitors",dd.getMonitors().size());
		addMonitors(dd, xdeMonitors, cbeMonitors, cbes);

		return cbes;
	}

	/**
	 * @param string
	 * @return
	 */
	private CommonBaseEvent createCBE(String type, CommonBaseEvent parentCbe, Vector cbes) {
		CommonBaseEvent childCbe = _simpleEventFactory.createCommonBaseEvent("CBE:"+type,(new Date()).getTime());
		
		childCbe.setSeverity((short) 10);	
		childCbe.setMsg(type);
		childCbe.setSourceComponentId(_compId);
		childCbe.setSituation(_situation);
		if (parentCbe !=null) {
			AssociatedEvent cbeAssociatedEvent = _simpleEventFactory.createAssociatedEvent();
			cbeAssociatedEvent.setAssociationEngineInfo(_associationEngine);
			cbeAssociatedEvent.addResolvedEvent(childCbe.getGlobalInstanceId());
			parentCbe.addAssociatedEvent(cbeAssociatedEvent);
		}
		cbes.add(childCbe);
		
		return childCbe; 
	}

	/**
	 * @param dd
	 * @param xdeThreads
	 * @param cbeThreads
	 */
	private void addThreads(DumpData dd, ExtendedDataElement xdeThreads, CommonBaseEvent cbeThreads, Vector cbes) {
		CommonBaseEvent		cbeCurrThread;
		ExtendedDataElement 	xdeCurrThread;
		Enumeration 			enumThreads;
		Thd						currThd;
		int i=0;

	 	cbeThreads.addExtendedDataElementWithIntValue("Thread Count",dd.getThreads().size());
	 	enumThreads = dd.getThreadEnum();
	 	
	 	while ((currThd = dd.getNextThread(enumThreads))!=null) {
//	 		createThreadXde(xdeThreads, currThd, i);		// imbed thread xdes in root cbe as well	
	 		createThreadCbe(cbeThreads, currThd, i, cbes);
		 	i++;
	 	}    
	}

	/**
	 * @param xdeCurrThread
	 * @param currThd
	 */
	private ExtendedDataElement createThreadXde(ExtendedDataElement xdeThreads, Thd currThd, int i) {
		ExtendedDataElement 	xdeStkEntries, xdeStkEntriesNative;

		ExtendedDataElement 	xdeCurrThread = xdeThreads.addChildWithIntValue("Thread_"+i,i);
		
 		xdeCurrThread.addChild("threadId",currThd.getThreadId());
 		xdeCurrThread.addChild("name",currThd.getName());
 		xdeCurrThread.addChild("priority",currThd._priority);
 		xdeCurrThread.addChild("state",currThd._state);
 		xdeCurrThread.addChildWithIntValue("waitMonitorID",currThd.getWaitMonitorId());
 		if(currThd._waitMonitor != null) {
	 		xdeCurrThread.addChild("waitMonitorKey",currThd._waitMonitor.getName());
 		}
		xdeStkEntries = xdeCurrThread.addChildWithIntValue("StackEntries",currThd.getStkSize());		
		xdeStkEntriesNative = xdeCurrThread.addChildWithIntValue("StackEntriesNative",currThd.getStkNativeSize());
		addStackEntries(currThd, xdeStkEntries);
 		addStackEntriesNative(currThd, xdeStkEntriesNative);
		return xdeCurrThread;	 		
	}

	/**
	 * @param currThd
	 * @param i	 
	 *	 */
	private CommonBaseEvent createThreadCbe(CommonBaseEvent parentCbe, Thd currThd, int i, Vector cbes) {
		ExtendedDataElement 	xdeStkEntries, xdeStkEntriesNative;

		CommonBaseEvent 		cbeCurrThread = createCBE("Thread"+i, parentCbe, cbes);
	 	cbeCurrThread.addExtendedDataElementWithIntValue("Thread Index",i);
	 	 		
		cbeCurrThread.addExtendedDataElement("threadId",currThd.getThreadId());
 		cbeCurrThread.addExtendedDataElement("name",currThd.getName());
 		cbeCurrThread.addExtendedDataElement("priority",currThd._priority);
 		cbeCurrThread.addExtendedDataElement("state",currThd._state);
 		cbeCurrThread.addExtendedDataElementWithIntValue("waitMonitorID",currThd.getWaitMonitorId());
 		if(currThd._waitMonitor != null) {
	 		cbeCurrThread.addExtendedDataElement("waitMonitorKey",currThd._waitMonitor.getName());
 		}
 		xdeStkEntries = cbeCurrThread.addExtendedDataElementWithIntValue("StackEntries",currThd.getStkSize());
 		xdeStkEntriesNative = cbeCurrThread.addExtendedDataElementWithIntValue("StackEntriesNative",currThd.getStkNativeSize());
 		addStackEntries(currThd,xdeStkEntries);
 		addStackEntriesNative(currThd,xdeStkEntriesNative);
 		return cbeCurrThread;
	}

	/**
	 * @param currThd
	 * @param xdeCurrThread
	 */
	private void addStackEntries(Thd currThd, ExtendedDataElement xdeStkEntries) {
		ExtendedDataElement 	xdeCurrStkEntry;
		StkEntry				currStkEntry;

 		for (int j=0; j<currThd.getStkSize(); j++) {
 			currStkEntry=currThd.getStkEntry(j);
 			xdeCurrStkEntry=xdeStkEntries.addChildWithIntValue("StkEntry_"+j,j);
 			xdeCurrStkEntry.addChild("module",currStkEntry._module);
 			xdeCurrStkEntry.addChild("pkg",currStkEntry._pkg);
 			xdeCurrStkEntry.addChild("method",currStkEntry._method);
 			xdeCurrStkEntry.addChildWithIntValue("lineno",currStkEntry._lineno);
 			xdeCurrStkEntry.addChildWithBooleanValue("isNative",currStkEntry._isNative);
 		}
		return;
	}

	/**
	 * @param currThd
	 * @param xdeCurrThread
	 */
	private void addStackEntriesNative(Thd currThd, ExtendedDataElement xdeStkEntriesNative) {
		ExtendedDataElement 	xdeCurrStkEntryNative;
		StkEntryNative			currStkEntryNative;

 		for (int j=0; j<currThd.getStkNativeSize(); j++) {
 			currStkEntryNative=currThd.getStkEntryNative(j);
 			xdeCurrStkEntryNative=xdeStkEntriesNative.addChildWithIntValue("StkEntryNative_"+j,j);
 			xdeCurrStkEntryNative.addChild("function",currStkEntryNative._function);
 			xdeCurrStkEntryNative.addChild("hexAddress",currStkEntryNative._hexAddress);
 		}
		return;
	}
	
	/**
	 * @param dd
	 * @param xdeMonitors
	 * @param cbeMonitors
	 */
	private void addMonitors(DumpData dd, ExtendedDataElement xdeMonitors, CommonBaseEvent cbeMonitors, Vector cbes) {

		Enumeration 			enumMonitors;
		Monitor					currMon;
		int i=0;
			
		cbeMonitors.addExtendedDataElementWithIntValue("Monitor Count",dd.getMonitors().size());
	 	enumMonitors = dd.getMonitorEnum();
	 	while ((currMon = dd.getNextMonitor(enumMonitors))!=null) {
//	 		createMonitorXde(xdeMonitors, currMon, i);   	// imbed monitor xdes in root cbe as well
  	 		createMonitorCbe(cbeMonitors, currMon, i, cbes);   
	 		i++;
	 	}    
	}
	/**
	 * @param xdeMonitors
	 * @param currMon
	 * @param i
	 */
	private void createMonitorXde(ExtendedDataElement xdeMonitors, Monitor currMon, int i) {
		ExtendedDataElement xdeCurrMonitor = xdeMonitors.addChildWithIntValue("Monitor"+i,i);
		ExtendedDataElement xdeWaiters;
		int					 waitersSize	= currMon._waiters.size();
	 		
	 	xdeCurrMonitor.addChild("name",currMon.getName());
	 	if (currMon._owner!=null) {
	 		xdeCurrMonitor.addChild("ownerThreadId",currMon._owner._threadId);
	 		xdeCurrMonitor.addChild("ownerThreadName",currMon._owner.getName());
	 	}	
	 	xdeCurrMonitor.addChildWithIntValue("monitor_type",currMon._monitor_type);
	 	
		if (waitersSize>0) {
			xdeWaiters = xdeCurrMonitor.addChildWithIntValue("Waiters",waitersSize);
			addWaiters(currMon,xdeWaiters);
		}	
	}
	/**
	 * @param cbeMonitors
	 * @param currMon
	 * @param i
	 * @param cbes
	 */
	private void createMonitorCbe(CommonBaseEvent parentCbe, Monitor currMon, int i, Vector cbes) {
		CommonBaseEvent 		cbeCurrMonitor = createCBE("Monitor"+i, parentCbe, cbes);
		ExtendedDataElement 	xdeWaiters;
		int						waitersSize	= currMon._waiters.size();
	 	cbeCurrMonitor.addExtendedDataElementWithIntValue("Monitor Index",i);
		cbeCurrMonitor.addExtendedDataElement("name",currMon.getName());
	 	if (currMon._owner!=null) {
	 		cbeCurrMonitor.addExtendedDataElement("ownerThreadId",currMon._owner._threadId);
	 		cbeCurrMonitor.addExtendedDataElement("ownerThreadName",currMon._owner.getName());
	 	}	
	 	cbeCurrMonitor.addExtendedDataElementWithIntValue("monitor_type",currMon._monitor_type);
		if (waitersSize>0) {
			xdeWaiters = cbeCurrMonitor.addExtendedDataElementWithIntValue("Waiters",waitersSize);
			addWaiters(currMon,xdeWaiters);
		}	
	}
	/**
	 * @param currMon
	 * @param xdeCurrMonitor
	 */
	private void addWaiters(Monitor currMon, ExtendedDataElement xdeWaiters) {
		ExtendedDataElement 	xdeCurrWaiter;
		Thd     				currWaiter;
		Vector					waiters=currMon._waiters;
		int						waitersSize=currMon._waiters.size();

	 	for (int j=0; j<waitersSize; j++) {
	 		currWaiter=(Thd)waiters.elementAt(j);
	 		xdeCurrWaiter=xdeWaiters.addChildWithIntValue("Waiter_"+j,j);
	 		xdeCurrWaiter.addChild("ThreadId",currWaiter.getThreadId());
	 		xdeCurrWaiter.addChild("ThreadName",currWaiter.getName());
 		}
		return;
	}

	/**
	 *
	 */
	public ThreadDumpProcessorImpl() {
		super();
		String hostname = null;
		_simpleEventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();
		try {
			hostname = InetAddress.getLocalHost().getHostName();
		} catch (UnknownHostException e) {
			hostname = "localhost"; // if we couldn't resolve the local hostname we just print "localhost"
		}
		try {
			ReportSituation repSituation;
		    repSituation=_simpleEventFactory.createReportSituation();
			repSituation.setReportCategory("Trace");
			
			_situation 	= _simpleEventFactory.createSituation();
	        _situation.setCategoryName("ReportSituation");
	        _situation.setSituationType(repSituation);
			
			_compId = _simpleEventFactory.createComponentIdentification();
			_compId.setComponentIdType("ProductName");
	        _compId.setComponent("Dummy");
	        _compId.setComponentType("JAVA JVM");
	        _compId.setSubComponent("Thread/Monitor dump");
	        _compId.setLocationType("Hostname");
	        _compId.setLocation(hostname);
			
			_associationEngine=_simpleEventFactory.createAssociationEngine();
			_associationEngine.setType("Root");
			_associationEngine.setName("Root");
					
		} catch (Exception e) {
			e.printStackTrace();
		}	
	}	
	
}