/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AdapterPlugin.java,v 1.8 2005/03/23 08:09:06 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * 
 * Change History:
 * Bugzilla  Description
 * 85211     Made getContextListeners synchronized to avoid lock contention
 *           exception by Eclipse class loader.
 **********************************************************************/
package org.eclipse.hyades.logging.adapter;

import java.util.ArrayList;
import java.util.List;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.hyades.logging.adapter.parsers.ISubstitutionExtension;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.osgi.framework.BundleContext;

/**
 * This class contains various operations that allows the Generic Adapter
 * to run within Eclipse.  This class is where all of the workbench interaction
 * should be placed.  Anyone who uses wither the class object or instances of
 * this class must code in such a mannor as to handle linkage/resolution problems
 * at runtime.  This is because the Eclipse runtime is not nessesary to run the
 * Generic Adapter.
 * 
 * @author rduggan
 * @author paules
 */
public class AdapterPlugin extends Plugin {
	// The shared instance.
    private static AdapterPlugin instance = null;

	//Resource bundle.
	private static ResourceBundle resourceBundle = null;
	
    public AdapterPlugin() {

        super();

        if (instance == null) {
            instance = this;
        }
		try {
			resourceBundle = Platform.getResourceBundle(Platform.getBundle("org.eclipse.hyades.logging.adapter"));
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
    }

    public static AdapterPlugin getPlugin() {
        return instance;
    }

    public static ResourceBundle getResourceBundle() {
    	if (resourceBundle == null) {
    		try {
    			resourceBundle = Platform.getResourceBundle(Platform.getBundle("org.eclipse.hyades.logging.adapter"));
    		} catch (MissingResourceException x) {
    			resourceBundle = null;
    		}
    	}	
        return resourceBundle;
    }

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}


    /**
     * Locate all of the IContextListeners that have registers via the extension point.
     * @return a java.util.List containing the IContextListener instances.
     */
    public static synchronized List getContextListeners() throws AdapterException {

        IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.logging.adapter", "contextListener");
        
        List results = new ArrayList();
        
        for (int i = 0; i < elements.length; i++) {
        
            IConfigurationElement elem = elements[i];

            try {
                IContextListener listener = (IContextListener) elem.createExecutableExtension("name");
                results.add(listener);
            }
            catch (Throwable e) {
            	throw new AdapterException(Messages.getString("HyadesGAInitialization_GetContextListeners_Failure_ERROR_"), e);
            }
        }

        return results;
    }

    public static synchronized IComponent getComponent(String classname) throws AdapterException {

        IComponent component = null;
        IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.logging.adapter", "adapterComponent");

        for (int i = 0; i < elements.length; i++) {

            IConfigurationElement elem = elements[i];

            if (elem.getAttribute("name").equals(classname)) {

                try {

                    component = (IComponent) elem.createExecutableExtension("name");

                    return component;
                }
                catch (Throwable e) {
                	throw new AdapterException(Messages.getString("HyadesGAInitialization_GetComponent_Failure_ERROR_", classname), e);
                }
            }
        }
        return component;
    }
    
    /** 
     * Get the substitution extension class instance for the specified extension point
     */
	public static synchronized ISubstitutionExtension getSubstitutionExtension(String classname) throws ClassCastException, AdapterException {

		ISubstitutionExtension extension = null;
		IConfigurationElement[] elements = null;
		
		try {
			elements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.logging.adapter", "substitutionExtension");
		}
		catch (Throwable e) {
			// Just return null if we're not running in eclipse
			return null;
		}
		
		for (int i = 0; i < elements.length; i++) {

			IConfigurationElement elem = elements[i];

			if (elem.getAttribute("name").equals(classname)) {

				try {

					extension = (ISubstitutionExtension) elem.createExecutableExtension("name");

					return extension;
					
				}
				catch (ClassCastException e) {
					// We want to know if the object is not of the correct type.
					throw e;
				}
				catch (Throwable e) {
					throw new AdapterException(Messages.getString("HyadesGAInitialization_GetSubstitutionExtension_Failure_ERROR_", classname), e);
				}
			}
		}
		return extension;
	}
}
