/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BasicContext.java,v 1.17 2005/04/28 20:22:27 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;
import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IProcessUnit;
import org.eclipse.hyades.logging.adapter.ISensor;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.Situation;
/**
 * BasicContext provide the simplest possible implementation of a Context.
 * It is a good example for other custom context implementation
 */
public class BasicContext extends Context
{
	
	
	/**
	 * 
	 */
	public BasicContext()
	{
		super();
	}
	public boolean init()
	{
		super.init();
		//test the configuration of this context
		return testRun();
	}
	private boolean testRun()
	{
		ISensor sensor = (ISensor) getComponents()[0];
		Object[] msgs = sensor.testGetNext();
		try
		{
			testProcessMessages(msgs, 1);
		}
		catch (AdapterInvalidConfig e)
		{
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    event.setMsg(e.toString());
		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
			
		    log(event);
			
		    return false;
		}
		catch (Exception e)
		{
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    event.setMsg(e.toString());
		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
			
		    log(event);

		    return false;
		}
		return true;
	}
	private void testProcessMessages(Object[] msgs, int level) throws AdapterInvalidConfig
	{
		IComponent contextComponents[] = getComponents();
		if (level < contextComponents.length)
		{
			Object[] oa = ((IProcessUnit) contextComponents[level]).testProcessEventItems(msgs);
			level++;
			if (oa != null && oa[0] != null)
				testProcessMessages(oa, level);
		}
	}
	public void run()
	{
		/* Log the fact we have started the context */
		CommonBaseEvent startEvent = getEventFactory().createCommonBaseEvent();

		startEvent.setMsg(Messages.getString("HyadesGAContext_Started_INFO_",getName()));
		startEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
		Situation startSituation = getEventFactory().createSituation();
		startSituation.setStartSituation("INTERNAL","START COMPLETED","SUCCESSFUL");
		startEvent.setSituation(startSituation);
	    log(startEvent);
	    status.setActive(true);

	    IComponent contextComponents[] = getComponents();
	    
		for (int i = 0; i < contextComponents.length; i++) {
			/* Log the fact we have started the component */
			startEvent = getEventFactory().createCommonBaseEvent();

			startEvent.setMsg(Messages.getString("HyadesGAComponent_Started_INFO_",contextComponents[i].getName()));
			startEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
			startSituation = getEventFactory().createSituation();
			startSituation.setStartSituation("INTERNAL","START COMPLETED","SUCCESSFUL");
			startEvent.setSituation(startSituation);
		    log(startEvent);
		    ((Component)contextComponents[i]).getComponentStatus().setActive(true);
		}
		
		try	{
			/* Get our sensor.  It should be the first component in the component chain */
			ISensor sensor = (ISensor) getComponents()[0];
			
			try	{	
				while (!isHardStop() && !stopping) {
					
					/* If we have a IContextListener tell it we are invoking the sensor for more
					 * information
					 */
					if(listener!=null) {
						try {
							listener.preProcessEventItems(sensor, null);
						}
						catch(Throwable e) {
							/* Don't let the listener break us */
						}	
					}
					
					/* Get our data from the sensor */
					Object[] msgs = sensor.getNext();
					
					/* If we have a IContextListener tell it we have just returned from the 
					 * sensor and give it the lines we have retrieved.
					 */
					if(listener!=null) {
						try {
							listener.postProcessEventItems(sensor, msgs);
						}
						catch(Throwable e) {
							/* Don't let the listener break us */
						}	
					}
					
					
					/* If we have recieved no data from the sensor then
					 * we need to start our shutdown sequence.  This can
					 * be reset if we get data on the next call.
					 */
					if (msgs == null) {
						handleControledStop();
					}
					else {
						resetControledStop();
						processMessages(msgs, 1);
					}
				}
			}
			catch (Exception e)
			{
			    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
			    event.setMsg(e.toString());
			    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
				
			    log(event);

			}
		}
		catch (Throwable e)
		{
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    event.setMsg(e.toString());
		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);
			
		    log(event);
		}
		
		// Set context and component active status to false
		// and log that they have stopped.
		
		CommonBaseEvent stopEvent;
		Situation stopSituation;
		for (int i = 0; i < contextComponents.length; i++) {
			/* Log the fact we have stopped the component */
			stopEvent = getEventFactory().createCommonBaseEvent();

			stopEvent.setMsg(Messages.getString("HyadesGAComponent_Stopped_INFO_",contextComponents[i].getName()));
			stopEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
			stopSituation = getEventFactory().createSituation();
			stopSituation.setStopSituation("INTERNAL","STOP COMPLETED","SUCCESSFUL");
			stopEvent.setSituation(stopSituation);
		    log(stopEvent);
			((Component)contextComponents[i]).getComponentStatus().setActive(false);
		}
		
		/* Log the fact we have stopped the context */
		stopEvent = getEventFactory().createCommonBaseEvent();

		stopEvent.setMsg(Messages.getString("HyadesGAContext_Stopped_INFO_", getName()));
		stopEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
		stopSituation = getEventFactory().createSituation();
		stopSituation.setStopSituation("INTERNAL","STOP COMPLETED","SUCCESSFUL");
		stopEvent.setSituation(stopSituation);
	    log(stopEvent);
	    
		status.setActive(false);
	}
	/*
	 * processMessages builds an Object array and passes it on to the next component based on the level
	 */
	private void processMessages(Object[] msgs, int level)
	{
		IComponent contextComponents[] = getComponents();
		if (level > 0)
			if (level < contextComponents.length)
			{
				/* If we have a IContextListener tell it we are invoking the component as
				 * well as the data we are giving to teh component.
				 */
				if(listener!=null) {
					try {
						listener.preProcessEventItems(contextComponents[level], msgs);
					}
					catch(Throwable e) {
						/* Don't let the listener break us */
					}	
				}
				
				/* Invoke the component */
				Object[] oa = ((IProcessUnit) contextComponents[level]).processEventItems(msgs);
				
				/* If we have a IContextListener tell it we have finished invoking the
				 * component as well as the data we recieved back.
				 */
				if(listener!=null) {
					try {
						listener.postProcessEventItems(contextComponents[level], oa);
					}
					catch(Throwable e) {
						/* Don't let the listener break us */
					}	
				}
				
				level++;
				if (oa != null) {
					processMessages(oa, level);
				}
			}
	}
	
	public void stop()
	{
		
		/* We are going to fluch the sensor.  We will fake up that we are invoking the sensor
		 * on a getNext() for our listener if we have one.
		 */
		 
		if(listener!=null && !isHardStop()) {
			try {
				listener.preProcessEventItems(getComponents()[0], null);
			}
			catch(Throwable e) {
				/* Don't let the listener break us */
			}	
		}
		 
		Object[] flushedSensorContent=((ISensor) getComponents()[0]).flush();
		
		if(listener!=null && !isHardStop()) {
			try {
				listener.postProcessEventItems(getComponents()[0], flushedSensorContent);
			}
			catch(Throwable e) {
				/* Don't let the listener break us */
			}	
		}
		
		flushMessages(flushedSensorContent, 1);
		super.stop();
	}
	/*
	 * processMessages builds an Object array and passes it on to the next component based on the level
	 */
	private void flushMessages(Object[] msgs, int level)
	{
		IComponent contextComponents[] = getComponents();
		if (level > 0)
			if (level < contextComponents.length)
			{
				/* If we have a IContextListener tell it we are invoking the component as
				 * well as the data we are giving to teh component.
				 */
				if(listener!=null &&!isHardStop()) {
					try {
						listener.preProcessEventItems(contextComponents[level], msgs);
					}
					catch(Throwable e) {
						/* Don't let the listener break us */
					}	
				}
				Object[] oa = ((IProcessUnit) contextComponents[level]).flushEventItems(msgs);
				
				/* If we have a IContextListener tell it we have finished invoking the
				 * component as well as the data we recieved back.
				 */
				if(listener!=null &&!isHardStop()) {
					try {
						listener.postProcessEventItems(contextComponents[level], oa);
					}
					catch(Throwable e) {
						/* Don't let the listener break us */
					}	
				}
				
				level++;
				
				// bugzilla 87164
				// Always flush the next component, even if no messages are passed
				// to it.  The component may have some messages cached.

				flushMessages(oa, level);
			}
	}

}
