/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Filter.java,v 1.2 2005/03/28 08:38:46 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.internal.filters;


import java.util.Iterator;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.impl.ProcessUnit;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.eclipse.hyades.logging.adapter.IFilter;
import org.eclipse.hyades.logging.adapter.util.Messages;

/**
 * @author rduggan
 *
 * This is an implementation of a filter component that consumes 
 * filter criteria as specified in filters.xsd
 */
public class Filter extends ProcessUnit implements IFilter {

	private IFilterElement rootFilter;
	
	// The Range Filter helper object
	private RangeFilterHelper rangeHelper = null;
	
	// Flag to indicate if CommonBaseEvents need to be cached
	private boolean cacheCBE = false;
	
	/**
	 * Overrides the update implementation provided by the ProcessUnit
	 * superclass.  Updates the configuration.
	 *
	 * THIS IMPLEMENTATION IS TIGHTLY COUPLED WITH THE SCHEMA.  If the schema
	 * changes for the parser this will need to be changed accordingly.
	 *
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig{
		
		/* Let our superclass update itself first */
		super.update();
		
		/* Get the configuration that the adapter assigned us.  We
		 * are actually handed the cc:Parser node.  We are interested
		 * in the Parser:Rules child andwe will only consider the first
		 * one.  There should be only one!
		 */
		Element element = getConfiguration();
		
		if(!element.getNodeName().equals(Messages.getString("HyadesGAFilterTagName"))) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_Preparation_No_Filter_Element_ERROR_",Messages.getString("HyadesGAFilterTagName")));
		}
			
		/* Extract our filter information from the config.*/
		NodeList filters=element.getChildNodes();
		
		/* Prepare the filters */ 
		prepareFilters(filters);
	}
	
	/**)
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#testProcessEventItems(java.lang.Object[])
	 */
	public Object[] testProcessEventItems(Object[] msgs)
		throws AdapterInvalidConfig {

		return msgs;
	}
	
	/**
	 * 
	 * @param filters
	 * @throws AdapterInvalidConfig
	 */
	protected void prepareFilters(NodeList filters) throws AdapterInvalidConfig {
		int filterCount = filters.getLength();
		if (filterCount >0) {
			for (int i = 0; i < filterCount; i++)	{
				Element filter=null;
				try {
					filter=(Element)filters.item(i);
				}
				catch(ClassCastException e) {
					/* We can ignore this child as it is not a Element */
					continue; 	 	
				}
			
				if(filter!=null) {
					/* Create the root filter - there should be only one */
					if(filter.getNodeName().endsWith(Messages.getString("HyadesGAFilterRuleBlockElementTagName"))) {
						rootFilter=new FilterBlockImpl();
					}
					else if(filter.getNodeName().endsWith(Messages.getString("HyadesGAFilterRuleElementTagName"))) {
						rootFilter=new FilterAtomImpl();
					}
					else {
						/* This could be a specialized filter rule.  In the future we can instantiate the proper type and hand it the element to work with.
						 * For now we will throw an exception because we do not know how to handle it.
						 */
						throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_Preparation_Invalid_Filter_Element_ERROR_",filter.getNodeName()));
					}
					rootFilter.prepareFilter(filter);
					
					// requiresCache() returns the range limit for determining which events get cached.
					int rangeLimit = ((FilterElementImpl)rootFilter).requiresCache();
					cacheCBE = rangeLimit > 0;
					
					// Set the range helper data
					getRangeHelper().setCacheCBE(cacheCBE);
					getRangeHelper().setRangeLimit(rangeLimit);
					
					/* The root filter has been created so we can break out of the loop */
					break;
				}	
			}
		}
	}
	
	/**
	 * This implementation processes CommonBaseEvent instances 
	 * @see org.eclipse.hyades.logging.adapter.IProcessUntit#processEventItems(java.lang.Object msgs)
	 */
	public Object[] processEventItems(Object[] cbes) {
		if (cbes == null || cbes.length == 0) {
			return null;
		}
		// Filtered Common Base Events
		CommonBaseEvent[] filteredCBEs = new CommonBaseEvent[cbes.length];
		
		/* Walk through the CommonBaseEvent list and set all the filtered items to null */
		for(int i=0; i<cbes.length; i++) {
			CommonBaseEvent event=(CommonBaseEvent)cbes[i];
			// Make sure there is a Common Base Event
			if (event == null) {
				filteredCBEs[i] = null;
			}
			else {
				try {
					/* Update the rootFilter with teh right hand operands from the CBE */
					updateFilterWithCBEContent(rootFilter, event);
					
					/* Run the filter */
					boolean filterResult = rootFilter.evaluateFilter();

					// If Common Base Events are to be cached Then cache it.
					if (isCacheCBE()) {
						cache(event, filterResult);
					}
					else if(filterResult) {
						// If the Common Base Event satisfies the filter then save it
						filteredCBEs[i] = event;
					}
					else {
						// If the Common Base Event does not satisfy the filter then
						// discard it.
						filteredCBEs[i] = null;
					}
				}
				catch (AdapterException e) {
					// There is something wrong with the filters so fail the filter test.
					filteredCBEs[i] = null;
					/* TODO should log a message here */
				}
			}
		}
		
		// Return the Common Base Events that satisfied the filter
		return filteredCBEs;
	}
	
	/**
	 * Updates a IFilterElement with the corresponding value in a specific CBE
	 * instance
	 * @param element
	 * @param cbe
	 */
	protected void updateFilterWithCBEContent(IFilterElement element, CommonBaseEvent cbe) throws AdapterException {
		if (element == null || cbe == null) {
			throw new AdapterException();
		}
		
		/* Load the filter datastructure with this CBE's data */
		try {
			Iterator iterator=((IFilterBlock)element).getFilterElements().iterator();
			while(iterator.hasNext()) {
				updateFilterWithCBEContent((IFilterElement)iterator.next(), cbe);
			}
		}
		catch(ClassCastException e) {
			IFilterAtom current =(IFilterAtom)element;
			String leftOperand = null;
			try {
				leftOperand = CBEHelper.getValueFromPath(cbe, current.getAttributePath());
			}
			catch (InvalidAttributeNameException ie) {
				try {
					leftOperand = getRangeHelper().getValueFromPath(cbe, current.getAttributePath());
				}
				catch (InvalidAttributeNameException ie2) {
					/* TODO  Handle this problem */
				}
			}
			current.setLeftOperand(leftOperand);
		}
	}
	
	/**
	 * flushEventItems should be overridden by any Component that buffers data
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#flushEventItems(java.lang.Object[])
	 */
	public Object[] flushEventItems(Object[] msgs)
	{
		flushingMode = true;
		Object[] temp = processEventItems(msgs);
		if (cacheCBE) {
			temp = getRangeHelper().getCbeCacheArray();
		}
		flushingMode = false;
		return temp;
	}
	
	/**
	 * @return Returns the rangeHelper.
	 */
	protected RangeFilterHelper getRangeHelper() {
		if (rangeHelper == null) {
			rangeHelper = new RangeFilterHelper();
		}
		return rangeHelper;
	}
	/**
	 * @return Returns the cacheCBE.
	 */
	protected boolean isCacheCBE() {
		return cacheCBE;
	}
	
	protected void cache(CommonBaseEvent evt, boolean filterResult) {
		getRangeHelper().cache(evt, filterResult);
	}
}
