/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FilterAtomImpl.java,v 1.2 2005/03/28 08:38:46 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.internal.filters;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


/**
 * @author rduggan
 */
public class FilterAtomImpl extends FilterElementImpl implements IFilterAtom {

	private String implementationClassname;
	private List attributePath;
	private String leftOperand;
	private String rightOperand;
	private IFilterType evaluator;

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterElement#prepareFilter(org.w3c.dom.Node)
	 */
	public void prepareFilter(Element node) throws AdapterInvalidConfig {
		super.prepareFilter(node);
		
		/* Retrieve the implementation class */
		implementationClassname=node.getAttribute(Messages.getString("HyadesGAImplementationClassAttributeName"));
		if(implementationClassname==null || implementationClassname.equals("")) {
			/* We have no implementation class */
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterRule_Invalid_No_Impl_Class_ERROR_"));
		}
		
		/* We need to locate the filter class based upon the classname
		 * this needs to work both in eclipse and outside of eclipse.  For
		 * now we can get away with Class.forName as the impls are in the
		 * same plugin
		 */
		try {
			Class filterClassObject=Class.forName(implementationClassname);
			evaluator=(IFilterType)filterClassObject.newInstance();
		}
		catch(ClassNotFoundException e) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterType_Class_Not_Found_ERROR_",implementationClassname));
		}
		catch(InstantiationException e) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterType_Class_Instantiation_ERROR_",implementationClassname,e.getMessage()));
		}
		catch(IllegalAccessException e) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterType_Class_Instantiation_ERROR_",implementationClassname,e.getMessage()));
		}	
		
		/* Retrieve the value we are going to filter against (The right hand side operand)*/
		rightOperand=node.getAttribute(Messages.getString("HyadesGAAttributeValueAttributeName"));
		if(rightOperand==null) {
			/* We have no filter specified */
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterRule_Invalid_No_Value_ERROR_"));
		}
		
		/* Iterate through the child 'nodes' to determine what our filtering path is. */
		NodeList children=node.getChildNodes();
		
		for(int i=0; i<children.getLength(); i++) {
			Element child=null;
			try {
				child=(Element)children.item(i);
			}
			catch(ClassCastException e) {
				/* We can ignore this child as it is not a Element */	
				continue; 	
			}
			if(child.getNodeName().endsWith(Messages.getString("HyadesGAttributeNamePathElementTagName"))) {
				/* We have a node element.  Lets pull out its CDATA */
				NodeList nodes=child.getChildNodes();
				for(int j=0; j<nodes.getLength(); j++) {
					Element nodeValue=null;
					try {
						nodeValue=(Element)nodes.item(j);
					}
					catch(ClassCastException e) {
						/* We can ignore this child as it is not a Element */	
						continue; 	
					}
					if(nodeValue.getNodeName().endsWith(Messages.getString("HyadesGANodeElementTagName"))) {
						/* We have a node element.  Lets pull out its CDATA */
						NodeList values=nodeValue.getChildNodes();
						for(int k=0; k<values.getLength(); k++) {
							CharacterData value=null;
							try {
								value=(CharacterData)values.item(k);
							}
							catch(ClassCastException e) {
								/* We can ignore this child as it is not a Element */	
								continue; 	
							}
							if(attributePath==null) {
								attributePath=new ArrayList();
								attributePath.add(value.getData());
							}
							else {
								attributePath.add(value.getData());
							}
						}
					}
				}	
			}
			else if(child.getNodeName().endsWith(Messages.getString("HyadesGAttributeNameElementTagName"))) {
				/* We have a node element.  Lets pull out its CDATA */
				NodeList values=child.getChildNodes();
				for(int j=0; j<values.getLength(); j++) {
					CharacterData value=null;
					try {
						value=(CharacterData)values.item(j);
					}
					catch(ClassCastException e) {
						/* We can ignore this child as it is not a Element */	
						continue; 	
					}
					if(attributePath==null) {
						attributePath=new ArrayList(1);
						attributePath.add(value.getData());
					}
				}	
			}
		}
		
		/* Validate the operator */
		String ops[] = evaluator.getOperators();
		int in;
		for (in=0; in<ops.length; in++) {
			if (ops[in].equals(getOperator())) {
				break;
			}
		}
		
		if (in == ops.length) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterRule_Invalid_Operator_ERROR_", (String)attributePath.get(attributePath.size()-1), getOperator()));
		}
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#getLeftOperand()
	 */
	public String getLeftOperand() {
		return leftOperand;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#getRightOperand()
	 */
	public String getRightOperand() {
		return rightOperand;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#setLeftOperand(java.lang.String)
	 */
	public void setLeftOperand(String operand) {
		leftOperand=operand;
		
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#setRightOperand(java.lang.String)
	 */
	public void setRightOperand(String operand) {
		rightOperand=operand;	
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#setAttributePath()
	 */
	public void setAttributePath(List attributePath) {
		this.attributePath=attributePath;	
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterAtom#getAttributePath()
	 */
	public List getAttributePath() {
		return attributePath;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterElement#evaluateFilter()
	 */
	public boolean evaluateFilter() {
		return evaluator.processRule(this);
	}

	/**
	 * @return Returns the evaluator.
	 */
	public IFilterType getEvaluator() {
		return evaluator;
	}
	
	/** 
	 * Indicates whether filtered items need to be cached.
	 * @return int size of cache
	 */
	public int requiresCache() {
		// If there is an evaluator and it is a NumericalRangeFilterType object and the rule is a last rule
		// Then return the range limit
		if (evaluator != null && evaluator instanceof NumericalRangeFilterType && getOperator().equals(NumericalRangeFilterType.LAST)) {
			return Integer.parseInt(rightOperand);
		}

		return 0;
	}
}
