/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FilterBlockImpl.java,v 1.2 2005/03/28 08:38:46 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.internal.filters;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This is a filter block that knows how to process the following logical operators.
 * Filters are evaluated left to right, depth first.
 * 
 * AND - all filter atoms and blocks must evaluate to true
 * OR - any filter atom or filter block that is a direct child must evaluate to true.
 */
public class FilterBlockImpl extends FilterElementImpl implements IFilterBlock {

	public static final String AND = "AND";
	public static final String OR = "OR";
	private List childFilters=new ArrayList();
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterElement#prepareFilter(org.w3c.dom.Node)
	 */
	public void prepareFilter(Element  node) throws AdapterInvalidConfig {
		super.prepareFilter(node);
		
		/* validate the operator */
		
		if (!getOperator().equals(AND) && !getOperator().equals(OR)) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAFilter_FilterRuleBlock_Invalid_Operator_ERROR_", getOperator()));
		}
		
		childFilters=new ArrayList();
		
		
		/* Iterate through the child 'nodes' to determine what our filtering path is. */
		NodeList children=node.getChildNodes();
		
		for(int i=0; i<children.getLength(); i++) {
			Element child=null;
			try {
				child=(Element)children.item(i);
			}
			catch(ClassCastException e) {
				/* We can ignore this child as it is not a Element */	
				continue; 	
			}
			if(child.getNodeName().endsWith(Messages.getString("HyadesGAFilterRuleBlockElementTagName"))) {
				IFilterElement element=new FilterBlockImpl();
				element.prepareFilter(child);
				childFilters.add(element);
			}
			else if(child.getNodeName().endsWith(Messages.getString("HyadesGAFilterRuleElementTagName"))) {
				IFilterElement element=new FilterAtomImpl();
				element.prepareFilter(child);
				childFilters.add(element);
			}
		}
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterBlock#getFilterElements()
	 */
	public List getFilterElements() {
		return childFilters;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterBlock#addFilterElement(org.eclipse.hyades.logging.adapter.filters.IFilterElement)
	 */
	public void addFilterElement(IFilterElement element) {
		childFilters.add(element);
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.filters.IFilterElement#evaluateFilter()
	 */
	public boolean evaluateFilter() {
		return processFilterElement(this);
	}
	
	/**
	 * Process the rule tree left to right depth first
	 * @param element
	 * @return
	 */
	private boolean processFilterElement(IFilterElement element) {
		/* Is this a block or an atom.  */
		try {
			IFilterBlock current=(IFilterBlock)element;
			String operator=current.getOperator();
			if(operator.equals(AND)) {
				return and(current);
			}
			else if (operator.equals(OR)) {
				return or(current);
			}
		}
		catch(ClassCastException e) {
			return ((IFilterAtom)element).evaluateFilter();
		}
		return false;
	}
	
	/**
	 * Process the rule subtree left to right depth first for an AND operation
	 * @param block
	 * @return
	 */
	private boolean and(IFilterBlock block) {
		Iterator rules=block.getFilterElements().iterator();
		while(rules.hasNext()) {
			IFilterElement element=(IFilterElement)rules.next();
			if(!processFilterElement(element)) {
				return block.isNegated();
			}
		}
		return !block.isNegated();
	}
	
	/**
	 * Process the rule subtree left to right depth first for an OR operation
	 * @param block
	 * @return
	 */
	private boolean or(IFilterBlock block) {
		Iterator rules=block.getFilterElements().iterator();
		while(rules.hasNext()) {
			IFilterElement element=(IFilterElement)rules.next();
			if(processFilterElement(element)) {
				return !block.isNegated();
			}
		}
		return block.isNegated();
	}

	/** 
	 * Indicates whether filtered items need to be cached.
	 * @return int size of cache
	 */
	public int requiresCache() {
		// Find a rule in the block that requires caching
		Iterator rules=getFilterElements().iterator();
		while(rules.hasNext()) {
			FilterElementImpl element=(FilterElementImpl)rules.next();
			if(element.requiresCache() > 0) {
				return element.requiresCache();
			}
		}

		return 0;
	}
}
