/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TimeFilterType.java,v 1.1 2005/03/23 23:37:14 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.internal.filters;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author smith
 *
 * This class represents a time stamp filter type.  String time stamps are converted
 * to Date objects in order to compare them.
 */
public class TimeFilterType extends NumericalFilterType {
	
	// Formatters for different XML dateTime time stamp formats
	private static SimpleDateFormat formatter1 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.S Z");
	private static SimpleDateFormat formatter2 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.S");
	private static SimpleDateFormat formatter3 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss Z");
	private static SimpleDateFormat formatter4 = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
	
	// Pattern for timezone specification
	private static Pattern tzPattern=Pattern.compile("(.*)([-,+]\\d{2}:\\d{2})");
	
	protected boolean equals(String lhs, String rhs) {
		// check for null operands
		if (lhs == null && rhs == null ) {
			return true;
		}
		else if (lhs == null || rhs == null) { 
			return false;
		}
	
		try {
			Date ldt = getDate(lhs);
			Date rdt = getDate(rhs);

			return ldt.equals(rdt);
		}
		catch (ParseException e) {
			// If we can't convert the time stamps to Date's then return false
		}
		

		return false;	
	}

	/*
	 * greater than operator
	 */
	protected boolean greaterThan(String lhs, String rhs) {
		// check for null operands
		if (lhs == null || rhs == null) { 
			return false;
		}
		try {
			Date ldt = getDate(lhs);
			Date rdt = getDate(rhs);
	
			return ldt.after(rdt);
		}
		catch (ParseException e) {
			// If we can't convert the time stamps to Date's then return false
		}
		
		return false;	
	}
	
	/* 
	 * less than operator
	 */
	protected boolean lessThan(String lhs, String rhs) {
		// check for null operands
		if (lhs == null || rhs == null) { 
			return false;
		}
		
		try {
			Date ldt = getDate(lhs);
			Date rdt = getDate(rhs);
	
			return ldt.before(rdt);
		}
		catch (ParseException e) {
			// If we can't convert the time stamps to Date's then return false
		}
		return false;	
	}
	
	/*
	 * Converts the time stamp string to a Date object using the SimpleDateFormat parser.
	 */
	private Date getDate(String timestr) throws ParseException {
		Date dt;
		
		/* Handle the different timezone specifications.
		 * XML dateTime format for timezone is Z, for UTC time, or +/- hh:mm
		 * SimpleDataFormat parsers recognize the timezone as GMT+/-hh:mm or just GMT for UTC
		 * so add the GMT.
		 */
		if (timestr.endsWith("Z")) {
			timestr = timestr.substring(0,timestr.length()-1) + " GMT";
		}
		else {
			Matcher matcher = tzPattern.matcher(timestr);
			if(matcher.find()) {
				/* Perform the substitution */
				timestr = matcher.replaceAll("$1 GMT$2");
			}
		}
		
		// Parse the timestamp until we get a match
		try {
			dt = formatter1.parse(timestr);
		} catch (ParseException e) {
			try {
				dt = formatter2.parse(timestr);
			} catch(ParseException e2) {
				try {
					dt = formatter3.parse(timestr);
				} catch (ParseException e3) {
					dt = formatter4.parse(timestr);
				}
			}
		}
		
		return dt;
	}
}
