/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SubstituteStatement.java,v 1.19 2005/04/28 21:38:29 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * 
 * Change History:
 * Bugzilla  Description
 * 91218     Performance fix - use a single ArrayList for the result 
 *           instead of instantiating a new one every time.
 * 
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IContext;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.internal.util.SubstitutionExtensionLoaderUtil;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.adapter.util.TimeParser;
import org.w3c.dom.Element;

/**
 * @author rduggan
 *
 * This class represents a substitution rule.  It uses the Jakarta-ORO library
 * for processing regular expressions.
 */
public class SubstituteStatement implements IStatement {
	
	/* The fully qualified name for this substitution rule */
	protected String name;
	
	/* The positions string as specified in the rule */
	protected String positions;
	
	/* The match string as specified in the rule */
	protected String match;
	
	/* The substitute string as specified in the rule */
	protected String substitute;
	
	/* The compiled pattern for this SubstitutionStatement */
	protected Pattern pattern;

	/* The list of positions that will be built based upon the positions string */
	protected List positionsList;
	
	/* The process time indicator */
	protected boolean processTime;
	
	/* Time parser class */
	protected TimeParser timeParser;
		
	/* Members to implement the java callout for processing the parsed data */	
	/* The user class to be used to process the parsed data to generate a value */
	protected String substitutionExtensionClass;
	
	/* The statefulSubstitutionExtension object */
	protected IStatefulSubstitutionExtension statefulExtension = null;
	
	/* The indicator for callout processing */
	protected boolean callout = false;
	
	/* The indicator to always create a new instance of the callout class */
	protected boolean calloutNewInstance = true;
	
	/* The substititution extension class object */
	protected Class substitutionExtensionClassClass = null;
			
	/* The builtin function indicator */
	protected boolean useBuiltin;
	
	/* The parsed path for this rule.  This is loaded into the IDirectedGraph to identify 
	 * this rule uniquely to the next component in the chain.
	 */
	protected List path;
	
	protected IDirectedGraph resultGraph;
	
	/* The parser that this SubstitutionStatement is a child of */
	private IParser parser;
	
	/* Variable to save exception from matchAndSubstitute */
	private Exception matchException = null;
	
	/* List to store the result of running the rule. (bugzilla 91218)
	 * The rule will only produce one result. */
	private List resultList=new ArrayList(1);
	
	public SubstituteStatement(IParser parser) {
		this.parser=parser;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
		positions = node.getAttribute(Messages.getString("HyadesGAPositionsAttributeName"));
		match=node.getAttribute(Messages.getString("HyadesGAMatchAttributeName"));
		substitute=node.getAttribute(Messages.getString("HyadesGASubstituteAttributeName"));
		substitutionExtensionClass=node.getAttribute(Messages.getString("HyadesGAsubstitutionExtensionClassAttributeName"));
		/* User can specify a Java time format that the substitution value will be in.
		 * This format will be used to convert the time string to a long value in milliseconds  
		 */
		String timeFormat=node.getAttribute(Messages.getString("HyadesGAtimeFormatAttributeName"));
		String builtin=node.getAttribute(Messages.getString("HyadesGAuseBuiltInFunctionAttributeName"));
		if(builtin!=null && !builtin.equals("")) {
			if(builtin.equals("true")) {
				useBuiltin=true;
			}
			else {
				useBuiltin=false;
			}
		}
		else {
			useBuiltin=false;
		}

		/* Save the name. */
		this.name=path;
		
		/* Store the path as a list. The path may contain indexes for elements
		 * which are multiples.
		 */
		this.path=DirectedGraphImpl.createPath(path);

		/* If a time format was specified and use builtin was not then set up the time
		 * parser.
		 */
		if (timeFormat!=null && !timeFormat.equals("") && !useBuiltin) {
			processTime = true;
			
			/* Locate our locale as set in our configuration for our current context*/
			Locale locale=null;
			for(IComponent current=parser; current!=null; current=current.getParent()) {
				if(current instanceof IContext) {
					locale=((IContext)current).getLocale();
					break;
				}
			}
			try {
				/* Build our timeparser using the specified locale if we were successful looking
				 * it up.  Otherwise use the default locale (should never happen)
				 */
				if(locale!=null) {
					timeParser = new TimeParser(timeFormat, locale);
				}
				else {
					timeParser = new TimeParser(timeFormat, Locale.getDefault());
				}
			}
			catch (AdapterException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_Invalid_Time_Format_ERROR_",timeFormat), e);
				details.setDigraph(this.name);
				throw details;
			}
		}
								
		/* If a substition extension class was specified and use builtin was not then
		 * check if it is a stateful or stateless class.  
		 * If stateful, get an instance of the class to use when processing all records.
		 */
		if (substitutionExtensionClass!=null && !substitutionExtensionClass.equals("") && !useBuiltin && !processTime) {
			callout = true;
			ISubstitutionExtension extension = null;

			try {
				/* Try to get the substitution extension class class in case we have to instantiate
				 * the class at run time (for stateless classes)
				 */
				substitutionExtensionClassClass = Class.forName(substitutionExtensionClass);
			}
			catch (Throwable e) {
				// We don't need to do anything at this point.
			}
				
			try {
				// load the class and instantiate an object
				extension = SubstitutionExtensionLoaderUtil.instantiate(substitutionExtensionClassClass, substitutionExtensionClass);
			}
			catch (ClassCastException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Run_SubstitutionExtensionClass_Invalid_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}			
			catch (ClassNotFoundException e) {
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_NotFound_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}
			catch(Exception e){
				PreparationException details = new PreparationException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_Exception_ERROR_",substitutionExtensionClass), e);
				details.setDigraph(this.name);
				throw details;
			}
			
			try {
				// Check which type of class we have (stateless or stateful
				statefulExtension = (IStatefulSubstitutionExtension)extension;
				calloutNewInstance = false;
			}
			catch (ClassCastException e) {
				calloutNewInstance = true;
			}
			
		}

		/* Clear the result graph */
		resultGraph=null;
		
		/* Parse the positions attribute value into a tokenized list */
		if(positions!=null && !positions.equals("")) {
			positionsList=PositionParser.getPositionedString(positions, true);
		}
		
	
		/* Compile the match statement */
		if (match!=null && !match.equals("")) {
			compileMatchString(match);
		}

	}

	/*
	 * Compile the match string and set the match pattern
	 */
	protected void compileMatchString(String match) throws PreparationException {
		try {
			pattern=Pattern.compile(match);
		}
		catch(PatternSyntaxException e) {
			/* This is a badly formed regular expression.  Throw it as an error */
			PreparationException details=new PreparationException(Messages.getString("HyadesGAMalformedParserExpression_ERROR_"));
			String detailsString = Messages.getString("HyadesGAPatternSyntaxException_Details", e.getDescription(), e.getPattern());
			if (e.getIndex() >= 0) {
				detailsString += "  " + Messages.getString("HyadesGAPatternSyntaxException_Index", String.valueOf(e.getIndex()));
			}
			details.setDetails(detailsString);
			details.setDigraph(this.name);
			throw details;
		}
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String)
	 */
	public List run(String line, HashMap inputs) throws AdapterException, StatementException {
		String tmpLine = line;
		String newLine;
		boolean result=false;
		
		/* Clear the result list before starting. */
		resultList.clear();
		
		/* If we are running a builtin function then ignore the reset of the content */
		if(useBuiltin) {
			resultGraph=new DirectedGraphImpl(this.path,"##BUILTIN");
			resultList.add(resultGraph);
			return resultList;
		}
		
		/* If we have a positions string modify our inputs accordingly */
		if(positions!=null && !positions.equals("")) {
			tmpLine=positionsToString(line, inputs);
		}
		
		/* Continue only if the specified position string results in valid data */
		if (tmpLine != null) {
			/* If we have a match pattern to look for then lets get the results.  Otherwise
			 * we just place the substitution string in the value.
			 */			
			if (match!=null && !match.equals("")) {
				/* There must be a a substitute value in order to do the substitution */
				if(substitute!= null && !substitute.equals("")) {
					matchException = null;
					newLine=matchAndSubstitute(tmpLine);
                    /* If an exception was thrown inside matchAndSubstitute Then throw an
                     * AdapterException
                     */	
					if (matchException != null) {
						throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Run_Match_And_Substitution_Failure_ERROR_", substitute, match, (String)path.get(path.size()-2) , matchException.getLocalizedMessage()));
					}
				}
				else {
					newLine = null;
				}
				
				result=(newLine!=null && newLine.length()>0);	
			}
			else {
				if (substitute!=null && !substitute.equals("")) {
					newLine=substitute;	
				}
				else {
					newLine=tmpLine;	
				}
				result=true;
			}
			
			/* If we have a match add this match to the result set.  This
			 * is quite a bit of object creation.  Perhaps we could clean this
			 * up some more.
			 */
			if(result) {
				/* If there is a time to parse then do it */
				if (processTime) {
					long convertedTime = timeParser.parse(newLine);
					
					if (convertedTime == 0) {
						resultGraph = null;
						throw StatementException.instance();						
					}
					else {
						/* Create the directed graph and add it to the result list */
						resultGraph = new DirectedGraphImpl(path, new Long(convertedTime));
						resultList.add(resultGraph);
						return resultList;
					}
				}
				
				/* If user wants to use a callout method, do it now */
				if (callout) {
	
					if (calloutNewInstance) {
						ISubstitutionExtension statelessExtension; 
						try {
							if (substitutionExtensionClassClass == null) {
								// If we couldn't find the class in prepare, then load it using the plugin extension
								statelessExtension = SubstitutionExtensionLoaderUtil.instantiate(substitutionExtensionClass);
							}
							else {
								// Instantiate an object
								statelessExtension = (ISubstitutionExtension)substitutionExtensionClassClass.newInstance();
							}
						}
						catch (Throwable e) {
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Preparation_SubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);
						}
						
						try {
							/* Call the appropriate substitution extension class method */	
							if (newLine == line) {
								newLine = statelessExtension.processRecord(newLine);
							}
							else {
	
								newLine = statelessExtension.processMatchResult(newLine);
							}
						}
						catch(Exception e){
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Run_StatelessSubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);
						}
					}
					else {
						try {
							/* Call the appropriate substitution extension class method */
							if (newLine == line) {
								newLine = statefulExtension.processRecord(newLine);
							}
							else {			
								newLine = statefulExtension.processMatchResult(newLine);
							}
						}
						catch (Exception e) {
							throw new AdapterException(Messages.getString("HyadesGA_CBE_Parser_Run_StatefulSubstitutionExtensionClass_Exception_ERROR_", substitutionExtensionClass) , e);				
						}
					}
					if (newLine == null) {
						resultGraph = null;
						throw StatementException.instance(); 
					}
				}
				
				/* RKD:  We are instantiating a new instance here each pass.  This isn't
				 * necessary providing the contract between the Parser and the next element
				 * in the chain is on the same thread and the contract states they will
				 * not be changed.  I am leaving the instantiation because I believe we should
				 * support each component running in its own thread. 
				 */

				// Create a directed graph and add it to the result list.
				resultGraph = new DirectedGraphImpl(path,newLine);
				resultList.add(resultGraph);
				return resultList;
			}

		}
		
		resultGraph=null;
		
		throw StatementException.instance(); 
	}
	
	/**
	 * Check if we have a match and if so do the substitution
	 * @param line to check for pattern match and to do substitution on
	 * @return String resulting string from matching and substitution 
	 */

	protected String matchAndSubstitute (String line) {
		/* We must always have a substitute string */
		if(substitute!= null && !substitute.equals("")) {
			String result;
			try {
				/* Return null if we don't have a match */
				Matcher matcher = pattern.matcher(line);
				if(!matcher.find()) {
					return null;
				} 
				/* Perform the substitution */		
				result = matcher.replaceAll(substitute);
			}
			catch (Exception e) {
				/* Exception occurred when matching or substituting.
				 * Save the exception and return
				 * bugzilla 91218 - not throwing exception here because for
				 * some JVM's it causes the method not to be JIT compiled.
                 */
				matchException = e;
				return null;
			}

			return result.trim(); 
		}
		return null;
		
	}
	
	/**
	 * This method iterates over the positions string and builds a new string
	 * based upon the environment entries.  For each position that is a key into
	 * the enviornment we extract the value and append it to the previous.  All 
	 * of the values are separated by the postion separator.
	 * 
	 * @param line
	 * @param inputs
	 * @return
	 */
	protected String positionsToString(String line, HashMap inputs) {
		if (positionsList!=null && positionsList.size() > 0)	{
			
			/* Iterate over the tokenized postion information and build
			 * a new line.  All positions specified in the list must exist 
			 * in the hash map, otherwise null is returned
			 */
			 StringBuffer result=new StringBuffer();
			 ListIterator iterator=positionsList.listIterator();
			 boolean first=true;
			 while(iterator.hasNext()) {
			 	/* Grab the next token.  It may be a String or a Long */
			 	Object current=iterator.next();
			 	if(first) {
					first=!first;	
			 	}
			 	else {
			 		result.append(PositionParser.POSITION_SEPARATOR);
			 	}
			 	if(current instanceof String) {
					String val=(String)inputs.get(current);
					if (val!=null && val.length()>0) {
						result.append(val);
					}
					else {
						return null;
					}
			 	}
			 	else if(current instanceof Long) {
					Long item = (Long)current;
					int index = item.intValue()-1;
					String val = (String)inputs.get(new Long(index));
					if (val!=null && val.length()>0) {
						result.append(val);
					}
					else {
						return null;
					}
				}
			}
			
			return result.toString();			
		}
		return line;	
	}

	/**
	 * Retrieve the IDirectedGraph created as a result of this rule being run.
	 * @return
	 */
	public IDirectedGraph getResultGraph() {
		return resultGraph;
	}

	/**
	 * Create the IDirectedGraph for this rule with a specific value.
	 * @return
	 */
	public IDirectedGraph createResultGraph(Object value) {
		resultGraph=new DirectedGraphImpl();
		resultGraph.setPath(this.path);
		resultGraph.setValue(value);
		return resultGraph;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isRequiredByParent()
	 */
	public boolean isRequiredByParent() {
		return false;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isChildChoice()
	 */
	public boolean isChildChoice() {
		return false;
	}
}
