/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BaseCorrelatorDataImpl.java,v 1.4 2005/02/16 22:20:15 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import org.eclipse.hyades.collection.correlation.exceptions.InsufficientBufferLengthException;

/**
 * This is the base implementation of a correlator.  Application writers should sub-class this class,  and may
 * override the methods of this class.
 * 
 * @author Qiyan Li
 */
public class BaseCorrelatorDataImpl implements ICorrelatorData {
    private long applicationCounter = 0; // the counter specifying the total order of all events in an application
    private long operationCounter = 0; // the counter specifying the number of push operations
    protected IApplication application = null; // the reference to application associated with this correlator

    /**
     * Invalidates this correlator by resetting it to an initialized correlator.
     */
    public void invalidate() {
        
        applicationCounter = operationCounter = 0;
        application = null;
    }

    /**
     * Sets <code>applicationCounter</code> to <code>n</code>.
     * 
     * @param n the new value of <code>applicationCounter</code>
     */
    public void setApplicationCounter(long n) {
        
        applicationCounter = n;
    }

    /**
     * Returns <code>applicationCounter</code>.
     * 
     * @return n    the new value of <code>applicationCounter</code>
     */
    public long getApplicationCounter() {
        
        return applicationCounter;
    }

    /**
     * Sets <code>operationCounter</code> to <code>n</code>.
     * 
     * @param n the new value of <code>operationCounter</code>
     */
    public void setOperationCounter(long n) {
        
        operationCounter = n;
    }

    /**
     * Returns <code>operationCounter</code>.
     * 
     * @return  the current value of <code>operationCounter</code>.
     */
    public long getOperationCounter() {
        
        return operationCounter;
    }

    /**
     * Converts the <code>offset...offset + 4</code> bytes of the given buffer into an unsigned integer.
     * 
     * @param buffer    the buffer containing the integer
     * @param offset    the offset inside the buffer where the most significant byte of the integer is read
     * @return          the unsigned integer corresponding to the <code>offset..offset + 4</code> bytes.
     */
    public long convertBytesToInteger(byte[] buffer, int offset) {
        
        return ((long) buffer[offset] & 0xff) << 24 | ((long) buffer[offset + 1] & 0xff) << 16 |
               ((long) buffer[offset + 2] & 0xff) << 8 | ((long) buffer[offset + 3] & 0xff);
    }

    /**
     * Converts an unsigned integer into 4 bytes, and saves them in the buffer.
     * 
     * @param n         an unsigned integer to be converted
     * @param buffer    the buffer to stored the integer
     * @param offset    the offset inside the buffer where the most significant byte of the integer is written
     */
    public void convertIntegerToBytes(long n, byte[] buffer, int offset) {
        
        buffer[offset]     = (byte) (n >> 24);
        buffer[offset + 1] = (byte) (n >> 16);
        buffer[offset + 2] = (byte) (n >> 8);
        buffer[offset + 3] = (byte) n;
    }
    
    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#readBinary(byte[], int, int)
     */
    public int readBinary(byte[] buffer, int offset, int length) throws InsufficientBufferLengthException {
        
        // 8 bytes are needed for the application and operation counters, i.e., only the lower order 4 bytes of
        // each counter are used.
        if (length - offset < 8) {
            throw new InsufficientBufferLengthException(offset + 8);
        }

        // Read the application and operation counters from the buffer (big endian).
        applicationCounter = convertBytesToInteger(buffer, offset);
        operationCounter = convertBytesToInteger(buffer, offset + 4);
        return offset + 8;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#writeBinary(byte[], int, int)
     */
    public int writeBinary(byte[] buffer, int offset, int length) throws InsufficientBufferLengthException {
        
        // 8 bytes are needed for the application and operation counters, i.e., only the lower order 4 bytes of
        // each counter are used.
        if (length - offset < 8) {
            throw new InsufficientBufferLengthException(offset + 8);
        }

        // Write the application and operation counters to the buffer (big endian).
        convertIntegerToBytes(applicationCounter, buffer, offset);
        convertIntegerToBytes(operationCounter, buffer, offset + 4);
        return offset + 8;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#readXML(StringBuffer, int, int)
     */
    public int readXML(StringBuffer data, int offset, int length) {
        String buffer; // the buffer corresponding to the data
        int m, n; // the indices where the string should be splitted

        // Clearly, this is an overly simplified implementation of a parser, assuming all goes well.
        buffer = new String(data);
        n = buffer.indexOf("<CorrelationData applicationCounter=\"") + 37;
        m = buffer.indexOf("\" operationCounter=\"", n);
        applicationCounter = Long.parseLong(buffer.substring(n, m));
        
        m += 20;
        n = buffer.indexOf("\"/>", m);
        operationCounter = Long.parseLong(data.substring(m, n));
        return n + 3;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#writeXML(StringBuffer)
     */
    public StringBuffer writeXML(StringBuffer data) {

        data.append("<CorrelatorData applicationCounter=\"" + applicationCounter + "\" " +
                    "operationCounter=\"" + operationCounter + "\"/>");
        return data;
    }

    /**
     * Returns the application associated with this correlator where application-instance-specific data can
     * be retrieved.
     * 
     * @param   the application associated with this correlator.
     */
    void setApplication(IApplication application) {
        
        this.application = application;
    }

    /**
     * @see org.eclipse.hyades.collection.correlation.ICorrelatorData#getApplication()
     */
    public IApplication getApplication() {
        
        return application;
    }

}