/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Stack.java,v 1.4 2005/02/16 22:20:15 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

/**
 * This class offers a simple and efficient implementation of a thread-safe, growable stack.
 * 
 * @author Qiyan Li
 */
public class Stack {
    /**
     * The stack of objects.
     */
    protected Object[] stack;
    
    /**
     * The top-most item on the stack.  A value of -1 indicates that the stack is currently empty.
     */
    protected int top;
    
    /**
     * The size by which the stack grows if more space is required.
     */
    protected int increment;
    
    /**
     * Any private storage that may be necessary for this class.  This is particularly useful when the
     * generality of a super-class and the specialization of a sub-class are desired at the same time,
     * making sub-classing unfeasible.  This field is not managed by the class.
     */
    public Object privateStorage;

    /**
     * Creates an empty stack of <code>size</code> with an <code>increment</code> for holding <code>Object</code>s.
     * 
     * @param size      the initial size of the stack
     * @param increment the size by which the stack grows if its current capacity is insufficient
     */
     public Stack(int size, int increment) {
        stack = new Object[size];
        top = -1;
        this.increment = increment;
    }

    /**
     * Peeks at the top-most item on the stack, or returns <code>null</code> if the stack is empty.
     * 
     * @return  the top-most item on the stack, or <code>null</code> if the stack is empty.
     */
    public synchronized Object peek() {
        try {
            return stack[top];
        } catch (ArrayIndexOutOfBoundsException e) {
            return null;
        }
    }

    /**
     * Peeks at the <code>offset</code>th item from the top of the stack, with the top-most item being <code>0</code>.
     * 
     * @param offset    any non-negative value indicating the desired item on the stack
     * @return          the <code>offset</code>th item from the top of the stack, or <code>null</code> if the offset
     *                  is out of range.
     */
    public synchronized Object peek(int offset) {
        return (offset < 0 || top < offset) ? null : stack[top - offset];
    }

    /**
     * Pushes an item onto the stack.  If there is no room left on the stack, the size of the stack is incremented
     * automatically by <code>increment</code>, after which the item is pushed onto the new stack.
     * 
     * @param item  the object to be pushed onto the stack
     */
    public synchronized void push(Object item) {
        try {
            stack[++top] = item;
        } catch (ArrayIndexOutOfBoundsException e) {
            Object[] newStack; // the new stack for holding more items
            
            newStack = new Object[top + increment];
            System.arraycopy(stack, 0, newStack, 0, top);
            stack = newStack;
            stack[top] = item;
        }
    }

    /**
     * Pops the top-most item off the stack, or <code>null</code> if the stack if empty.
     * 
     * @return  the top-most item on the stack, or <code>null</code> if the stack is empty.
     */
    public synchronized Object pop() {
        try {
            return stack[top--];
        } catch (ArrayIndexOutOfBoundsException e) {
            top++;
            return null;
        }
    }
}
