/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: MultiLevelStatisticViewer.java,v 1.14 2005/03/22 19:06:16 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.adapter.internal;

import java.util.Enumeration;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.filters.FilterTraceViewer;
import org.eclipse.hyades.trace.views.internal.MultiLevelStatisticView;
import org.eclipse.hyades.trace.views.internal.TracePluginImages;
import org.eclipse.hyades.trace.views.internal.TraceUIPlugin;
import org.eclipse.hyades.trace.views.util.internal.OpenSource;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.IPage;

public abstract class MultiLevelStatisticViewer extends FilterTraceViewer {
	
	protected Action _packageLevel;
	protected Action _classLevel;
	protected Action _openSource;   
	protected Action _chooseColumns;   
	protected Action _printColumns;   

	public MultiLevelStatisticViewer() {
		super();
	}

	public abstract String getViewTitle();	
	public abstract TraceViewerPage createPage(EObject mofObject);	

	public void setCheckedLevels(MultiLevelStatisticPage page, int newLevel)
	{
		if (page.getView().getLevel() == newLevel)
		{
			page.getView().updateButtons();
		}
		else
		{
			page.getView().setLevel(newLevel);
			page.getView().updateButtons();
			page.getView().redrawTable();
		}
	}
	
	/**
	 * return the selected agent in the Profiling monitor view
	 * called by addViewPage() -handelProfileEvent() - refreshVisiblePage()and
	 * removePage() of the parent class (TraceViewer)
	 */
	public EObject getObjectToView(EObject selObject) {
		if (selObject == null)
			return selObject;

		if (selObject instanceof TRCProcessProxy) {
			int nbOfAgents = 0;
			TRCAgentProxy a = null;
			EList agents = ((TRCProcessProxy) selObject).getAgentProxies();
			for (int idx = 0; idx < agents.size(); idx++) {
				TRCAgentProxy agent = (TRCAgentProxy) agents.get(idx);
				if (agent == null || agent.eIsProxy())
					continue;
				if (agent
					.getType()
					.equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
					nbOfAgents++;
					a = agent;
				}
			}
			if (nbOfAgents == 1) {
				return a;
			}
		}
		return selObject;
	}

	/**
	 * @see IWorkbenchPart#dispose
	 */
	public void dispose() {
		Enumeration pages = _pages.elements();
		while (pages.hasMoreElements()) {
			IPage page = (IPage) pages.nextElement();

			if (page instanceof MultiLevelStatisticPage)
				 ((MultiLevelStatisticPage) page).dispose();
		}

		setActionsNull();

		if (fContextMenu != null)
			fContextMenu.dispose();

		fContextMenu = null;

		super.dispose();
	}

	protected void setActionsNull()
	{
		_classLevel = null;
		_packageLevel = null;
		_openSource = null;		
	}
	
	/**
	 * set the focus on the current page
	 */
	public void setFocus() {
		MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();
		if (page != null)
			page.setFocus();
	}
	
	protected void getLevelActions(Vector actions)
	{
		String strPackageLevel =
			TraceUIPlugin.getString("STR_ST_PACKAGE_LEVEL");
		_packageLevel = new Action(strPackageLevel) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();

				if(page == null)
				{
				  revertChecked(_packageLevel);
				  return;
				}

				setCheckedLevels(page, MultiLevelStatisticView.PACKAGE_LEVEL);
			}
			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}
		};

		_packageLevel.setText(strPackageLevel);
		TracePluginImages.setImageDescriptors(
			_packageLevel,
			TracePluginImages.T_LCL,
			TracePluginImages.IMG_PACKAGE);
		_packageLevel.setDescription(strPackageLevel);
		_packageLevel.setToolTipText(strPackageLevel);
		_packageLevel.setEnabled(true);
		_packageLevel.setChecked(true);

		String strClassLevel = TraceUIPlugin.getString("STR_ST_CLASS_LEVEL");

		_classLevel = new Action(strClassLevel) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();

				if(page == null)
				{
				  revertChecked(_classLevel);
				  return;
				}
				
				setCheckedLevels(page, MultiLevelStatisticView.CLASS_LEVEL);
			}

			public int getStyle() {
				return IAction.AS_CHECK_BOX;
			}
		};

		_classLevel.setText(strClassLevel);
		TracePluginImages.setImageDescriptors(
			_classLevel,
			TracePluginImages.T_LCL,
			TracePluginImages.IMG_CLASS);
		_classLevel.setDescription(strClassLevel);
		_classLevel.setToolTipText(strClassLevel);
		_classLevel.setEnabled(true);
		
		actions.addElement(new Separator());		
		actions.addElement(_packageLevel);
		actions.addElement(_classLevel);
	}
	
	protected void getActions(Vector actions)
	{
		String openSourceStr = TraceUIPlugin.getString("STR_SOURCE");
		_openSource = new Action(openSourceStr)
		{
			public void run()
			{
				MultiLevelStatisticPage page = (MultiLevelStatisticPage)getCurrentPage();

				if(page == null)
				  return;

				OpenSource.openSource(UIPlugin.getDefault().getSelectionModel(page.getMOFObject()).getFirstElement());
			}
		};

		_openSource.setText(openSourceStr);
		TracePluginImages.setImageDescriptors(_openSource,TracePluginImages.T_TOOL,TracePluginImages.IMG_SOURCE);	
		_openSource.setDescription(openSourceStr);   
		_openSource.setToolTipText(openSourceStr);
		_openSource.setEnabled(false);

		String strPrintColumns = TraceUIPlugin.getString("STR_ST_PRINT_COL");
		_printColumns = new Action(strPrintColumns) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage) getCurrentPage();
				if (page == null) {
					return;
				}
				page.getView().getPrintColumnsAction().run();
			}
		};
		_printColumns.setText(strPrintColumns);
		TracePluginImages.setImageDescriptors(_printColumns, TracePluginImages.T_LCL, TracePluginImages.IMG_PRINTCOLUMNS);
		_printColumns.setDescription(strPrintColumns);
		_printColumns.setToolTipText(strPrintColumns);
		_printColumns.setEnabled(true);

		String strChooseColumns = TraceUIPlugin.getString("STR_ST_COL_SETTINGS");
		_chooseColumns = new Action(strChooseColumns) {
			public void run() {
				MultiLevelStatisticPage page = (MultiLevelStatisticPage)getCurrentPage();
				if (page == null) {
					return;
				}
				MultiLevelStatisticView view = page.getView();
				view.getChooseColumnsAction(view.getColumnDataList(), view.getColumnsPreferencesKey()).run();
			}
		};
		_chooseColumns.setText(strChooseColumns);
		TracePluginImages.setImageDescriptors(_chooseColumns, TracePluginImages.T_LCL, TracePluginImages.IMG_CHOOSECOLUMNS);
		_chooseColumns.setDescription(strChooseColumns);
		_chooseColumns.setToolTipText(strChooseColumns);
		_chooseColumns.setEnabled(true);

		actions.addElement(_openSource);
		actions.addElement(new Separator());
		actions.addElement(_chooseColumns);
		actions.addElement(_printColumns);
	}

	/**
	 * Three action buttons are defined:
	 * -packageLevel: display all the tree (package/class/method)
	 * -classlevel: display only the class tree
	 * -method level: display only the methods
	 */
	public void makeActions() {
		if (isInitializedMenu())
			return; //create the toolbar only once

		super.makeActions();
		
		initializedMenu(true);

		Vector actions = new Vector();
		actions.addElement(new Separator());		
		getLevelActions(actions);
		actions.addElement(new Separator());		
		getActions(actions);
		
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();

		//tbm.removeAll();

		for (int i=0; i < actions.size(); i++)
		{
			if (actions.elementAt(i) instanceof IContributionItem)
				tbm.add((IContributionItem)actions.elementAt(i));
			else if(actions.elementAt(i) instanceof IAction)
				tbm.add((IAction)actions.elementAt(i));
		}


		// require update because toolbar control has been created by this point,
		// but manager does not update it automatically once it has been created
		getViewSite().getActionBars().updateActionBars();
	}

	/**
	 * @return the class level action
	 */
	public Action getClassLevel() {
		return _classLevel;
	}

	/**
	 * @return the package level action
	 */
	public Action getPackageLevel() {
		return _packageLevel;
	}

	public Action openSource() {
		return _openSource;
	}
	
	protected void revertChecked(Action action)
	{
		action.setChecked(!action.isChecked());
	}	
	
	public String[] getFilterScope()
	{
		String[] scope = new String[1];
		scope[0] = HyadesConstants.PROFILING_AGENT_FILTER_SCOPE;
		return scope;
	}	
	
	public void filterUpdated()
	{
		TraceViewerPage page = (TraceViewerPage)getCurrentPage();
		if (page != null) {
			if (fPartVisible)
				page.refreshPage();
			else
				fForceRefresh = true;
		}
	}		
}
