/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JCanvas.java,v 1.4 2005/02/16 22:24:04 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.resource.*;
import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

public class JCanvas implements SelectionListener,
								KeyListener
{
	protected final float MAG_STEP = 2;
	protected final static int ARROW_FACTOR = 4;
	protected final static int MIN_FONT_SIZE = 5;
	protected final static int MAX_FONT_SIZE = 72;
	protected final static float GROWTH_FACTOR = 1.25f;
	protected StdDrawStrategy _objDS;
	protected float _left, _top;              // extremes of normalized space
	protected float _prefWidth, _prefHeight;

	protected float _xscale = 1f, _yscale = 1f;
	protected float _xscaleMax = 64f, _yscaleMax = 64f;
	protected Point _origin = new Point(0, 0);    // scrolling offset
	protected Rectangle _prevSize = new Rectangle(0, 0, 0, 0);

	private Cursor _zoomInCursor;
	private Cursor _zoomOutCursor;	
	private Cursor _defaultCursor;
	private boolean _zoomIn = false;
	private boolean _zoomOut = false;

	protected boolean _controlDown = false;

	protected BufferedCanvas fCanvas;
	protected int fCanvasStyle;
	public final static int fScrollBarWidth = 12;
	protected int fHSel = 0;
	protected int fVSel = 0;        
	protected int fHeight;

	public JCanvas (Composite parent, int style)
	{
		this(parent, style, 0);		
	}
	public JCanvas(Composite parent, int style, int height)
	{
		fCanvasStyle = style;
		fHeight = height;
		
		ImageDescriptor zoomInSource = TracePluginImages.DESC_IMG_ZOOMIN_SOURCE;
		ImageDescriptor zoomOutSource = TracePluginImages.DESC_IMG_ZOOMOUT_SOURCE;
		ImageDescriptor zoomMask = TracePluginImages.DESC_IMG_ZOOMMASK;

		_zoomInCursor = new Cursor(Display.getCurrent(), zoomInSource.getImageData(), zoomMask.getImageData(), 0, 0);	
		_zoomOutCursor = new Cursor(Display.getCurrent(), zoomOutSource.getImageData(), zoomMask.getImageData(), 0, 0);	
				
		createControl(parent);
		
		_defaultCursor = new Cursor(Display.getCurrent(), SWT.CURSOR_ARROW);	
	}
	// Align the vertical aspects of this canvas with the state
	// of another JCanvas.  Returns true if alignment was needed.
	public boolean alignVertical(JCanvas source) {
		if (_top != source._top ||
			_origin.y != source._origin.y ||
			_prefHeight != source._prefHeight ||
			_yscale != source._yscale)
			{
			_top = source._top;
			_origin.y = source._origin.y;
			_prefHeight = source._prefHeight;
			yscale(source.yscale());
			return true;
			}
		 else
			return false;
	}
	public float bottom()   { return _top + height(); }
/**
 * Insert the method's description here.
 * Creation date: (10/16/2000 12:43:11 PM)
 * @return com.ibm.swt.widgets.Canvas
 */
public Canvas canvas() {
	return fCanvas;
}
	/**
	 * Returns SWT Widget that will be made the client of the view pane.
	 * Subclassers should override doCreateWindow instead
	 */
	protected final Control createControl(Composite parent)
	{
		// Initialize Canvas
		fCanvas = new BufferedCanvas(parent, fCanvasStyle)
		{
			public void doPaint(GC gc) {
				paint(gc);
			}           
		};
		fCanvas.setBackground(SpectrumColorMap.getBackgroundColor());	  
		
		ScrollBar horizontalBar = fCanvas.getHorizontalBar();
		ScrollBar verticalBar = fCanvas.getVerticalBar();
		
		if(horizontalBar != null)
			horizontalBar.addSelectionListener(this);

		if(verticalBar != null)
			verticalBar.addSelectionListener(this);
		
		fCanvas.addKeyListener(this);

		return fCanvas;     
	}
	public int denormX (float x)
	{
//      return Math.round((x - _left)*_xscale);     
		return Math.round((x - _left)*_xscale - _origin.x);
	}
	public int denormY (float y)
	{
		return Math.round((y - _top)*_yscale - _origin.y);
//      return Math.round((y - _top)*_yscale);
	}
	/**
	 * Zoom and center on (cx, cy), in pre-zoom canvas coordinates.
	 */
	protected void denormZoom (float zx, float zy, float cx, float cy) {
		float lzx = Math.min(_xscaleMax/xscale(), zx);
		float lzy = Math.min(_yscaleMax/yscale(), zy);

/////////// wrong optimization in case zx and zy are 1 but cx and cy have changed ///////////

		if (lzx != 1 || lzy != 1) {
			xscale(xscale() * lzx);
			yscale(yscale() * lzy);

			zoomJCanvasScroller(lzx, lzy, cx, cy);
		}

		updateScrollbars();
	}
	public void drawArrow (GC g,
		float nx1, float ny1, float nx2, float ny2, Color col
	) {
		int x1 = denormX(nx1);
		int y1 = denormY(ny1);
		int x2 = denormX(nx2);
		int y2 = denormY(ny2);

		int xd = (x2-x1);
		int yd = (y2-y1);

		int len = (int)Math.sqrt(xd*xd+yd*yd);
		if (len >1) {
			xd = Math.round(xd * ARROW_FACTOR * _xscale / len);
			yd = Math.round(yd * ARROW_FACTOR * _yscale / len);
		}
		if (xd != 0 || yd != 0)
		{
			g.setForeground(col);
			g.setBackground(col);
			g.drawLine(x2, y2, x2-xd-yd/2, y2+xd/2-yd);
			g.drawLine(x2, y2, x2-xd+yd/2, y2-xd/2-yd);         
		}
		
		g.setForeground(col);
		g.setBackground(col);
		g.drawLine(x1, y1, x2, y2);     
	}
	// Draw or fill diamond
	public void drawDiamond (GC g,
		float nx, float ny, float nw, float nh, Color col, boolean fill
	)
	{
		int x = denormX(nx);
		int y = denormY(ny);
		float xs = _xscale;
		float ys = _yscale;

		int w = Math.round(xs * nw);
		int h = Math.round(ys * nh);

		if (w < 1) w = 1;
		if (h < 1) h = 1;

		int points[] = {x+w/2, y, x+w, y+h/2, x+w/2, y+h, x, y+h/2};
		g.setForeground(col);
		g.setBackground(col);

		if (fill)
			g.fillPolygon(points);
		else
			g.drawPolygon(points);  
			
		_objDS.setHeight(Math.max(_objDS.getHeight(), y+h));
		_objDS.setWidth(Math.max(_objDS.getWidth(), x+w));
				
	}
	public void drawLine (GC g,
		float nx1, float ny1, float nx2, float ny2, Color col
	) {

		g.setForeground(col);
		g.setBackground(col);

		int x1 = denormX(nx1);
		int y1 = denormY(ny1);

		int x2 = denormX(nx2);
		int y2 = denormY(ny2);
		
		g.drawLine(x1, y1, x2, y2);
	}
	public void drawLoop (GC g, float nx, float ny, float nw, float nh, Color col) {
		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);
		int xd = Math.round(ARROW_FACTOR * _xscale);
		int yd = Math.round(ARROW_FACTOR * _yscale);

		g.setForeground(col);
		g.setBackground(col);               
		g.drawOval(x-w/2, y-h, w, h);
	}
	public void drawRect (GC g, float nx, float ny, float nw, float nh, Color col) {

		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);

		if (w < 1 || h < 1)
		{
			w=1; h=1;
			return;
		}           
		
		g.setForeground(col);
		g.setBackground(col);       
			
		g.drawRectangle(x, y, w, h);

		_objDS.setHeight(Math.max(_objDS.getHeight(), y+h));
		_objDS.setWidth(Math.max(_objDS.getWidth(), x+w));
		
	}
	public void drawRectXor (float nx, float ny, float nw, float nh) {

		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);

		Display display = Display.getCurrent();
		if(display == null) display = Display.getDefault();
		
		GC g = new GC(fCanvas);
		g.setXORMode(true);
		g.setForeground(display.getSystemColor(SWT.COLOR_WHITE));                       
		g.setBackground(display.getSystemColor(SWT.COLOR_BLACK));               

		if (w >= 1 && h >= 1)
			g.drawRectangle(x, y, w, h);

		g.dispose();            
	}
	public void drawRectXor (GC g, float nx, float ny, float nw, float nh) {

		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);

		Display display = Display.getCurrent();
		if(display == null) display = Display.getDefault();
		
		g.setXORMode(true);
		g.setForeground(display.getSystemColor(SWT.COLOR_WHITE));                       
		g.setBackground(display.getSystemColor(SWT.COLOR_BLACK));               

		if (w >= 1 && h >= 1)
			g.drawRectangle(x, y, w, h);
	}
	public void drawSquare (GC g, float nx, float ny, float nw, Color col) {
		
		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);

		if (w < 1) return;

		g.setForeground(col);
		g.setBackground(col);               
		g.drawRectangle(x, y, w, w);
	}
	public void drawString (GC gc, String s, float nx, float ny, Color col) {

		int x = denormX(nx);
		int y = denormY(ny);

		gc.setForeground(col);
		gc.drawString(s, x, y-10, true);            
	}
	public void drawString (GC gc, String s, float nx, float ny, Color col, Color bgColor) {

		int x = denormX(nx);
		int y = denormY(ny);

		gc.setForeground(col);
		gc.setBackground(bgColor);      
		gc.drawString(s, x, y-10);
			

	}
	protected boolean extendedTo (float x, float y) {
		boolean outside = false;
		float dx = 0, dy = 0;

		if (x < left()) {
			dx = (left() - x) * GROWTH_FACTOR;
			_prefWidth += dx;
			left(x);
			outside = true;

		} else if (x > left()+width()) {
			_prefWidth = (x - left()) * GROWTH_FACTOR;
			outside = true;
		}

		if (y < top()) {
			dy = (top() - y) * GROWTH_FACTOR;
			_prefHeight += dy;
			top(y);
			outside = true;

		} else if (y > top()+height()) {
			_prefHeight = (y - top()) * GROWTH_FACTOR;
			outside = true;
		}

		if (outside) {
			int cdx = Math.round(dx * _xscale);
			int cdy = Math.round(dy * _yscale);

			_origin.x += cdx;
			_origin.y += cdy;
		}

		return outside;
	}
	protected float fdenormX (float x)
	{
		return (x - _left)*_xscale - _origin.x;
	}
	protected float fdenormY (float y) {
		return (y - _top)*_yscale - _origin.y;
	}
	public void fill3DDiamond (GC g,
		float nx, float ny, float nw, float nh, Color col) {

		int x = denormX(nx);
		int y = denormY(ny);
		float xs = _xscale;
		float ys = _yscale;

		int w = Math.round(xs * nw);
		int h = Math.round(ys * nh);

		if (w < 1) w = 1;
		if (h < 1) h = 1;

		int xPoints[] = {x+w/2, x+w, x+w/2, x};
		int yPoints[] = {y, y+h/2, y+h, y+h/2};

		int points[] = {x+w/2, y, x+w, y+h/2, x+w/2, y+h, x, y+h/2};

		g.setForeground(col);
		g.setBackground(col);       
		
		g.fillPolygon(points);

		Color darker = new Color(null, 0, col.getGreen(), col.getBlue());

		g.setForeground(darker);
		g.setBackground(darker);        
		g.drawLine(x+w/2, y, x+w, y+h/2);
		g.drawLine(x+w, y+h/2, x+w/2, y+h);
		g.drawLine(x+w+1, y+h/2, x+w/2, y+h+1);

		Color brighter = new Color(null, col.getRed(), col.getGreen(), 0);
		g.setForeground(brighter);
		g.setBackground(brighter);      

		g.drawLine(x+w/2, y+h, x, y+h/2);
		g.drawLine(x, y+h/2, x+w/2, y);
		g.drawLine(x-1, y+h/2, x+w/2, y-1);

		_objDS.setHeight(Math.max(_objDS.getHeight(), y+h));
		_objDS.setWidth(Math.max(_objDS.getWidth(), x+w));
		
	}
	public void fill3DRect (GC g,
		float nx, float ny, float nw, float nh, Color col
	) {

		int x = denormX(nx);
		int y = denormY(ny);
		  
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);
		
		if (w < 1) w = 1;
		if (h < 1) h = 1;

		g.setForeground(col);
		g.setBackground(col);       
		
		g.fillRoundRectangle(x, y, w, h, 1, 1);
		g.fillRoundRectangle(x+1, y+1, w-2, h-2, 1, 1); // what if w==1?

		_objDS.setHeight(Math.max(_objDS.getHeight(), y+h));
		_objDS.setWidth(Math.max(_objDS.getWidth(), x+w));
		
	}
	/**
	 * This method should fill (or update) the given menu. It is called just before
	 * the menu opens. 
	 */
	public void fillContextMenu(IMenuManager menu)
	{
//		menu.add(new Separator("additions"));
//		menu.add(new Separator("additions-end"));
		
		if(_objDS != null)
		  _objDS.fillContextMenu(menu);
	}
	public void fillOval (GC g, float nx, float ny, float nw, float nh, Color col) {

		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);

		g.setForeground(col);
		g.setBackground(col);               
		g.fillOval(x, y, w, h);
	}
	public void fillRect (GC g,
		float nx, float ny, float nw, float nh, Color col
	) {
			
		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);
		int h = Math.round(_yscale * nh);

		if (w < 1) w = 1;
		if (h < 1) h = 1;

		g.setForeground(col);
		g.setBackground(col);               
		g.fillRectangle(x, y, w, h);

		_objDS.setHeight(Math.max(_objDS.getHeight(), y+h));
		_objDS.setWidth(Math.max(_objDS.getWidth(), x+w));
	}
	public void fillSquare (GC g, float nx, float ny, float nw, Color col)
	{
		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(_xscale * nw);

		if (w < 1) w=1;

		g.setBackground(col);               
		g.fillRectangle(x, y, w, w);        
	}
/**
 * Insert the method's description here.
 * Creation date: (12/11/2000 1:43:40 PM)
 * @param style int
 */

/**
 * Returns the primary control associated with this viewer.
 */
public Control getControl()
{
	return fCanvas;
}
/**
 * Insert the method's description here.
 * Creation date: (10/13/2000 11:53:57 AM)
 * @return com.ibm.swt.graphics.Point
 */
public Point getOrigin() {
	return _origin;
}
   protected Point getPreferredSize ()
   {
		int width = Math.round(_prefWidth*xscale());
		int height = Math.round(_prefHeight*yscale());

		return new Point(width, height);
	}
/**
 * Insert the method's description here.
 * Creation date: (6/14/2001 4:40:28 PM)
 * @return int
 */
public int getScrollX() {
	return fHSel;
}
/**
 * Insert the method's description here.
 * Creation date: (06/15/2000 10:16:24 AM)
 * @return com.ibm.swt.graphics.Rectangle
 */
public Rectangle getSize()
{   
	Rectangle rect = fCanvas.getClientArea();
	
	return rect;        
}
	/**
	 * Return the visible area in canvas coordinates.
	 */
	protected Rectangle getVisible ()
	{
		Rectangle rect = fCanvas.getClientArea();
		rect.height-=fHeight;       
		
		return rect;
	}
	public float height ()  { return _prefHeight; }
	public void home () {
		if (xscale() == 1 && yscale() == 1) {
			// optimization in case there's no zooming
			scrollTo(0, 0);

		} else {
			xscale(1);
			yscale(1);

			_origin.x=0;
			_origin.y=0;
		}
	}
	public void keyPressed (KeyEvent e)  {
		if (e.keyCode == SWT.CONTROL) {
			_controlDown = true;

		} else if (e.keyCode == SWT.PAGE_UP) {
			pageUp();

		} else if (e.keyCode == SWT.PAGE_DOWN) {
			pageDown();

		} else if (e.keyCode == SWT.ARROW_UP) {
			page(0, -.1f);

		} else if (e.keyCode == SWT.ARROW_DOWN) {
			page(0, .1f);

		} else if (e.keyCode == SWT.ARROW_LEFT) {
			if (_controlDown)
				page(-.9f, 0);
			else
				page(-.1f, 0);

		} else if (e.keyCode == SWT.ARROW_RIGHT) {
			if (_controlDown)
				page(.9f, 0);
			else
				page(.1f, 0);

		}
		else if (e.keyCode == SWT.HOME)
		{
		   zoomToFill(1f, 1000f);
		   redraw();
		}
		else if (e.keyCode == SWT.INSERT || e.character == '=' || e.character == '+'
		)
		{
			zoomIn();
			redraw();

		}
		else if (e.character == '_' || e.character == '-')
		{
			zoomOut();
			redraw();

		}
	}
	public void keyReleased (KeyEvent e)  {
		if (e.keyCode == SWT.CONTROL) {
			_controlDown = false;
		}
	}
	public void keyTyped (KeyEvent e)  { }
	public float left ()    { return _left; }
	public void left (float x)   { _left = x; }
	public float normX (int x) { return ((x + _origin.x)/_xscale + _left); }
	public float normY (int y) { return ((y + _origin.y)/_yscale + _top ); }
	protected void page (float fx, float fy)
	{
		page(fx, fy, true);
	}
	protected void page (float fx, float fy, boolean updateScrollbars)
	{
		if(fx !=0 )
		{
			int visibleWidth = (int)getVisible().width;
			int totalWidth = getPreferredSize().x;
			int hiddenLeft = _origin.x;
			int hiddenRight = totalWidth - visibleWidth - hiddenLeft;

			if(fx>0 && hiddenRight <=0)
			{
			   _origin.x=getPreferredSize().x-getVisible().width;
			 if(updateScrollbars)
				 updateScrollbars();  
				
				return;
			}
				
			if(fx<0 && hiddenLeft <=0 )
			{
			   _origin.x=0;
			 if(updateScrollbars)
				 updateScrollbars();  
			   return;
			}
		}

		if(fy != 0)
		{
			int visibleHeight = (int)getVisible().height;
			int totalHeight = getPreferredSize().y;
			int hiddenTop = _origin.y;
			int hiddenBottom = totalHeight - visibleHeight - hiddenTop;

			if(fy<0 && hiddenTop <=0)
			{
			   _origin.y = 0;
			 if(updateScrollbars)
				 updateScrollbars();  
				return;
			}
				
			if(fy>0 && hiddenBottom <=0 )
			   return;
		}
		  
		Rectangle visible = getVisible();
		scrollBy(Math.round(visible.width*fx), Math.round(visible.height*fy), updateScrollbars);

	}
	public void pageDown () { page(0, .8f); }
	public void pageUp () { page(0, -.8f); }
/**
 * Insert the method's description here.
 * Creation date: (6/7/2001 10:30:38 AM)
 * @param gc org.eclipse.swt.graphics.GC
 */
protected void paint(GC gc)
{
	if(_objDS == null)
	  return;
		
	FontData data= SpectrumColorMap.getFontData();
	Font font = new Font(Display.getDefault(), data);
	gc.setFont(font);
		
	_objDS.setDirty();
	if(_objDS.isDirtyAndReset())
	{
		_objDS.bgRedraw(gc);

	  if(sizeChanged())
		updateScrollbars();
		  
	  _prevSize = getVisible();
	}

	font.dispose();
}
	public void redraw ()
	{
	  Display display = Display.getCurrent();
	  if (display == null) {
	  // This is not the UI thread => fork
	    Display.getDefault().asyncExec (new Runnable() {
	      public void run() {
	        fCanvas.redraw();
	      }
	    });
	  } else {
		  fCanvas.redraw();
		}			
	}
	public float right()    { return _left + width(); }
	/**
	 * Scroll by an amount specified in canvas coordinates.
	 */
	public void scrollBy (int dx, int dy) {
		scrollBy(dx, dx, true);
	}
	/**
	 * Scroll by an amount specified in canvas coordinates.
	 */
	public void scrollBy (int dx, int dy, boolean updateScrollbars) {
		scrollTo(_origin.x + dx, _origin.y + dy, updateScrollbars);
	}
	/**
	 * Scroll to a position specified in canvas coordinates.
	 */
	public void scrollTo (int x, int y) 
	{
		scrollTo(x, y, true);
	}
	/**
	 * Scroll to a position specified in canvas coordinates.
	 */
	public void scrollTo (int x, int y,boolean updateScrollbars)    
	{
		if(x < 0)
		  x =0;

		if(y < 0)
		  y = 0;
		   
		Point p = new Point(x, y);

		if (p.x != _origin.x || p.y != _origin.y) {
			_origin = p;
			redraw();

			if(updateScrollbars)
			  updateScrollbars(p.x, p.y);
		}

	}
	/**
	 * Specify the size of the scrollable area in normalized coordinates.
	 */
	public void setArea (float width, float height) {
		_prefWidth = width;
		_prefHeight = height;
	}
/**
 * Insert the method's description here.
 * Creation date: (06/16/2000 2:48:55 PM)
 * @param strategy jinsight.views.StdDrawStrategy
 */
public void setDrawStrategy(StdDrawStrategy strategy)
{
	_objDS = strategy;

	if(fCanvas == null || _objDS == null)
	  return;
	
	fCanvas.addMouseListener(_objDS);
	fCanvas.addMouseMoveListener(_objDS);   
	
}
	/**
	 * Attaches a contextmenu listener to the table
	 */
	public void setMenuListener(IMenuListener menuListener)
	{
		MenuManager menuMgr= new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(menuListener);
		Menu menu = menuMgr.createContextMenu(fCanvas);
		
		fCanvas.setMenu(menu);      
	}
	public void setOrigin (Point p) { _origin = p; }
	public void setZoomLimits (float zx, float zy) {
		_xscaleMax = zx;
		_yscaleMax = zy;
	}
	protected boolean sizeChanged () {
		
		Rectangle size = getVisible();
		return size.width != _prevSize.width || size.height !=_prevSize.height;
	}
	public float top ()     { return _top; }
	public void top (float y)    { _top = y; }
public void updateScrollbars ()
{
	
	if(_objDS == null)
	  return;

	ScrollBar hb = fCanvas.getHorizontalBar();
	ScrollBar vb = fCanvas.getVerticalBar();

	if(hb != null)
	{
		int visibleWidth = (int)getVisible().width;
		int totalWidth = getPreferredSize().x;
		int hiddenLeft = _origin.x;
		int hiddenRight = totalWidth - visibleWidth - hiddenLeft;
		hb.setMinimum(0);
		hb.setMaximum(totalWidth-visibleWidth);

		if(totalWidth <= visibleWidth )
		{
			hb.setVisible(false);           
		}
		else
		{
			hb.setVisible(true);            

			int fHPageIncrement = Math.round(visibleWidth*0.1f);            
			if(fHPageIncrement == 0)
			  return;

			int steps = hb.getMaximum()/fHPageIncrement+2;

			hb.setMaximum(steps*fHPageIncrement);               
			hb.setThumb(hb.getMaximum()/steps);             
			hb.setSelection(hiddenLeft);
			hb.setPageIncrement(fHPageIncrement);           
			hb.setIncrement(fHPageIncrement);

			fHSel = hb.getSelection();
		}
	}   

	if(vb != null)
	{
		int visibleHeight = (int)getVisible().height;
		int totalHeight = getPreferredSize().y;
		int hiddenTop = _origin.y;
		int hiddenBottom = totalHeight - visibleHeight - hiddenTop;
		vb.setMinimum(0);
		vb.setMaximum(totalHeight-visibleHeight);

		if(totalHeight <= visibleHeight )
		{
			vb.setVisible(false);           
		}
		else
		{
			vb.setVisible(true);            
			int fVPageIncrement = Math.round(visibleHeight*0.1f);
			if(fVPageIncrement == 0)
			  return;
				
			int steps = vb.getMaximum()/fVPageIncrement+2;

			vb.setMaximum(steps*fVPageIncrement);           
			vb.setThumb(vb.getMaximum()/steps);             
			vb.setSelection(hiddenTop);
			vb.setPageIncrement(fVPageIncrement);           
			vb.setIncrement(fVPageIncrement);

			fVSel = vb.getSelection();
		}
		
	}
}
protected void updateScrollbars (int dx, int dy)
{
	
	if(_objDS == null)
	  return;

	ScrollBar hb = fCanvas.getHorizontalBar();
	ScrollBar vb = fCanvas.getVerticalBar();
	
	if(hb != null)
	{
		hb.setSelection(dx);
		fHSel = hb.getSelection();  
	}   

	if(vb != null)
	{
		vb.setSelection(dy);
		fVSel = vb.getSelection();  
	}
	
}
	public boolean visible (
		float nx, float ny, float nw, float nh
	)
	{
		int x = denormX(nx);
		int y = denormY(ny);
		int w = Math.round(xscale()*nw);
		int h = Math.round(yscale()*nh);

		Rectangle visible = getVisible();

		return
			(x <= visible.width) &&  (x+w >= 0) &&
			(y <= visible.height) && (y+h >= 0);
	}
	public float visibleBottom () { return normY(getVisible().height); }
	public float visibleHeight () { return getVisible().height / yscale(); }
	public float visibleLeft () { return normX(0); }
	public float visibleRight () { return normX(getVisible().width);}
	public float visibleTop () { return normY(0); }
	public float visibleWidth () { return getVisible().width / xscale(); }
	public void widgetDefaultSelected (SelectionEvent e) {
	}
	public void widgetSelected (SelectionEvent e)
	{
		if(e.widget == fCanvas.getHorizontalBar())
		{
			ScrollBar hb = (ScrollBar)e.widget;
			int sel = hb.getSelection();
			
			if(sel != 0 && sel >= fHSel && fHSel >= 0)
			{
				int diff = sel - fHSel;
				if(diff > 0 && hb.getPageIncrement() > 0)
				{
					float pageIncr = (diff*.1f)/hb.getPageIncrement();
					page(pageIncr, 0, false);
				}
			}
			else
			{
				int diff = fHSel - sel;
				if(diff > 0 && hb.getPageIncrement() > 0)
				{
					float pageIncr = (diff*.1f)/hb.getPageIncrement();
					page(-pageIncr, 0, false);
				}
			}
				
			fHSel = sel;
		}
		else if(e.widget == fCanvas.getVerticalBar())   
		{
			ScrollBar vb = (ScrollBar)e.widget;
			int sel = vb.getSelection();
	
			if(sel != 0 && sel >= fVSel && fVSel >= 0)
			{
				int diff = sel - fVSel;
				if(diff > 0 && vb.getPageIncrement() > 0)
				{
					float pageIncr = (diff*.1f)/vb.getPageIncrement();
					page(0, pageIncr, false);
				}
			}
			else
			{
				int diff = fVSel-sel;
				if(diff > 0 && vb.getPageIncrement() > 0)
				{
					float pageIncr = (diff*.1f)/vb.getPageIncrement();
					page(0, -pageIncr, false);
				}
			}

			fVSel = sel;    

		}   
	}
	public float width ()   { return _prefWidth; }
	public float xscale () 
	{
		return _xscale;
	}
	protected void xscale (float s)
	{
		_xscale = Math.min(s, _xscaleMax);
	}
	public float yscale ()
	{
		return _yscale;
	}
	protected void yscale (float s)
	{
		_yscale = Math.min(s, _yscaleMax);
	}
	/**
	 * Zoom about center of visible area.
	 */
	public void zoom (float cx, float cy)
	{
		if(_zoomIn)
			denormZoom((float)1.5, (float)1.5, cx, cy);

		if(_zoomOut)
			denormZoom(1/MAG_STEP, 1/MAG_STEP, cx, cy);

		redraw();
	}
	/**
	 * Zoom and center on (cx, cy), in normalized coordinates.
	 */
	public void zoom (float zx, float zy, float cx, float cy)
	{
		denormZoom(zx, zy, fdenormX(cx), fdenormY(cy));
	}
	public void zoomIn ()
	{
		if(_zoomIn)
			fCanvas.setCursor(_zoomInCursor);			
		else	
			fCanvas.setCursor(_defaultCursor);
	}
	/**
	 * Zoom scrollpane and center on (cx, cy), in pre-zoom canvas coordinates.
	 */
	protected void zoomJCanvasScroller (
		float zx, float zy, float cx, float cy
	)
	{           
		_origin.x = Math.round((_origin.x + cx)*zx - cx);
		_origin.y = Math.round((_origin.y + cy)*zy - cy);

	}
	public void zoomOut ()
	{
		if(_zoomOut)
			fCanvas.setCursor(_zoomOutCursor);
		else	
			fCanvas.setCursor(_defaultCursor);
	}
	
	public void setDefautCursor()
	{
		fCanvas.setCursor(_defaultCursor);
	}
	
	/**
	 * Zoom to fill rectangular area (x, y, w, h), in normalized coordinates.
	 */
	public void zoomRect (float x, float y, float w, float h)
	{
		Rectangle visible = getVisible();

		float dw = w * xscale();
		float dh = h * yscale();

//        float factor = (visible.width * dh > visible.height * dw) ?
//            (float) visible.height / dh : (float) visible.width / dw;
// wdp : introduced separate factorx and factory to allow for zoom with diff x and y magnification

		float factorx = (float) visible.width / dw;
		float factory = (float) visible.height / dh;

		float cx = fdenormX(x + w/2f);
		float cy = fdenormY(y + h/2f);
		float dx = cx - visible.width/2f;
		float dy = cy - visible.height/2f;

		cx += dx/factorx;
		cy += dy/factory;

		denormZoom(factorx, factory, cx, cy);

	}
	public void zoomToFill (float xScaleMax, float yScaleMax)
	{       
		if(_objDS == null || fCanvas == null)
		  return;
		  
		float pixwidth = getVisible().width;
		float pixheight = getVisible().height;
		 
		if(pixwidth <= 0)  pixwidth=500;
		if(pixheight <= 0) pixheight=250;

		float objDS_width = _objDS.width();
		objDS_width = (objDS_width==0)?1:objDS_width;
		
		float objDS_height = _objDS.height();
		objDS_height = (objDS_height==0)?1:objDS_height;

		if (objDS_width > 1f && pixwidth/objDS_width < xScaleMax) {
			xscale(pixwidth/objDS_width);
		} else {
			xscale(xScaleMax);
		}
		
		if (objDS_height > 1f && pixheight/objDS_height < yScaleMax) {
			yscale(pixheight/objDS_height);
		} else {
			yscale(yScaleMax);
		}

		_origin.x=0;
		_origin.y=0;

		updateScrollbars();

	}
	public void zoomToFit ()
	{
		zoomToFill (1f, 1f);
	}
	
	public void dispose()
	{
		if(_zoomInCursor != null && !_zoomInCursor.isDisposed())
		   _zoomInCursor.dispose();

		if(_zoomOutCursor != null && !_zoomOutCursor.isDisposed())
		   _zoomOutCursor.dispose();
	
		if(_defaultCursor != null && !_defaultCursor.isDisposed())
		   _defaultCursor.dispose();
		   
		_objDS = null;
		_prevSize = null;
		_origin = null;
		fCanvas = null;
		_zoomInCursor = null;  
		_zoomOutCursor = null;  		
		_defaultCursor = null;  		    
	}
	
    public void isZoomIn(boolean zoom)
    {
    	_zoomIn = zoom;
    	if(zoom)
    		_zoomOut = false;
    }
    
    public void isZoomOut(boolean zoom)
    {
    	_zoomOut = zoom;
    	if(zoom)
    	  _zoomIn = false;
    }		
    
    public boolean isZoom()
    {
        return (_zoomIn || _zoomOut);	
    }
}
