/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProbekitUICompileAction.java,v 1.15 2005/02/16 22:21:28 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.probekit.ui.popup.actions;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

import java.util.*;
import java.io.*;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.probekit.CompilerFactory;
import org.eclipse.hyades.probekit.IProbeCompiler;
import org.eclipse.hyades.probekit.ProbekitCompileProblemException;
import org.eclipse.hyades.probekit.ui.IProbeCompileActionDelegateFactory;
import org.eclipse.hyades.probekit.ui.ProbekitUI;

/**
 * This class is a wrapper for the true IObjectActionDelegate.
 * When you construct an instance, this class will check for
 * plugins that extend a specific extension point. If there is one,
 * then it will get a "factory" from the extension point, and
 * call its "create" method to get the true IObjectActionDelegate to use.
 * 
 * If there are no extenders, then a default IObjectActionDelegate
 * is used -- an instance of DefaultProbeCompileObjectActionDelegate.
 * 
 * Eclipse only constructs one of these per action, so the same
 * instance of the IObjectActionDelegate (whichever we choose) gets
 * used for all invocations of the Compile action.
 * 
 * There should only be one extender of this extension point:
 * this replacement system picks the first one it sees.
 * 
 * The id of the extension point is in extensionPointID:
 * 		org.eclipse.hyades.probekit.ui.probekitCompileAction
 * 
 */
public class ProbekitUICompileAction implements IObjectActionDelegate {
	/**
	 * Return the default action delegate. This is the one that
	 * fires when nobody has extended the extension point to
	 * supply one.
	 *
	 * Any extender is free to call this function to get the
	 * default ActionDelegate, and then they can call its methods
	 * in addition to doing whatever it is they wanted
	 * to extend.
	 */
	public static IObjectActionDelegate getDefaultActionDelegate() {
		return new DefaultProbeCompileObjectActionDelegate();
	}
	
	//----------------------------------------------
	// Subsystem for finding out who uses our extension point,
	// and for getting an instance of that class.
	//
	
	/**
	 * The extension point ID string
	 */
	public final static String extensionPointID = "probekitCompileAction";
	
	/**
	 * The extender instance. We will call this instance's "create"
	 * method when we want a new instance, and we will delegate all
	 * calls to our IObjectActionDelegate to the resulting object.
	 */
	static boolean hasCheckedForExtender = false;
	static IProbeCompileActionDelegateFactory extenderFactory = null;
	IObjectActionDelegate delegate;
	
	/**
	 * Return a new instance of the proper action delegate to use.
	 * If there is an extender, call the extender's create() method.
	 * If we haven't checked for an extender yet, check; then call its 
	 * create() method if there was one.
	 * 
	 * If we've checked and there isn't any extender, return an
	 * instance of DefaultProbeCompilerActionDelegate.
	 * @return the IObjectActionDelegate instance to defer to
	 */
	IObjectActionDelegate getActionDelegate() {
		if (!hasCheckedForExtender) {
			// We haven't checked yet. Record the fact that we have
			// checked, then check.
			hasCheckedForExtender = true;
	
			// Check for an extender of our extension point.
			String idString = ProbekitUI.getDefault().getBundle().getSymbolicName() + "." + extensionPointID; //$NON-NLS-1$
			IExtensionPoint extensionPoint = Platform.getExtensionRegistry().getExtensionPoint(idString);
			if (extensionPoint != null) {
				IExtension[] extensions = extensionPoint.getExtensions();
				for (int i = 0; i < extensions.length; i++) {
					IExtension e = extensions[i];
					IConfigurationElement[] configElements = e.getConfigurationElements();
					boolean anyCompilerElements = false;
					for (int j = 0; j < configElements.length; j++) {
						IConfigurationElement ce = configElements[j];
						if (ce.getName().equals("compileActionFactory")) { //$NON-NLS-1$
							anyCompilerElements = true;
							Object extensionClass;
							try {
								extensionClass = ce.createExecutableExtension("class"); //$NON-NLS-1$
							}
							catch (CoreException ex) {
								// The workbench automatically records an error in this case,
								// so we don't have to report another one and stop looking.
								extensionClass = null;
								// fall through, keep looking
							}
							if (extensionClass != null) {
								if (extensionClass instanceof IProbeCompileActionDelegateFactory) {
									// We got one!
									extenderFactory = (IProbeCompileActionDelegateFactory)extensionClass; 
									break;
								}
								else {
									// This plug-in didn't follow the rules.
									// Log an error to the ProbekitUI error log.
									// But only once per session!
									IStatus status = new Status(IStatus.ERROR, 
											"org.eclipse.hyades.probekit", 0,  //$NON-NLS-1$
											ProbekitUI.getResourceString("UIFactory.ExtenderError") +  //$NON-NLS-1$
											getPrintableExtenderName(e) + 
												ProbekitUI.getResourceString("UIFactory.DoesNotImplementIObjectActionDelegate"), //$NON-NLS-1$
											null);
									ProbekitUI.getDefault().getLog().log(status);
								}
							}
							// Else extensionClass is null.
							// Is that even possible, or would createExecutableExtension have thrown?
						}
						// else this configurationElement name isn't "compiler," so ignore it.
					} // end for each configuration element
					if (extenderFactory != null) {
						// We found one in the inner loop, so break from the outer.
						break;
					}
					
					// If we get here it's because we failed: this plugin claims to extend
					// our extension point, but it doens't really. Report an error to the
					// error log, and keep looking. (Other types of errors, that the
					// class can't be instantiated or doesn't implement the required interface, 
					// was reported elsewhere.
					if (anyCompilerElements == false) {
						IStatus status = new Status(IStatus.ERROR, 
								"org.eclipse.hyades.probekit", 0,  //$NON-NLS-1$
								ProbekitUI.getResourceString("UIFactory.ExtenderError") +  //$NON-NLS-1$
									getPrintableExtenderName(e) + 
									ProbekitUI.getResourceString("UIFactory.NoFactoryElements"), //$NON-NLS-1$
								null);
						ProbekitUI.getDefault().getLog().log(status);
					}
				} // end for each extension
			}
		}
		
		// Now we know we've checked. 
		// Return the extender's create() method result,
		// or (if no extender) return "this."
		if (extenderFactory != null)
			return extenderFactory.create();
		else
			return getDefaultActionDelegate();
	}

	/**
	 * Returns a printable name for the plug-in an extension is in:
	 * its label in quotes, plus its id in parens, like this:
	 * <P>
	 * "My probe compiler plug-in" (com.sample.mycompiler)
	 * @param e the extension whose name you want to format
	 * @return a string in the above format that identifies the plug-in
	 */
	private String getPrintableExtenderName(IExtension e) {
		String extenderLabel = e.getLabel();
		String extenderId = e.getUniqueIdentifier();
		String extenderName = "\"" + extenderLabel + "\" (" + extenderId + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		return extenderName;
	}

	/**
	 * Constructor for Action1.
	 */
	public ProbekitUICompileAction() {
		super();
		delegate = getActionDelegate();
	}

	/**
	 * @see IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		delegate.setActivePart(action, targetPart);
	}

	/**
	 * Compile the selected probe(s) into Java source; compile the Java source into class files;
	 * collect the class files into a jar file; write the jar file and the engine script to the
	 * user's temporary directory, with names Probekit.jar and Probekit.txt.
	 * 
	 */
	public void run(IAction action) {
		delegate.run(action);
	}
	
	/**
	 * @see IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	
	public void selectionChanged(IAction action, ISelection selection) {
		delegate.selectionChanged(action, selection);
	}

}

class DefaultProbeCompileObjectActionDelegate implements IObjectActionDelegate {
	// No need to do anything in our constructor, so let it default.
	
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		// Nothing to do for setActivePart			
	}
	public void run(IAction action) {
		// Progress indicator meanings:
		//		Zero ticks: initializing
		//		One tick: just before compiling probes
		//		Two ticks: probes compiled
		//		Three ticks: saved java source
		//		Four ticks: saved engine script
		//		Five ticks: finished (same as above, really)

		Shell shell = new Shell();
		try {
			IRunnableWithProgress op = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor)
					throws InvocationTargetException, InterruptedException 
				{
					try {
						runWithProgress(monitor);
					}
					catch (Throwable e) {
						throw new InvocationTargetException(e);
					}
				}
			};
			new ProgressMonitorDialog(shell).run(true, /*cancelable =*/ false, op);
		 } catch (InvocationTargetException e) {
			// Handle exceptions by throwing something the workbench will catch
		 	// and add to the error log.
		 	// But ignore ProbekitCompileProblemException.
		 	if (e.getTargetException() instanceof ProbekitCompileProblemException) {
		 		// do nothing: compile problems were reported as markers.
		 	}
		 	else {
		 		throw new Error(e);
		 	}
		 } catch (InterruptedException e) {
			// handle cancelation - nothing to do.
		 }
	}
	public void selectionChanged(IAction action, ISelection selection) {
		// This is only called when the selection matches our filter pattern
		// from the plugin.xml file, so we know we should be enabled.
		action.setEnabled(true);

		// The Compile menu item defaults to being a toggle, checked/unchecked.
		// Make it unchecked always.
		action.setChecked(false);

		// Clear out the selection list from last time and populate it
		// with all the *.probe files from this time.
		selectedFileList = new LinkedList();
		if (selection != null && selection instanceof IStructuredSelection) {
			for (Iterator i = ((IStructuredSelection)selection).iterator(); i.hasNext(); ) {
				Object o = i.next();
				if (o instanceof IFile) {
					IFile f = (IFile)o;
					if (f.getName().endsWith(".probe")) //$NON-NLS-1$
						selectedFileList.add(f);
				}
			}
		}
	}
	private void reportError(String msg, IResource ires) {
		try {
			IMarker m = ires.createMarker(IProbeCompiler.PROBEKIT_PROBLEM_MARKER);
			m.setAttribute(IMarker.MESSAGE, msg);
			m.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
		} catch (CoreException e) {
			// Report this failure to the error log
			IStatus status = new Status(IStatus.ERROR, 
					"com.ibm.rational.pd.probekit", 0,  //$NON-NLS-1$
					ProbekitUI.getResourceString("UI.InternalError"), //$NON-NLS-1$
					null);
			ProbekitUI.getDefault().getLog().log(status);
		}
	}

	/**
	 * Remove all markers of the Probekit problem type from the indicated resource.
	 * Called when we add something to the compiler.
	 * @param ires the resource for which to remove all Probekit markers.
	 */
	private void removeProbekitMarkers(IResource ires) {
		try {
			ires.deleteMarkers(IProbeCompiler.PROBEKIT_PROBLEM_MARKER, true, IResource.DEPTH_ZERO);
		} catch (CoreException e) {
			// Well, we tried...
		}
	}

	/**
	 * This is is main operation function. It takes a progress monitor
	 * so we can inform the user about long-running operations.
	 * Prior to running this, selectionChanged() should have filled in
	 * the selectedFileList.
	 * 
	 * @param monitor the monitor 
	 * @throws Throwable
	 */
	public void runWithProgress(IProgressMonitor monitor) throws Throwable {
		monitor.beginTask(ProbekitUI.getResourceString("UI.Compiling"), 3); //$NON-NLS-1$
		IFile firstFile = null;
		try {			
			if (selectedFileList == null || selectedFileList.size() == 0) {
				// This should not happen - you souldn't get here if you didn't
				// select at least one probe source file
				return;
			}
	
			// Choose the base name from the first input file
			firstFile = (IFile)selectedFileList.get(0);
			removeProbekitMarkers(firstFile);
			String basename = getBaseName(firstFile.getName());
	
			// Create a compiler, add all selected files to it, compile,
			// save the Java output and the probescript output.
			IProbeCompiler c = CompilerFactory.INSTANCE.createCompiler();
			
			// Pass the basename to the compiler as the class prefix, but
			// cleanse it of characters that aren't allowed in Java identifiers.
			// (Eclipse bugzilla #59675.)
			basename = c.makeValidJavaIdentifier(basename);
			c.setClassPrefix(basename);
	
			// Add the IFiles from the selected file list to the compiler.
			for (Iterator i = selectedFileList.iterator(); i.hasNext() ; ) {
				IFile f = (IFile)i.next();
				removeProbekitMarkers(f);
				c.addIFile(f);
			}
	
			monitor.subTask(ProbekitUI.getResourceString("UI.Generating")); //$NON-NLS-1$
			monitor.worked(1);
			
			// call getGeneratedSource now, so any exceptions 
			// occur before we create output files.
			String generatedSource = c.getGeneratedSource();
	
			monitor.subTask(ProbekitUI.getResourceString("UI.SavingJava")); //$NON-NLS-1$
			monitor.worked(1);
			
			// Put the generated files in the directory that the (first/only) 
			// probe file was in. When we write the *.java file, it will 
			// automagically be compiled. 
			// Eclipse bugzilla #59684: use getClassSuffix
			String suffix = c.getClassSuffix();
			IFile javaSourceIFile = firstFile.getParent().getFile(new Path(basename + suffix + ".java")); //$NON-NLS-1$
			ByteArrayInputStream newJavaContents = new ByteArrayInputStream(generatedSource.getBytes());
			// TODO: why do I have to test exists() and use setContents vs. create? Doesn't "force" handle that?
			if (javaSourceIFile.exists()) {
				javaSourceIFile.setContents(newJavaContents, true, false, monitor);
			}
			else {
				javaSourceIFile.create(newJavaContents, true, monitor);
			}
	
			monitor.subTask(ProbekitUI.getResourceString("UI.SavingEngineScript")); //$NON-NLS-1$
			monitor.worked(1);
			
			// Write the engine script to Probekit.txt in the project bin directory
			// This overwrites any existing Probekit.jar output file from before.
			// Note: the fix to Bugzilla 76716 is to use the UTF-8 arg to getBytes().
			String engineScript = c.getEngineScript();
			IFile engineScriptIFile = firstFile.getParent().getFile(new Path(basename + ".probescript")); //$NON-NLS-1$
			ByteArrayInputStream newScriptContents = new ByteArrayInputStream(engineScript.getBytes("UTF-8")); //$NON-NLS-1$
			// TODO: why do I have to test exists() and use setContents vs. create? Doesn't "force" handle that?
			if (engineScriptIFile.exists()) {
				engineScriptIFile.setContents(newScriptContents, true, false, monitor);
			}
			else {
				engineScriptIFile.create(newScriptContents, true, monitor);
			}
		}
		catch (ProbekitCompileProblemException pcpe) {
			// Don't report these exceptions: they've already
			// appeared in the Problems view.
		}
		catch (Exception e) {
			// Report other exceptions as a Problem on the selected file, and in the error log
			if (firstFile != null) {
				reportError(ProbekitUI.getResourceString("UI.InternalError"), firstFile); //$NON-NLS-1$
			}
			IStatus status = new Status(IStatus.ERROR, 
					"com.ibm.rational.pd.probekit", 0,  //$NON-NLS-1$
					ProbekitUI.getResourceString("UI.InternalError"), //$NON-NLS-1$
					e);
			ProbekitUI.getDefault().getLog().log(status);
			return;
		}
		finally {
			monitor.done();
		}
	}

	/**
	 * Remove the suffix (the last period character and everything after it)
	 * from the input string. Returns null for null; returns the original
	 * string if it has no dots.
	 * @param name the basename of the input string
	 * @return the basename of the input name. 
	 */
	private String getBaseName(String name) {
		if (name == null) return null;
		int lastdot = name.lastIndexOf("."); //$NON-NLS-1$
		if (lastdot == -1) return name;
		return name.substring(0, lastdot);
	}

	/**
	 * The list of selected files, maintained by selectionChanged() and used by run()
	 */
	List selectedFileList;

}
