/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestGenerator.java,v 1.6 2005/02/16 22:20:27 qiyanli Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.internal.execution.testgen;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.execution.testgen.TestgenPlugin;
import org.eclipse.hyades.internal.execution.testgen.ui.TestgenUIUtility;
import org.eclipse.hyades.internal.execution.testgen.ui.dialogs.TestGenProgressDialog;

/**
 * Object to be extended by objects which will generate tests in the Testgen infrastructure
 * This object is associated with extension point org.eclipse.hyades.execution.testgen.TestGenerator
 * @author Ernest Jessee
 */
public abstract class TestGenerator implements Runnable
{
	
	
	/**
	 * reference to the .recmodel file associated with the generated test
	 */
	protected IFile recModelFile = null;
	
	/**
	 * the output filename (ends with .testsuite)
	 */
	protected String outputFileName = null;	
	
	/**
	 * the progress dialog
	 */
	private TestGenProgressDialog dlg=null;
	
	/**
	 * testgen completed successfully
	 */
	private boolean bSuccess=false;
	/**
	 * method called to begin test generation
	 * @return
	 * @throws TestgenException
	 */
	public abstract boolean runTestGen() throws TestgenException;
	
	/**
	 * method called to determine whether or not a progress dialog should be shown with test generation.
	 * @return
	 */
	public abstract boolean showProgress();
	
	
	
	
	/**
	 * method called to initialize the Test Generator with needed information and references
	 * @param recModelFile
	 * @param outputFileName
	 */	
	public void initialize(IFile recModelFile, String outputFileName)
	{
		this.recModelFile = recModelFile;
		this.outputFileName = outputFileName;
		
	}

	
	/**
	 * this method is called by the display to perform the test generation.  Testgenerators are executed via Display.getDefault().asyncExec(Runnable);	 * 
	 */
	public void run()
	{
		if(showProgress())
		{	
			dlg = new TestGenProgressDialog();
		
			dlg.setBlockOnOpen(false);
			
			dlg.open();
		}	
		
		try
		{
			bSuccess = this.runTestGen(); 

			if (bSuccess) {
				//open the test suite
				IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(outputFileName));
				TestgenUIUtility.selectAndReveal(file, TestgenPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow());
				TestgenUIUtility.openEditor(file, null, false);
			}
		}
		catch (TestgenException e)
		{
			e.printStackTrace();
		}
	}
	
	/**
	 * called to set the percent complete of the progress dialog
	 * @param percentComplete
	 */
	public void setProgress(int percentComplete)
	{
		if(dlg!=null)
			dlg.setProgress(percentComplete);
	}
	
	
	/**
	 * called to change the displayed message on the progress dialog.
	 * These messages must be localized
	 * @param message
	 */
	public void setMessage(String message)
	{
		if(dlg!=null)
			dlg.setMessage(message);
	}
	
	/**
	 * called to close the progress dialog
	 *
	 */
	public void closeProgressDialog()
	{
		if(dlg!=null)
			dlg.close();
	}

	

	

	/**
	 * @return
	 */
	public boolean isSuccess() {
		return bSuccess;
	}

}