/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogUIPlugin.java,v 1.27 2005/04/25 21:31:32 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.log.ui.internal.util.IRecordPaneViewerListener;
import org.eclipse.hyades.log.ui.internal.util.LogUIConstants;
import org.eclipse.hyades.log.ui.internal.util.LogUtil;
import org.eclipse.hyades.log.ui.internal.util.RecordChangeEvent;
import org.eclipse.hyades.log.ui.internal.views.LogFiltersStandardUI;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.internal.sdb.impl.SDBFactoryImpl;
import org.eclipse.hyades.models.internal.sdb.impl.SDBPackageImpl;
import org.eclipse.hyades.models.internal.sdb.impl.SDBResourceSetImpl;
import org.eclipse.hyades.trace.ui.filters.internal.util.FilterQueries;
import org.eclipse.hyades.trace.ui.filters.internal.util.FilterResourceHandler;
import org.eclipse.hyades.trace.ui.internal.filters.ModelUtils;
import org.eclipse.hyades.ui.internal.util.StringUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

public class LogUIPlugin extends AbstractUIPlugin {

	private static LogUIPlugin inst;
	private static ResourceBundle fResourceBundle = null;
	private SDBFactoryImpl factory;
	private ResourceSet resourceSet;
	private ArrayList recordChangeListeners = null; 
	private IConfigurationElement newSymptomWizardElement = null;
	private boolean newSymptomWizardElementInit;
	public static final String PLUGIN_ID = "org.eclipse.hyades.log.ui"; //$NON-NLS-1$

	public LogUIPlugin() {

		if (inst == null)
			inst = this;
	}
	public static LogUIPlugin getDefault() {
		return inst;
	}

	public static String getPluginId() {
		return PLUGIN_ID;
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (2/16/2001 4:57:29 PM)
	 * @return com.ibm.jface.resource.ImageDescriptor
	 * @param name java.lang.String
	 */
	public static ImageDescriptor getImageDescriptor(String name) {
		return LogUIPluginImages.getImageDescriptor(name);
	}
	public static ResourceBundle getResourceBundle() {
		if(fResourceBundle == null)
		{
			try {
				fResourceBundle = Platform.getResourceBundle(Platform.getBundle(PLUGIN_ID));
			} catch (Exception e) {
				fResourceBundle = null;
			}
		}
		
		return fResourceBundle;
	}
	public static String getResourceString(String key) {
		ResourceBundle bundle = LogUIPlugin.getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}
	public static String getResourceString(String key, String[] variables) {
		String value = getResourceString(key);
		for(int i=variables.length; i>0; i--)
			value = StringUtil.replace(value, "%"+i, variables[i-1]);
		return value;
	}
	public static Shell getActiveWorkbenchShell() {
		return getActiveWorkbenchWindow().getShell();
	}
	public static IWorkbenchWindow getActiveWorkbenchWindow() {
		return getDefault().getWorkbench().getActiveWorkbenchWindow();
	}

	/**
	 * Logs an exception in Eclipse's .log file.
	 * @param e The exception to log.
	 */
	public static void log(Throwable e) {
		String msg = e.getMessage();
		getDefault().getLog().log(new Status(IStatus.ERROR, getPluginId(), IStatus.ERROR, msg == null ? "" : msg, e));
	}	

	/* (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		LogUIPluginImages.shutdown();
		
		savePluginPreferences();		
		fResourceBundle = null;
		
	}
	
	/* (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		SDBPackageImpl.init();

		factory = new SDBFactoryImpl();

		// init resourceSet
		resourceSet = SDBResourceSetImpl.getInstance();
		
		addDefaultFilters();
	}

	public SDBFactoryImpl getSymptomDBFactory() {
		if (factory == null)
			factory = new SDBFactoryImpl();
		return factory;
	}

	/*
	 * 
	 */
	public ResourceSet getResourceSet() {
		return resourceSet;
	}

	public static DateFormat getDateTimeFormat(){
		return DateFormat.getDateTimeInstance(getDateStyle(), getTimeStyle(), Locale.getDefault());	
	}

	public static int getDateStyle () {
		IPreferenceStore store = inst.getPreferenceStore();
		String dateFormat = store.getString(LogUIConstants.LOG_ANALYZER_FORMAT_DATE);

		boolean valid = false;
		for (int i = 0; i < LogUIConstants.DATE_FORMAT_LIST.length; i++)
			if (dateFormat.equals(LogUIConstants.DATE_FORMAT_LIST[i])) {
				valid = true;
				break;
			}
		if (!valid)
			dateFormat = LogUIConstants.LOG_ANALYZER_FORMAT_DATE_DEFAULT;

		if (dateFormat.equals(LogUIConstants.DATE_FORMAT_FULL))
			return DateFormat.FULL;
		else if (dateFormat.equals(LogUIConstants.DATE_FORMAT_LONG))
			return DateFormat.LONG;
		else if (dateFormat.equals(LogUIConstants.DATE_FORMAT_MEDIUM))
			return DateFormat.MEDIUM;
		else if (dateFormat.equals(LogUIConstants.DATE_FORMAT_SHORT))
			return DateFormat.SHORT;
		else
			return DateFormat.MEDIUM;
	}
	
	public static int getTimeStyle () {
		IPreferenceStore store = inst.getPreferenceStore();
		String timeFormat = store.getString(LogUIConstants.LOG_ANALYZER_FORMAT_TIME);

		boolean valid = false;
		for (int i = 0; i < LogUIConstants.TIME_FORMAT_LIST.length; i++)
			if (timeFormat.equals(LogUIConstants.TIME_FORMAT_LIST[i])) {
				valid = true;
				break;
			}
		if (!valid)
			timeFormat = LogUIConstants.LOG_ANALYZER_FORMAT_TIME_DEFAULT;

		if (timeFormat.equals(LogUIConstants.TIME_FORMAT_FULL))
			return DateFormat.FULL;
		else if (timeFormat.equals(LogUIConstants.TIME_FORMAT_LONG))
			return DateFormat.LONG;
		else if (timeFormat.equals(LogUIConstants.TIME_FORMAT_MEDIUM))
			return DateFormat.MEDIUM;
		else if (timeFormat.equals(LogUIConstants.TIME_FORMAT_SHORT))
			return DateFormat.SHORT;
		else
			return DateFormat.MEDIUM;
	}
	
	/**
	 * Sets default preference values. These values will be used
	 * until some preferences are actually set using the Preference dialog.
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {
		
		LogUtil.initializePreferenceStore(store);
	}

	
	
	public void addRecordPaneViewerListener(IRecordPaneViewerListener listener){
		
		if(recordChangeListeners==null){
			recordChangeListeners = new ArrayList();
		}
		
		if(!recordChangeListeners.contains(listener)){
			recordChangeListeners.add(listener);
		}
	}
	
	public void removeRecordPaneViewerListener(IRecordPaneViewerListener listener){
		if(recordChangeListeners!=null){
			recordChangeListeners.remove(listener);
		}
	}
	
	public void fireRecordChangeNotification(RecordChangeEvent e){
		if(recordChangeListeners!=null){
			for(int i=0;i<recordChangeListeners.size();i++){
				((IRecordPaneViewerListener)recordChangeListeners.get(i)).handleRecordChangeEvent(e);
			}
		}
		
	}
	
	public INewWizard getNewSymDBWizard() {
		try {
			if (!newSymptomWizardElementInit) {
				newSymptomWizardElementInit = true;
				loadWizardExtention();

			}

			if (newSymptomWizardElement != null)
				return (INewWizard) newSymptomWizardElement
						.createExecutableExtension("class");
		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}

	public void loadWizardExtention() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IConfigurationElement[] configurationElements = registry.getConfigurationElementsFor("org.eclipse.ui.newWizards");//$NON-NLS-1$		
		if (configurationElements!=null) {
			for (int i = 0; i < configurationElements.length; i++) {
				IConfigurationElement configurationElement = configurationElements[i];
				if (configurationElement.getName().equals("wizard") //$NON-NLS-1$
					&& configurationElement.getAttribute("id").endsWith("wizard.new.symptomdb")) {//$NON-NLS-1$
					newSymptomWizardElement = configurationElement;
				}
			}
		}
	}
	
	private void addDefaultFilters()
	{
		String logUIPluginDefaultFiltersKey = "org.eclipse.hyades.log.ui";
		
		if (!FilterResourceHandler.isDefaultFiltersAddedForKey(logUIPluginDefaultFiltersKey))
		{
			List filtersToAdd = new ArrayList();
			
			FilterQueries fq = FilterResourceHandler.createFilterQueries("org.eclipse.hyades.log.ui.LogFilterType", getResourceString("STR_ST_DEFAULT_LOG_FILTER_SHOW_ERRORS"), null, null);
			SimpleSearchQuery query = fq.standard();
			addSeverityBetweenArguement(query, LogFiltersStandardUI.VALUES_SEVERITY_ERROR);
			filtersToAdd.add(fq);
			
			fq = FilterResourceHandler.createFilterQueries("org.eclipse.hyades.log.ui.LogFilterType", getResourceString("STR_ST_DEFAULT_LOG_FILTER_SHOW_WARNINGS"), null, null);
			query = fq.standard();
			addSeverityBetweenArguement(query, LogFiltersStandardUI.VALUES_SEVERITY_WARNING);			
			filtersToAdd.add(fq);			

			fq = FilterResourceHandler.createFilterQueries("org.eclipse.hyades.log.ui.LogFilterType", getResourceString("STR_ST_DEFAULT_LOG_FILTER_SHOW_INFORMATION"), null, null);
			query = fq.standard();
			addSeverityBetweenArguement(query, LogFiltersStandardUI.VALUES_SEVERITY_INFORMATION);
			filtersToAdd.add(fq);			

			fq = FilterResourceHandler.createFilterQueries("org.eclipse.hyades.log.ui.LogFilterType", getResourceString("STR_ST_DEFAULT_LOG_FILTER_SHOW_CORRELATED"), null, null);
			fq.standard(ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery());
			filtersToAdd.add(fq);			
			
			FilterResourceHandler.addFiltersOnInstanceLoad(filtersToAdd);
			FilterResourceHandler.defaultFiltersAddedForKey(logUIPluginDefaultFiltersKey, true);				
		}
	}
	
	private void addSeverityBetweenArguement(SimpleSearchQuery query, String[] betweenValues)
	{
		LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		logicalExpression.setOperator(LogicalOperators.AND_LITERAL);

		LogicalExpression severityLogicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		severityLogicalExpression.setOperator(LogicalOperators.OR_LITERAL);
		EList arguments = severityLogicalExpression.getArguments();
		arguments.add(ModelUtils.createBinaryExpression(LogFiltersStandardUI.ATTRIBUTE_SEVERITY_CONTROL, LogFiltersStandardUI.OPERATOR_BETWEEN, betweenValues, false));
		
		logicalExpression.getArguments().add(severityLogicalExpression);	
		query.setWhereExpression(logicalExpression);
	}
}
