/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.log.ui.internal.util;


import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.WrappedException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.views.ImportLogFiltersStandardUI;
import org.eclipse.hyades.logging.adapter.model.internal.adapter.AdapterType;
import org.eclipse.hyades.logging.adapter.model.internal.adapter.DocumentRoot;
import org.eclipse.hyades.logging.adapter.model.internal.configuration.ContextInstanceType;
import org.eclipse.hyades.logging.adapter.model.internal.context.Component;
import org.eclipse.hyades.logging.adapter.model.internal.context.ContextFactory;
import org.eclipse.hyades.logging.adapter.model.internal.context.ContextType;
import org.eclipse.hyades.logging.adapter.model.internal.context.ContextsType;
import org.eclipse.hyades.logging.adapter.model.internal.context.RoleNames;
import org.eclipse.hyades.logging.adapter.model.internal.filter.AbstractFilterRuleType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.FilterConfigType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.FilterFactory;
import org.eclipse.hyades.logging.adapter.model.internal.filter.FilterRuleBlockType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.LogicalOperatorType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.NumericalFilterRuleType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.NumericalRelationalOperatorType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.PathExpressionType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.RangeFilterRuleType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.RangeRelationalOperatorType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.StringFilterRuleType;
import org.eclipse.hyades.logging.adapter.model.internal.filter.StringRelationalOperatorType;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.ui.internal.util.StringUtil;

/**
 * @author apnan - Created March 11, 2005
 *
 */
public class GLAFilterHelper {
	
	public static final String GLALogVersionTagName = "version";
	public static final String GLADefaultLogVersionTagName = "Default";
	public static final String GLAConfigPathTagName = "config_path";
	public static final String GLAConfigFileRootAttributeName = "ConfigFileRoot";
	private static final String WindowsAdapterRootDir = "C:\\";
	private static final String UnixAdapterRootDir = "/tmp/";
	private static final String GLARangeFilterAttributeNamePathRoot = "RangeFilter";
	private static final String GLARangeFilterByEventsAttributeName = "events";
	private static final String GLARangeFilterBySecondsAttributeName = "seconds";
	private static final String dateTimeFormat = "yyyy-MM-dd'T'HH:mm:ss.S Z";

	public static FilterConfigType createGLAFilter(SimpleSearchQuery query) {
		FilterConfigType filter = null;
		if (query.getWhereExpression() != null){
			WhereExpression expression = query.getWhereExpression();
			EObject filterRule = createFilterElement(expression);
			if(filterRule==null)
				return null;
			if(filterRule instanceof AbstractFilterRuleType){
				filter = FilterFactory.eINSTANCE.createFilterConfigType();
				filter.setAbstractFilterRule((AbstractFilterRuleType)filterRule);
			}else if(filterRule instanceof FilterRuleBlockType){
				filter = FilterFactory.eINSTANCE.createFilterConfigType();
				filter.setFilterRuleBlock((FilterRuleBlockType)filterRule);
			}			
		}
		return filter;
	}
	
	private static FilterRuleBlockType createFilterBlock(LogicalExpression logicalExpression){
		FilterRuleBlockType parentBlock = FilterFactory.eINSTANCE
		.createFilterRuleBlockType();
		parentBlock.setName(logicalExpression.getName()!=null?logicalExpression.getName():"filterBlock");
		parentBlock.setOperator(LogicalOperatorType.get(logicalExpression
				.getOperator().getName()));
		EList filterElements = logicalExpression.getArguments();

		for (int i = 0; i < filterElements.size(); i++) {
			Object element = filterElements.get(i);	
			if(element instanceof BinaryExpression){
					BinaryExpression binaryExpression = (BinaryExpression) filterElements
							.get(i);
					EObject filterRule = createFilterRule(binaryExpression);
					if(filterRule!=null){
						if(filterRule instanceof AbstractFilterRuleType){
							parentBlock.getAbstractFilterRule().add(filterRule);
						}else if(filterRule instanceof FilterRuleBlockType){
							parentBlock.getFilterRuleBlock().add(filterRule);
						}						
					}
			}else if(element instanceof LogicalExpression){
				EObject filterRule = createFilterElement((LogicalExpression)element);
				if(filterRule!=null){

				}
			}
		}
		return parentBlock;
	}

	protected static AdapterType getAdapter(Resource resource, 
			SimpleSearchQuery query) {
		
		if (resource.getContents().size() > 0) {
			DocumentRoot rootObject = (DocumentRoot) resource.getContents().get(0);
			Object object = rootObject.getAdapter();
			if (object instanceof AdapterType) {
				FilterConfigType filter = createGLAFilter(query);
				if(filter==null){
					return null;
				}
				AdapterType adapter = (AdapterType)object;
				List contextInstance = adapter.getConfiguration().getContextInstance();
				ContextInstanceType contextType = null;
				if(contextInstance!=null && contextInstance.size()>0){
					Object element = contextInstance.get(0);
					if(element!=null && element instanceof ContextInstanceType){
						contextType = (ContextInstanceType)element;
						contextType.getFilter().add(filter);
					}
				}
				ContextsType contextsType = adapter.getContexts();
				if(contextsType!=null){
					List contexts = contextsType.getContext();
					if(contexts!=null && contexts.size()>0){
						ContextType context = (ContextType)contexts.get(0);
						filter.setDescription("CBE Filter Component");						
						context.getComponent().add(context.getComponent().size()-1,createFilterComponent(filter));						
					}
				}
				return adapter;
			}
		}		
		return null;
	}
	
	public static String addFilterToAdapter(Node node, ResourceSet resourceSet, Map table, SimpleSearchQuery query, String nameSpace) throws Exception{
		String adapterFilePath = getLocalAdapterPath(table, nameSpace);
		Resource resource  = null;
		try{
			resource = resourceSet.getResource(URI.createFileURI(adapterFilePath),
				true);
		}catch(Exception  e1){
			if(e1 instanceof WrappedException && ((WrappedException)e1).exception() instanceof FileNotFoundException){
				String message = getMessage("No_Config_File_ERROR_", new String[]{adapterFilePath});
				throw new Exception(message);				
			}else{
				 throw e1;
			}
		}
		AdapterType adapter = getAdapter(resource, query);
		if(adapter==null){
			return "";
		}
		File destination_file = null;
		String tempFilePath = "";
		try{	
			//IPath dirPath = Platform.getPlugin("org.eclipse.hyades.logging.adapter.config").getStateLocation();
			IPath dirPath = LogUIPlugin.getDefault().getStateLocation(); 
			String destPath = dirPath.toString();
			destPath = StringUtil.replace(destPath, "org.eclipse.hyades.log.ui","org.eclipse.hyades.logging.adapter.config");
			
			File destinationDir = new File(destPath);
			try {				
				destination_file = File.createTempFile("GLA", ".adapter",
						destinationDir);
			} catch (Exception e) {
				String tmpFileName = new String("GLA" + System.currentTimeMillis()
						+ ".adapter");
				destination_file = new File(destinationDir, tmpFileName);
			}
			resource.setURI(URI.createFileURI(destination_file.getAbsolutePath()));
			resource.save(Collections.EMPTY_MAP);
			String key = getAdapterPathKey(table);
			if(key!=null && key.trim().length()>0){	
				table.put(key,destination_file.getName());
				table.put(GLAConfigPathTagName,destination_file.getParent());
			}
			destination_file.deleteOnExit();
			tempFilePath = destination_file.getAbsolutePath();
			if(node!=null){				
				tempFilePath = sendAdapter(node, table, destination_file.getAbsolutePath(), destination_file.getName());
//				table.put("config_path",targetDir);
//				table.put(GLAConfigFileRootAttributeName,targetDir);
			}
		}catch(Exception e){
			
			try {		
				destination_file = File.createTempFile("GLA"+System.currentTimeMillis(), ".adapter");
				resource.setURI(URI.createFileURI(destination_file.getAbsolutePath()));
				resource.save(Collections.EMPTY_MAP);				
				String key = getAdapterPathKey(table);			
				if(key!=null && key.trim().length()>0){	
					table.put(key,destination_file.getName());
					table.put(GLAConfigFileRootAttributeName,destination_file.getParent());
				}
				destination_file.deleteOnExit();
				tempFilePath = destination_file.getAbsolutePath();
				if(node!=null){	
					tempFilePath = sendAdapter(node, table, destination_file.getAbsolutePath(),destination_file.getName());
//					table.put(GLAConfigFileRootAttributeName,targetDir);
				}
				
			}
			catch (Exception e2){
				// If a temporary file cannot be generated in the OS tmp directory
				// try creating it in the same directory as the current config file.				
				File oldFile = new File(adapterFilePath);
				String tmpFileName = new String("GLA" + System.currentTimeMillis() + ".adapter");				
				destination_file = new File(oldFile.getParent(), tmpFileName);
				resource.setURI(URI.createFileURI(destination_file.getAbsolutePath()));
				resource.save(Collections.EMPTY_MAP);
				String key = getAdapterPathKey(table);			
				if(key!=null && key.trim().length()>0){	
					table.put(key,destination_file.getName());
					table.put(GLAConfigFileRootAttributeName,destination_file.getParent());
				}
				tempFilePath = destination_file.getAbsolutePath();
				if(node!=null){	
					tempFilePath = sendAdapter(node, table, destination_file.getAbsolutePath(), destination_file.getName());
//					table.put(GLAConfigFileRootAttributeName,targetDir);
				}
			}
		}
		resource.unload();
		return tempFilePath;
	}
	
	private static EObject createFilterElement(WhereExpression expression){
		
		if(expression instanceof LogicalExpression){
			LogicalExpression logicalExpression = (LogicalExpression)expression;
			int s = logicalExpression.getArguments().size();
			if(s==0){
				return null;
			}
			if(logicalExpression.getArguments().size()==1){
				Object argument = logicalExpression.getArguments().get(0);
				if(argument instanceof BinaryExpression){					
					EObject filterRule = createFilterRule((BinaryExpression)argument);
					if(logicalExpression.getOperator().getValue()==LogicalOperators.NOT){
						if(filterRule instanceof AbstractFilterRuleType){
							((AbstractFilterRuleType)filterRule).setNegation(true);
						}else if(filterRule instanceof FilterRuleBlockType){
							((FilterRuleBlockType)filterRule).setNegation(true);
						}
					}
					return filterRule;
				}else if(argument instanceof LogicalExpression){
					return createFilterElement((LogicalExpression) argument);
				}
			}else{
				WhereExpression argument = null;				
				Object element = null;
				EObject rule = null;
				List rules = new ArrayList();
				for(int i=0;i<s;i++){
					element = logicalExpression.getArguments().get(i);
					if(!(element instanceof WhereExpression)){
						continue;
					}
					argument = (WhereExpression)element;
					if(argument!=null){
						if(argument instanceof BinaryExpression){
							rule = createFilterRule((BinaryExpression)argument);
						}else if(argument instanceof LogicalExpression){
							rule = createFilterElement((LogicalExpression)argument);
						}
					}
					if(rule!=null){
						rules.add(rule);
					}					
				}
				int sr = rules.size();
				if(sr>1){
					FilterRuleBlockType parentBlock = FilterFactory.eINSTANCE
					.createFilterRuleBlockType();
					parentBlock.setName(logicalExpression.getName()!=null?logicalExpression.getName():"filterBlock");
					parentBlock.setOperator(LogicalOperatorType.get(logicalExpression
							.getOperator().getName()));
					for(int i=0;i<sr;i++){
						if(rules.get(i) instanceof AbstractFilterRuleType){
							parentBlock.getAbstractFilterRule().add(rules.get(i));
						}else if(rules.get(i) instanceof FilterRuleBlockType){
							parentBlock.getFilterRuleBlock().add(rules.get(i));
						}
					}
					return parentBlock;
				}else if(sr==1){
					if(rules.get(0)!=null && rules.get(0) instanceof EObject){
						return (EObject)rules.get(0);						
					}

				}
			}
		}else if(expression instanceof BinaryExpression){
			return createFilterRule((BinaryExpression)expression);
		}
		return null;
	}
	
	private static EObject createFilterRule(BinaryExpression binaryExpression){
		
		EStructuralFeature feature = ((SimpleOperand)binaryExpression.getLeftOperand()).getFeature();
		Object attributeValue =  (((SimpleOperand) binaryExpression
				.getRightOperands().get(0)).getValue());
		String value = attributeValue.toString();
		if(binaryExpression.getName()!=null && binaryExpression.getName().equals(ImportLogFiltersStandardUI.RANGE_BY_EVENTCOUNT_OR_SECONDSCOUNT_BINARY_EXPR_NAME)){
			return createRangeFilterRuleType(binaryExpression);
		}else
		if(feature.getEType()== EcorePackage.eINSTANCE.getEString()){			
			 return createStringFilterRuleType(binaryExpression);
		}else if(feature == CBEPackage.eINSTANCE.getCBECommonBaseEvent_CreationTime() ||
				feature == CBEPackage.eINSTANCE.getCBECommonBaseEvent_ElapsedTime()){
			return createTimeFilterRuleType(binaryExpression);
		}else if(attributeValue instanceof Number){
			return createNumericFilterRule(binaryExpression);
		}
		return null;
	}
	
	
	public static String getLocalAdapterPath(Map table, String nameSpace) throws Exception{

		String key = getAdapterPathKey(table);
		String config_file = (String)table.get(key);
		if(config_file==null){
			//ResourceBundle resourceBundle = Platform.getResourceBundle(Platform.getBundle("org.eclipse.hyades.logging.adapter"));
			//throw new Exception(resourceBundle.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_"));
			String message = getMessage("No_Config_File_ERROR_", new String[]{(String)table.get(GLALogVersionTagName)});
			throw new Exception(message);
		}
		//get the adapter config file path
		String adapterFilePath = "";
		try {
			String location = Platform.getBundle(nameSpace).getLocation();
			int i = location.indexOf("/");
			if(i>-1){
				location = location.substring(i+1);
				if(config_file.indexOf("./")>-1){
					config_file = config_file.substring(2);
					adapterFilePath = location + config_file;
				}else{
					adapterFilePath = location + config_file;
				}

				File test = new File(adapterFilePath);
				if(!test.exists()){
					adapterFilePath = location + "config/"+config_file;
				}
			}			
		}
		catch	(Exception e) {
//			String fileSeparator = System.getProperty("file.separator");
//			
//			
//			// If we have the wrong file separator in the config file path then
//			if (config_file.indexOf('/') != -1 && fileSeparator != "/") {
//				// Replace file separators
//				config_file = config_file.replace('/', fileSeparator.charAt(0));
//			}
			// Remove ./
			if (config_file.startsWith(".")) {
				config_file = config_file.substring(2);
			}
			adapterFilePath = (String) table.get(GLAConfigFileRootAttributeName) + "/" + config_file;
			File test = new File(adapterFilePath);
			if(!test.exists()){
				adapterFilePath = (String) table.get(GLAConfigFileRootAttributeName) + "config/"+config_file;
			}			
		}
		
		return adapterFilePath;
	}
	
	
	protected static String getAdapterPathKey(Map table){
		String key = "";
		//ResourceBundle resourceBundle = Platform.getResourceBundle(Platform.getBundle("org.eclipse.hyades.logging.adapter"));
		// Resolve the version number of the log in order to load the correct config file
		//String version = (String) (table.get(resourceBundle.getString("HyadesGALogVersionTagName")));
		String version = (String) (table.get(GLALogVersionTagName));
		String config_file = null;
		
		// If no version attribute given, use default
		if(version == null)
		{
			//version = resourceBundle.getString("HyadesGADefaultLogVersionTagName");
			version = GLADefaultLogVersionTagName;
		}

		// Get the config file defined for this version
		config_file =  (String) (table.get(version));
		key = version;
		
		if(config_file == null)
		{
			// Cannot find the corresponding config file, use default
			//config_file = (String) (table.get(resourceBundle.getString("HyadesGADefaultLogVersionTagName")));
			config_file = (String) (table.get(GLADefaultLogVersionTagName));
			
			// The default (which should be) is not defined
			if(config_file != null) {
				//key = resourceBundle.getString("HyadesGADefaultLogVersionTagName"); 
				key = GLADefaultLogVersionTagName;
			}			
		}
		return key;
	
	}
	
	public static String sendAdapter(Node node, Map table, String sourceFile, String destinationFile) throws Exception{		
		IFileManager fileManager = FileManagerFactory.getInstance().create(node.getConnection());	
		String targetDirectory = WindowsAdapterRootDir;
		String targetFile = targetDirectory+destinationFile;
		String returnValue = "";
		try {			
			fileManager.putFile(sourceFile, targetFile);
			returnValue = targetFile;			
		} catch (Exception e1) {
			targetDirectory = UnixAdapterRootDir; 
			targetFile = targetDirectory+destinationFile;
			try{
				fileManager.putFile(sourceFile, targetFile);
				returnValue = targetFile;
			}catch(IOException e2){
				throw e2;
			}
		}
		if(targetDirectory!=null){
			table.put(GLAConfigFileRootAttributeName,targetDirectory);
		}
		return returnValue;
	}
	
	public static void deleteTempAdapter(Node node, String file){
		IFileManager fileManager = FileManagerFactory.getInstance().create(node.getConnection());
		if(!((file.startsWith(WindowsAdapterRootDir+"GLA") || file.startsWith(UnixAdapterRootDir+"GLA")) && file.endsWith("adapter"))){
			return;
		}
		try{
			fileManager.deleteFile(file);
		}catch(Exception e){
			
		}
	}
	
	private static String createDateTime() {
		DateFormat formatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
		Date today = new Date();
		String result = formatter.format(today);
		return result;
	}
	
	private static Component createFilterComponent(FilterConfigType filter)
	{
		String date = createDateTime();
		Component mcontext = ContextFactory.eINSTANCE.createComponent();
		mcontext.setDescription(filter.getDescription());
		mcontext.setUniqueID(filter.getUniqueID());
		mcontext.setExecutableClass("org.eclipse.hyades.logging.adapter.internal.filters.Filter");
		mcontext.setImplementationCreationDate(date);
		mcontext.setLoggingLevel("50");
		mcontext.setRole(RoleNames.CBE_FILTER_LITERAL);
		mcontext.setRoleCreationDate(date);
		mcontext.setName(mcontext.getDescription());
		mcontext.setImplementationVersion("1.0");
		mcontext.setRoleVersion("1.0");
				
		return mcontext;
	}
	
	private static EObject createNumericFilterRule(BinaryExpression binaryExpression){

		NumericalFilterRuleType filterRule = null;
		EStructuralFeature feature = ((SimpleOperand)binaryExpression.getLeftOperand()).getFeature();
		Object attributeValue =  (((SimpleOperand) binaryExpression
				.getRightOperands().get(0)).getValue());
		String value = attributeValue.toString();
		
		int operatorIndex = binaryExpression.getOperator().getValue();
		switch(operatorIndex){
			case RelationalOperators.EQ:{
				filterRule = createNumericalFilterRuleType(binaryExpression);
				filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.EQUALS));
				break;
			}
			
			case RelationalOperators.LT:{
				filterRule = createNumericalFilterRuleType(binaryExpression); 
				filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.LESS_THAN));
				break;					
			}
			
			case RelationalOperators.GT:{
				filterRule = createNumericalFilterRuleType(binaryExpression);
				filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.GREATER_THAN));
				break;					
			}
			
			case RelationalOperators.LE:{
				filterRule = createNumericalFilterRuleType(binaryExpression);
				filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.LESS_THAN_OR_EQUAL));				
				//return createBlock(feature, RelationalOperators.LE, value);
				break;
			}
			
			case RelationalOperators.GE:{
				filterRule = createNumericalFilterRuleType(binaryExpression);
				filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.GREATER_THAN_OR_EQUAL));
				//return createBlock(feature, RelationalOperators.GE, value);
				break;
			}
			
			case RelationalOperators.BETWEEN:{
				FilterRuleBlockType block = FilterFactory.eINSTANCE
				.createFilterRuleBlockType();
				block.setName("BetweenFilterBlock");
				block.setOperator(LogicalOperatorType.get(LogicalOperatorType.AND));
//				FilterRuleBlockType block1 = createBlock(feature, RelationalOperators.GE, value);
//				block.getFilterRuleBlock().add(block1);
				NumericalFilterRuleType leftBoundRule = createRule(feature, RelationalOperators.GE, value);
				block.getAbstractFilterRule().add(leftBoundRule);
				
				Object attributeValue1 =  (((SimpleOperand) binaryExpression
						.getRightOperands().get(1)).getValue());
				String value2 = attributeValue1.toString();
//				FilterRuleBlockType block2 = createBlock(feature, RelationalOperators.LE, value2);
//				block.getFilterRuleBlock().add(block2);	
				NumericalFilterRuleType rightBoundRule = createRule(feature, RelationalOperators.LE, value2);
				block.getAbstractFilterRule().add(rightBoundRule);
				
				return block;
			}
			
			case RelationalOperators.NEQ:{
				FilterRuleBlockType block = FilterFactory.eINSTANCE
				.createFilterRuleBlockType();
				block.setName("NotEqualFilterBlock");
				block.setOperator(LogicalOperatorType.get(LogicalOperatorType.OR));

				NumericalFilterRuleType lessThenRule = createRule(feature, RelationalOperators.LT, value);
				block.getAbstractFilterRule().add(lessThenRule);
				
				NumericalFilterRuleType greaterThenRule = createRule(feature, RelationalOperators.GT, value);
				block.getAbstractFilterRule().add(greaterThenRule);
				
				return block;
			}
		}			
		if(filterRule!=null){	
			filterRule.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.NumericalFilterType");
		}
		return filterRule;
	}
	
	private static EObject createTimeFilterRuleType(BinaryExpression binaryExpression){
		EObject filterRule = createNumericFilterRule(binaryExpression);
		if (filterRule != null) {
			if (filterRule instanceof NumericalFilterRuleType) {

				((NumericalFilterRuleType) filterRule)
						.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.TimeFilterType");
				Object attributeValue = (((SimpleOperand) binaryExpression
						.getRightOperands().get(0)).getValue());
				if (attributeValue instanceof Number) {
					SimpleDateFormat format = new SimpleDateFormat(
							dateTimeFormat);
					String value = format.format(new Date(
							((Number) attributeValue).longValue() / 1000));
					((NumericalFilterRuleType) filterRule)
							.setAttributeValue(value);
				}
			} else if (filterRule instanceof FilterRuleBlockType) {
				List containedRules = ((FilterRuleBlockType) filterRule)
						.getAbstractFilterRule();
				int s = containedRules.size();
				Object element = null;
				for (int i = 0; i < s; i++) {
					element = containedRules.get(i);
					if (element != null
							&& element instanceof NumericalFilterRuleType) {
						((NumericalFilterRuleType) element)
								.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.TimeFilterType");
						Double attributeValue = new Double(
								((NumericalFilterRuleType) element)
										.getAttributeValue());

						SimpleDateFormat format = new SimpleDateFormat(
								dateTimeFormat);
						String value = format.format(new Date(
								((Number) attributeValue).longValue() / 1000));
						((NumericalFilterRuleType) element)
								.setAttributeValue(value);
					}
				}
			}

		}
		return filterRule;
	}
	
	private static StringFilterRuleType createStringFilterRuleType(BinaryExpression binaryExpression){
		StringFilterRuleType filterRule = FilterFactory.eINSTANCE.createStringFilterRuleType();			
		EStructuralFeature feature = ((SimpleOperand)binaryExpression.getLeftOperand()).getFeature();
		Object attributeValue =  (((SimpleOperand) binaryExpression
				.getRightOperands().get(0)).getValue());
		String value = attributeValue.toString();
		
		PathExpressionType path = FilterFactory.eINSTANCE.createPathExpressionType();
		path.getNode().add(feature.getEContainingClass().getName().substring(3));
		path.getNode().add(feature.getName());
		filterRule.setAttributeNamePath(path);
		filterRule.setAttributeValue(value);
		
		if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.EQ)){
			filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.EQUALS));
		}
		else if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.LIKE)){
			int i = value.indexOf("*"); 
			if(i>0){
				filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.STARTS_WITH));
				value = value.substring(0,i);
				filterRule.setAttributeValue(value);
			}else if(i==0){
				int j = value.indexOf("*",1); 
				if(j>=0){
					filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.CONTAINS));
					value = value.substring(1,j);
					filterRule.setAttributeValue(value);
				}else{
					filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.ENDS_WITH));
					value = value.substring(1);
					filterRule.setAttributeValue(value);					
				}
			}
		}else if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.LT) ||
				binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.LE)){
			filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.ENDS_WITH));
		}else if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.GT) ||
				binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.GE)){
			filterRule.setOperator(StringRelationalOperatorType.get(StringRelationalOperatorType.STARTS_WITH));			
		}
		filterRule.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.StringFilterType");
		return filterRule;
	}
	
	private static FilterRuleBlockType createBlock(EStructuralFeature feature, int operator, String value){
		
		FilterRuleBlockType block = FilterFactory.eINSTANCE
		.createFilterRuleBlockType();		
		block.setOperator(LogicalOperatorType.get(LogicalOperatorType.OR));
		NumericalFilterRuleType filterRule = FilterFactory.eINSTANCE.createNumericalFilterRuleType();
		PathExpressionType path = FilterFactory.eINSTANCE.createPathExpressionType();
		path.getNode().add(feature.getEContainingClass().getName().substring(3));
		path.getNode().add(feature.getName());
		filterRule.setAttributeNamePath(path);
		filterRule.setAttributeValue(value);		
		
		if(operator==RelationalOperators.LE){
			block.setName("LessThenOrEqualFilterBlock");
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.LESS_THAN));
		}else if(operator==RelationalOperators.GE){
			block.setName("GreaterThenOrEqualFilterBlock");
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.GREATER_THAN));
		}
		filterRule.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.NumericalFilterType");
		
		NumericalFilterRuleType filterRule1 = FilterFactory.eINSTANCE.createNumericalFilterRuleType();
		
		PathExpressionType path1 = FilterFactory.eINSTANCE.createPathExpressionType();
		path1.getNode().add(feature.getEContainingClass().getName().substring(3));
		path1.getNode().add(feature.getName());
		filterRule1.setAttributeNamePath(path1);
		filterRule1.setAttributeValue(value);
		filterRule1.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.EQUALS));
		filterRule1.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.NumericalFilterType");
		
		block.getAbstractFilterRule().add(filterRule);
		block.getAbstractFilterRule().add(filterRule1);								
		return block;
		
	}
	
	private static NumericalFilterRuleType createRule(EStructuralFeature feature, int operator, String value){
		NumericalFilterRuleType filterRule = FilterFactory.eINSTANCE.createNumericalFilterRuleType();
		PathExpressionType path = FilterFactory.eINSTANCE.createPathExpressionType();
		path.getNode().add(feature.getEContainingClass().getName().substring(3));
		path.getNode().add(feature.getName());
		filterRule.setAttributeNamePath(path);
		filterRule.setAttributeValue(value);		
		
		if(operator==RelationalOperators.LE){
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.LESS_THAN_OR_EQUAL));
		}else if(operator==RelationalOperators.GE){
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.GREATER_THAN_OR_EQUAL));
		}else if(operator==RelationalOperators.LT){
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.LESS_THAN));
		}else if(operator==RelationalOperators.GT){
			filterRule.setOperator(NumericalRelationalOperatorType.get(NumericalRelationalOperatorType.GREATER_THAN));			
		}
		filterRule.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.NumericalFilterType");
		return filterRule;
		
	}
	
	private static NumericalFilterRuleType createNumericalFilterRuleType(BinaryExpression binaryExpression){
		
		NumericalFilterRuleType filterRule = FilterFactory.eINSTANCE.createNumericalFilterRuleType();
		
		EStructuralFeature feature = ((SimpleOperand)binaryExpression.getLeftOperand()).getFeature();
		Object attributeValue =  (((SimpleOperand) binaryExpression
				.getRightOperands().get(0)).getValue());
		String value = attributeValue.toString();
		
		PathExpressionType path = FilterFactory.eINSTANCE.createPathExpressionType();
		path.getNode().add(feature.getEContainingClass().getName().substring(3));
		path.getNode().add(feature.getName());
		filterRule.setAttributeNamePath(path);
		filterRule.setAttributeValue(value);		
		return filterRule;
	}
	
	private static RangeFilterRuleType createRangeFilterRuleType(BinaryExpression binaryExpression){
		RangeFilterRuleType filterRule = FilterFactory.eINSTANCE.createRangeFilterRuleType();
		Object value = null;
		if(binaryExpression.getRightOperands().size()>0){	
			value = (((SimpleOperand) binaryExpression
					.getRightOperands().get(0)).getValue());
			if(value!=null){
				filterRule.setAttributeValue(value.toString());						
			}
		}
		if(value==null){					
			filterRule.setAttributeValue("0");
		}
		if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.GT)){				
			filterRule.setOperator(RangeRelationalOperatorType.get(RangeRelationalOperatorType.FIRST));				
		}else if(binaryExpression.getOperator()==RelationalOperators.get(RelationalOperators.LT)){
			filterRule.setOperator(RangeRelationalOperatorType.get(RangeRelationalOperatorType.LAST));
		}			

		PathExpressionType path = FilterFactory.eINSTANCE.createPathExpressionType();

		if(((SimpleOperand) binaryExpression.getLeftOperand()).getName().equals(ImportLogFiltersStandardUI.EVENTCOUNT_OPERAND_NAME)){			
			path.getNode().add(GLARangeFilterAttributeNamePathRoot);
			path.getNode().add(GLARangeFilterByEventsAttributeName);
			filterRule.setAttributeNamePath(path);
			
		}else if(((SimpleOperand) binaryExpression.getLeftOperand()).getName().equals(ImportLogFiltersStandardUI.SECONDSCOUNT_OPERAND_NAME)){			
			path.getNode().add(GLARangeFilterAttributeNamePathRoot);
			path.getNode().add(GLARangeFilterBySecondsAttributeName);
			filterRule.setAttributeNamePath(path);
		}
		
		filterRule.setImplementationClass("org.eclipse.hyades.logging.adapter.internal.filters.NumericalRangeFilterType");
		return filterRule;
	}
	
	
	private static String getMessage(String key, String[] params){
		try{
			return LogUIPlugin.getResourceString(key, params);
		}catch(Exception e){
			
		}
		return key;
	}

}
