/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SymptomDBSelectionGroup.java,v 1.4 2005/02/16 22:20:28 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.log.ui.internal.util;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.hyades.log.ui.internal.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

/**
 * Workbench-level composite for choosing a container.
 */
public class SymptomDBSelectionGroup extends Composite {
	// The listener to notify of events
	private Listener listener;

	// Last selection made by user
	private Object selectedContainer;
	
	private TreeViewer treeViewer;

	// sizing constants
	private static final int SIZING_SELECTION_PANE_HEIGHT = 250;
	private static final int SIZING_SELECTION_PANE_WIDTH = 250;

/**
 * Creates a new instance of the widget.
 *
 * @param parent The parent widget of the group.
 * @param listener A listener to forward events to. Can be null if
 *	 no listener is required.
 * @param allowNewContainerName Enable the user to type in a new container
 *  name instead of just selecting from the existing ones.
 */
public SymptomDBSelectionGroup (Composite parent, Listener listener, boolean allowNewContainerName) {
	this(parent, listener, allowNewContainerName, null);
}
/**
 * Creates a new instance of the widget.
 *
 * @param parent The parent widget of the group.
 * @param listener A listener to forward events to.  Can be null if
 *	 no listener is required.
 * @param allowNewContainerName Enable the user to type in a new container
 *  name instead of just selecting from the existing ones.
 * @param message The text to present to the user.
 */
public SymptomDBSelectionGroup (Composite parent, Listener listener, boolean allowNewContainerName, String message) {
	super (parent, SWT.NONE);
	this.listener = listener;
	this.setFont(parent.getFont());
	createContents();
}
/**
 * The container selection has changed in the
 * tree view. Update the container name field
 * value and notify all listeners.
 */
public void containerSelectionChanged(Object container) {
	selectedContainer = container;
	
	// fire an event so the parent can update its controls
	if (listener != null) {
		Event changeEvent = new Event();
		changeEvent.type = SWT.Selection;
		changeEvent.widget = this;
		listener.handleEvent(changeEvent);
	}
}
/**
 * Creates the contents of the composite.
 */
public void createContents() {
	GridLayout layout = new GridLayout();
	layout.marginWidth = 0;
	setLayout(layout);
	setLayoutData(new GridData(GridData.FILL_BOTH));

	Label label = new Label(this,SWT.WRAP);
	label.setText(LogUIPlugin.getResourceString("STR_SELECT_DB_DESCR")); 
	label.setFont(this.getFont());

	// filler...
	new Label(this, SWT.NONE);
	createTreeViewer();
}
/**
 * Returns a new drill down viewer for this dialog.
 *
 * @return a new drill down viewer
 */
protected void createTreeViewer() {
	// Create drill down.
	SymptomDBDrillDownComposite drillDown = new SymptomDBDrillDownComposite(this, SWT.BORDER);
	GridData spec = new GridData(
		GridData.VERTICAL_ALIGN_FILL | GridData.HORIZONTAL_ALIGN_FILL |
		GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL);
	spec.widthHint = SIZING_SELECTION_PANE_WIDTH;
	spec.heightHint = SIZING_SELECTION_PANE_HEIGHT;
	drillDown.setLayoutData(spec);

	// Create tree viewer inside drill down.
	treeViewer = new TreeViewer(drillDown, SWT.NONE);
	drillDown.setChildTree(treeViewer);
	treeViewer.setContentProvider(new SymptomDBContentProvider());
	treeViewer.setLabelProvider(new SymptomDBLabelProvider());
	treeViewer.addSelectionChangedListener(
		new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection selection = (IStructuredSelection)event.getSelection();
				containerSelectionChanged(selection.getFirstElement()); // allow null
			}
		});
	treeViewer.addDoubleClickListener(
		new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				ISelection selection = event.getSelection();
				if (selection instanceof IStructuredSelection) {
					Object item = ((IStructuredSelection)selection).getFirstElement();
					if (treeViewer.getExpandedState(item))
						treeViewer.collapseToLevel(item, 1);
					else
						treeViewer.expandToLevel(item, 1);
				}
			}
		});

	// This has to be done after the viewer has been laid out
	treeViewer.setInput(ResourcesPlugin.getWorkspace());
}
/**
 * Returns the currently entered container name.
 * Null if the field is empty. Note that the
 * container may not exist yet if the user
 * entered a new container name in the field.
 */
public IPath getContainerFullPath()
{
	if (selectedContainer == null)
		return null;
	else if(selectedContainer instanceof IResource)
		return ((IResource)selectedContainer).getFullPath();
	else
	   return new Path(selectedContainer.toString()); 
}
/**
 * Gives focus to one of the widgets in the group, as determined by the group.
 */
public void setInitialFocus() {
		treeViewer.getTree().setFocus();
}
/**
 * Sets the selected existing container.
 */
public void setSelectedContainer(Object container)
{
	selectedContainer = container;
	
	//expand to and select the specified container
	ArrayList itemsToExpand = new ArrayList();

	if(container instanceof IContainer)
	{
		IContainer parent = ((IContainer)container).getParent();
		while (parent != null) {
			itemsToExpand.add(0,parent);
			parent = parent.getParent();
		}
		treeViewer.setExpandedElements(itemsToExpand.toArray()); 
		treeViewer.setSelection(new StructuredSelection(container),true);
	}
}
}
