/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogPaneTreeViewer.java,v 1.24 2005/02/19 01:19:07 apnan Exp $
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.log.ui.internal.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.analysis.engine.IAnalysisMonitor;
import org.eclipse.hyades.analysis.engine.ILogAnalyzer;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.ContextIds;
import org.eclipse.hyades.log.ui.internal.util.ProgressMonitorAdapter;
import org.eclipse.hyades.log.ui.internal.views.ExtensionPointHandler.ConfigurationElement;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionManager;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.help.WorkbenchHelp;

public class LogPaneTreeViewer extends TreeViewer {

	private LogViewerUI _logViewUI = null;
	private ExtensionPointHandler eph = ExtensionPointHandler
			.getExtensionPointHandler();
	private Collection _analyzedObjects;
	private ISelection sel;
	protected Action _updateAction;
	protected Action _analyzeAllAction;
	protected Action _analyzeAction;
	protected boolean _enableAnalyze = true;
	
	public static final int NO_ANALYZE = -1;
	public static final int ANALYZE_SEL = 0;
	public static final int ANALYZE_ALL = 1;
	

	private static Hashtable _analyzeActionForRuntimeIDs = new Hashtable();
	private static Hashtable _analyzeObjectsForRuntimeIds = new Hashtable();

	//-------------------------------
	// UpdateAction popup action
	//-------------------------------
	class UpdateAction extends Action {
		public UpdateAction(String name) {
			super(name);

			WorkbenchHelp.setHelp(this, ContextIds.ACTLOG_VIEW_POPUP_REFRESH);
		}

		/**
		 * Invoked when an action occurs. Argument context is the Window which
		 * contains the UI from which this action was fired. This default
		 * implementation prints the name of this class and its label.
		 */
		public void run() {
			((LogViewer)_logViewUI.getViewerPage().getTraceViewer()).refresh();
		}
	}
	
	private static String getIdForObject(EObject eObj)
	{
		String id = null;
		if (eObj instanceof TRCAgentProxy)
		{
			TRCAgentProxy ap = (TRCAgentProxy)eObj;
			id = ap.getProcessProxy().getNode().eResource().getURI().toString()+"."+ap.getName()+"."+ap.hashCode();
		}
		else if (eObj instanceof CorrelationContainerProxy)
		{
			CorrelationContainerProxy ap = (CorrelationContainerProxy)eObj;
			id = ap.getName()+"."+ap.hashCode();
		}
		else
			id = ""+eObj.hashCode();
			
		return id;
	}
	
	private static Collection getAnalyzeObjectsForAgent(EObject eObj)
	{
		String id = getIdForObject(eObj);		
		
		if (id != null)
			return (Collection)_analyzeObjectsForRuntimeIds.get(id);
		
		return null;
	}
	
	private static void putAnalyzeObjectsForAgent(EObject eObj, Collection objects)
	{
		String id = getIdForObject(eObj);
		
		if (id != null && objects !=null)		
			_analyzeObjectsForRuntimeIds.put(id, objects);
	}
	
	private static IAction getAnalyzeActionForAgent(EObject eObj)
	{
		String id = getIdForObject(eObj);		
		
		if (id != null)
			return (IAction)_analyzeActionForRuntimeIDs.get(id);
		
		return null;
	}
	
	private static void putAnalyzeActionForAgent(EObject eObj, IAction action)
	{
		String id = getIdForObject(eObj);
		
		if (id != null)		
			_analyzeActionForRuntimeIDs.put(id, action);

	    if (eObj instanceof CorrelationContainerProxy)
		{
	    	CorrelationContainerProxy cor = (CorrelationContainerProxy)eObj;
	    	
	    	EList list = cor.getCorrelatedAgents();
	    	
	    	for (int i = 0; i < list.size(); i++)
	    	{
	    		putAnalyzeActionForAgent((EObject)list.get(i), action);
	    	}
		} 
	}
	
	public void enableAnalyze(boolean enable) {
		_enableAnalyze = enable;
	}
	
	public LogPaneTreeViewer(LogViewerUI logViewUI, Composite parent,
			String role) {
		super(parent);

		_logViewUI = logViewUI;

		_updateAction = new UpdateAction(LogUIPlugin
				.getResourceString("REFRESH_VIEWS"));
	}
	
	
	public void setContentProvider(IContentProvider contentProvider)
	{
		super.setContentProvider(contentProvider);
		
		AnalyzeAction action = (AnalyzeAction)getAnalyzeActionForAgent(_logViewUI.getViewerPage().getMOFObject());
		_analyzedObjects = getAnalyzeObjectsForAgent(_logViewUI.getViewerPage().getMOFObject());
		if (action != null)
		{
			if (get_analyzeAllAction() == null)
				_analyzeAllAction = new AnalyzeAction(action.getText(), action.getLogAnalyzer(), true);	
			if (get_analyzeAction() == null)
				_analyzeAction = new AnalyzeAction(action.getText(), action.getLogAnalyzer(), false);
			
			if (action.analyzeAll)
				((LogContentProvider) getContentProvider()).setAnalyzeAction(get_analyzeAllAction());
			else 
				((LogContentProvider) getContentProvider()).setAnalyzeAction(get_analyzeAction());
		}
	}

	// Attaches a context menu listener to the table.
	public void setMenuListener(IMenuListener menuListener) {
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(menuListener);
		Menu menu = menuMgr.createContextMenu(getControl());
		getControl().setMenu(menu);
	}

	public void addSelectionListener(SelectionListener listener) {
		((Tree) getControl()).addSelectionListener(listener);
	}

	public void fillContextMenu(IMenuManager menu) {

		ISelection sel = getSelection();

		if (sel instanceof IStructuredSelection) {

			if (!sel.isEmpty())
				updateMenu(menu, ((IStructuredSelection) sel).getFirstElement());
		}

		menu.add(new Separator());
		menu.add(_updateAction);
		menu.add(new Separator());

		//popup menu contribution
		menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	public void updateMenu(IMenuManager menu, Object logObject) {

		final ISelection sel = getSelection();

		if (sel instanceof IStructuredSelection) {

			//Action analyzeAction = new
			// AnalyzeAction(LogUIPlugin.getResourceString("STR_ANALYZE_ACT"),
			// logType, false);

			//menu.add(analyzeAction);
			MenuManager manager = new MenuManager(LogUIPlugin
					.getResourceString("STR_ANALYZE_ACT"));
			menu.add(manager);
			fillMenu(manager, logObject);
			//WorkbenchHelp.setHelp(analyzeAction,
			// ContextIds.ACTLOG_VIEW_POPUP_ANALYZE);
			menu.add(new Separator());

			//Action analyzeAllAction = new
			// AnalyzeAction(LogUIPlugin.getResourceString("STR_ANALYZE_ALL_ACT"),
			// logType, true);
			// menu.add(analyzeAllAction);
			manager = new MenuManager(LogUIPlugin
					.getResourceString("STR_ANALYZE_ALL_ACT"));
			menu.add(manager);
			fillMenu(manager);
		}
	}

	public void dispose() {

		List logAnalyzers = eph.getLogAnalyzers();
		for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
			ILogAnalyzer element = ((ConfigurationElement) iter.next())
					.getAnalyzer();
			element.unloadDatabase();
		}		
	}

	public void resetAnalyzedObjects() {
		_analyzedObjects = null;
	}
	
	public Collection getAnalyzedObjects(){
		if(_analyzedObjects==null){
			_analyzedObjects = new HashSet();
		}
		return _analyzedObjects;
	}
	
	private List getObjectsToAnalyze(int option) {

		if (option == ANALYZE_SEL) {
			
			Display.getDefault().syncExec(new Runnable(){
				public void run(){
				sel = getSelection();
				}
			});
			List objects = new ArrayList();
			if (sel instanceof IStructuredSelection) {
				Iterator iterator = ((IStructuredSelection) sel).iterator();
				Object object = null;

				while (iterator.hasNext()) {
					object = iterator.next();
					if (object != null && object instanceof CBECommonBaseEvent) {
						getAnalyzedObjects().add(((CBECommonBaseEvent) object)
								.getGlobalInstanceId());
					}
				}
			}
			List input = ((LogContentProvider) getContentProvider()).getPageList(getInput());
			boolean inputXMI = ((LogContentProvider) getContentProvider()).isInputXMI(getInput());
			for (int i = 0; i < input.size(); i++) {
				Object object = input.get(i);
				if(!(object instanceof CBECommonBaseEvent)){
					continue;
				}
				CBECommonBaseEvent element = (CBECommonBaseEvent) object;
				
				if (getAnalyzedObjects().contains(element.getGlobalInstanceId())) {
					objects.add(element);
				}else if(inputXMI && element.isAnalyzed()){
					objects.add(element);
				}
			}
			return objects;
		} else if (option == ANALYZE_ALL) {
			getAnalyzedObjects().clear();
			return ((LogContentProvider) getContentProvider())
					.getPageList(getInput());
		}
		return Collections.EMPTY_LIST;
	}

	private boolean isObjectInList(String globalInstanceID, List input) {
		int s = input.size();
		for (int i = 0; i < s; i++) {
			if (((CBECommonBaseEvent) input.get(i)).getGlobalInstanceId()
					.equals(globalInstanceID))
				return true;
		}
		return false;
	}

	private void fillMenu(IContributionManager innerMgr) {

		innerMgr.removeAll();
		List logAnalyzers = eph.getLogAnalyzers();
		if (logAnalyzers != null) {
			for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
				ConfigurationElement config = (ConfigurationElement) iter
						.next();
				ILogAnalyzer analyzer = config
				.getAnalyzer();				
				_analyzeAllAction = new AnalyzeAction(config.getName(), new LogAnalyzerWrapper(analyzer), true);
				innerMgr.add(_analyzeAllAction);
			}
		}

	}

	private void fillMenu(IContributionManager innerMgr, Object logObject) {

		innerMgr.removeAll();
		List logAnalyzers = eph.getLogAnalyzer(logObject);
		
		if (logAnalyzers != null) {
			for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
				ConfigurationElement config = (ConfigurationElement) iter
						.next();
				ILogAnalyzer analyzer = config
				.getAnalyzer();
				_analyzeAction = new AnalyzeAction(config.getName(), new LogAnalyzerWrapper(analyzer), false);
				innerMgr.add(_analyzeAction);
			}
		}
	}

	abstract class BasicLoadAction extends Action {

		private LogAnalyzerWrapper logAnalyzer = null;
//		private boolean reload = false;

		public BasicLoadAction(String label, LogAnalyzerWrapper logAnalyzer) {
			super(label);
			this.logAnalyzer = logAnalyzer;
		}

		public LogAnalyzerWrapper getLogAnalyzer() {
			return logAnalyzer;
		}

//		public boolean getReload() {
//			return reload;
//		}
//
//		public void setReload(boolean newVal) {
//			reload = newVal;
//		}

//		public void run() {
//
//			BusyIndicator.showWhile(null, new Runnable() {
//
//				public void run() {
//
//					logAnalyzer.loadDatabase();
//
//					if (logAnalyzer.errorMsg() != null) {
//						MessageDialog.openError(getControl().getShell(),
//								LogUIPlugin.getResourceString("STR_LOG_MSG"),
//								logAnalyzer.errorMsg());
//						return;
//					}
//				}
//
//			});
//		}
		public boolean isEnabled() {
			return true;
		}
	}

//	class LoadAction extends BasicLoadAction {
//
//		public LoadAction(String label, ILogAnalyzer logAnalyzer) {
//			super(label, logAnalyzer);
//
//		}
//
//		public void run() {
//
//			SymptomDBDialog dialog = new SymptomDBDialog(getControl()
//					.getShell(), LogUIPlugin
//					.getResourceString("STR_SYMPTOM_DB_DLG_TITLE"), null);
//			setReload(false);
//			dialog.open();
//
//			if (dialog.getReturnCode() == Window.OK) {
//				setReload(true);
//			}
//		}
//
//	}

	class AnalyzeAction  extends BasicLoadAction{

		boolean analyzeAll = false;
		protected List sel;
		protected LogAnalyzerWrapper aLogAnalyzer;
		protected List result;

		public AnalyzeAction(String label, LogAnalyzerWrapper logAnalyzer) {
			super(label, logAnalyzer);
		}

		public AnalyzeAction(String label, LogAnalyzerWrapper logAnalyzer,
				boolean analyzeAll) {
			super(label, logAnalyzer);
			this.analyzeAll = analyzeAll;

		}

		public void run() {
			
			if (_enableAnalyze) {
			IRunnableWithProgress analyze = new IRunnableWithProgress() {
				public void run(IProgressMonitor m) {
					
					
					IAnalysisMonitor monitor = new ProgressMonitorAdapter(m);
					((LogContentProvider) getContentProvider()).setAnalyzeAction(AnalyzeAction.this);

//					if (analyzeAll)
					putAnalyzeActionForAgent(_logViewUI.getViewerPage().getMOFObject(), AnalyzeAction.this);

					List listToAnalyze = getObjectsToAnalyze(analyzeAll ? 1 : 0);
					
					monitor.beginTask(LogUIPlugin.getResourceString("STR_ANALYZE_PROGRESS"),listToAnalyze.size());
					monitor.subTask(LogUIPlugin
							.getResourceString("STR_ANALYZE_LOAD_DB"));
					getLogAnalyzer().loadDatabase();

					if (getLogAnalyzer().errorMsg() == null) {
						if(result==null){
							result = new ArrayList();
						}
						getLogAnalyzer().analyze(listToAnalyze, result, monitor);
					}
					monitor.done();
				}
			};

			try {
				LogUIPlugin.getActiveWorkbenchWindow().run(true, true, analyze);
			} catch (Exception e) {
				LogUIPlugin.log(e);
			}

			if (getLogAnalyzer().errorMsg() != null) {
				MessageDialog.openError(getControl().getShell(),
						LogUIPlugin.getResourceString("STR_LOG_MSG"),
						getLogAnalyzer().errorMsg());
						_analyzedObjects = getAnalyzeObjectsForAgent(_logViewUI.getViewerPage().getMOFObject());
				return;
			}

			//Refresh after analyzing all selected records.
			((LogContentProvider) getContentProvider()).setAnalyzed(true);
			refresh();
			((LogContentProvider) getContentProvider()).setAnalyzed(false);

			//All analyze objects only if analyis is done without error
			putAnalyzeObjectsForAgent(_logViewUI.getViewerPage().getMOFObject(),_analyzedObjects);
			
			//Need to send an event so that the Analysis Result area will be
			// updated.
			if(((Tree) getControl()).getSelection().length>0){
				Event event = new Event();
				event.item = ((Tree) getControl()).getSelection()[0];
				getControl().notifyListeners(SWT.Selection, event);
			}
		}
		}
	}

	/**
	 * @return Returns the _analyzeAction.
	 */
	public Action get_analyzeAction() {
		return _analyzeAction;
	}
	/**
	 * @return Returns the _analyzeAllAction.
	 */
	public Action get_analyzeAllAction() {
		return _analyzeAllAction;
	}
}