/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDatapoolEquivalenceClass.java,v 1.4 2005/02/16 22:21:43 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.edit.datapool;

/**
 * An <i><b>equivalence class</b></i> is a logical grouping of 
 * {@link IDatapoolRecord records} within a {@link IDatapool datapool}.
 * All records in an equivalence class have the same uniform length as
 * records in all other equivalence classes associated with a single
 * {@link IDatapool datapool}.  Note that each equivalence class is a
 * named element with an associated name and ID.
 * 
 * @see	IDatapool
 * @see IDatapoolRecord
 */
public interface IDatapoolEquivalenceClass extends
	org.eclipse.hyades.execution.runtime.datapool.IDatapoolEquivalenceClass,
	INamedElement
{
	/**
	 * Construct a new record with an appropriate number of elements for the 
	 * active datapool.  The number of elements equals the number of
	 * {@link IDatapool#getVariableCount() variables} in the datapool.  
	 *  
	 * @see #constructRecord(Object[])
	 * @see	IDatapool#getVariableCount()
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 */
	public IDatapoolRecord constructRecord();

	
	/**
	 * Construct a new record with an appropriate number of elements for the 
	 * active datapool.  The number of elements equals the number of
	 * {@link IDatapool#getVariableCount() variables} in the datapool.  The supplied
	 * values will be padded or truncated to match the number of variables
	 * available.  The supplied values may be <code>null</code> in which 
	 * case all elements in the returned record will be <code>null</code>.
	 * 
	 * @param	elements	The initial values defined for the record or
	 *						<code>null</code> if no initial values are desired.
	 * 
	 * @see #constructRecord()
	 * @see	IDatapool#getVariableCount()
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 */
	public IDatapoolRecord constructRecord(Object[] elements);

	/**
	 * Insert the specified record before the zero based record index.
	 * Any attempt to insert the same record into a datapool multiple times
	 * will result in an appropriate exception.
	 * 
	 * @param	data			The new record.
	 * @param	beforeRecord	The zero based record index to insert the new record before.
	 * 
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 * @see	#removeRecord(int)
	 */
	public void insertRecord(IDatapoolRecord data, int beforeRecord);
	
	/**
	 * Append the specified record to the end of the datapool.
	 * Any attempt to insert the same Record into a datapool multiple times
	 * will result in an appropriate exception.
	 * 
	 * @param	record	The new record.
	 * 
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 * @see	#removeRecord(int)
	 */
	public void appendRecord(IDatapoolRecord record);
	
	/**
	 * Update the specified record with the supplied data.  This method 
	 * simulates {@link #removeRecord(int) removeRecord} followed by 
	 * {@link #insertRecord(IDatapoolRecord,int) insertRecord} without the need
	 * to perform both calls.  The specified record index is zero
	 * based.  Any attempt to replace a record in the datapool with 
	 * a record already in the datapool will result in an appropriate
	 * exception.
	 * 
	 * @param	data			The new record data.
	 * @param	recordIndex		The zero based record index to update the new record data into.
	 * 
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#removeRecord(int)
	 */
	public void replaceRecord(IDatapoolRecord data, int recordIndex);
	
	/**
	 * Remove the specified record.
	 * 
	 * @param	recordIndex		The zero based record index of the record to remove.
	 * 
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 */
	public IDatapoolRecord removeRecord(int recordIndex);
	
	
	/**
	 * Move the specified record.
	 * 
	 * @param	sourceRecordIndex	The zero based record index of the record to move.
	 * @param	targetRecordIndex	The record index marking the target location for
	 *							    the move action.
	 * 
	 * @see	#insertRecord(IDatapoolRecord,int)
	 * @see	#appendRecord(IDatapoolRecord)
	 * @see	#replaceRecord(IDatapoolRecord,int)
	 */
	public void moveRecord(int sourceRecordIndex, int targetRecordIndex);
	
	
	/**
	 * Inform the equivalence class of a new order for records.  The current record index
	 * is specified in the array, with the index into the array expressing
	 * the desired order.  So if {3, 4, 2, 1, 0} is passed in, the third record
	 * if moved to the first position, the fourth to the second and so on.
	 * <p>
	 * If the specified set of records is the wrong length or contains redundant 
	 * record references an appropriate exception is thrown.
	 * 
	 * @param	recordOrder	The desired order for the records in the equivalence class.
	 */
	public void reorder(int[] recordOrder);
	
	
}