/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDatapoolRecord.java,v 1.4 2005/02/16 22:21:43 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.runtime.datapool;

/**
 * A logical grouping of values in a {@link IDatapool datapool}.  A datapool
 * is an ordered collection {@link IDatapoolEquivalenceClass equivalence classes}
 * which contain an order collection of records.  And to complete the 
 * hierarchy, a record is an ordered collection of {@link IDatapoolCell cells}.
 * <p>
 * Several restrictions exist on records within a datapool:
 * <ul>
 * <li>A record instance can only exist in one datapool.  Obviously a deep
 * copy can be performed so the same contents can appear in multiple datapools.</li>
 * <li>A record instance must be unique within the datapool.  It can not
 * be added to multiple equivalence classes or more then once to the
 * same equivalence class.</li>
 * <li>Cells within a record must each be unique to that instance of the
 * record and can not be duplicated in multiple records at once.
 * <li>Cells within a record can only be added to the record once and can
 * not be inserted into the same record multiple times.</li>
 * </ul>
 */
public interface IDatapoolRecord
{
	
	/**
	 * The number of cells available in a record.
	 * 
	 * @return	The number of cells available.
	 */
	public int getCellCount();
	
	/**
	 * The value for the cell in the current record and
	 * the specified variable is returned.  The variable is located by the
	 * specified zero based index.  A{@link DatapoolException} is thrown if 
	 * the name does not match a variable name in the record.
	 * 
	 * @param	cellIndex		The zero based cell index.
	 * @return	The value for the cell in the current record and
	 *			the specified variable.
	 */
	public IDatapoolCell getCell(int cellIndex);
	
	/**
	 * The value for the cell in the current record and
	 * the specified variable is returned.  The variable is located by the
	 * specified name, which must match a variable name in the record.  A
	 * {@link DatapoolException} is thrown if the name does not match a variable
	 * name in the record.
	 * 
	 * @param	variableName	The variable name to identify the cell by.
	 * @return	The value for the cell in the current record and
	 *			the specified variable name.
	 */
	public IDatapoolCell getCell(String variableName);

	/**
	 * Access to the {@link IDatapoolEquivalenceClass equivalenceClass} that is the 
	 * parent of this record.
	 * 
	 * @return	The {@link IDatapoolEquivalenceClass equivalenceClass} that is the 
	 * 			parent of this record.
	 */
	public IDatapoolEquivalenceClass getEquivalenceClass();
	
}
