/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: StaticParserWrapper.java,v 1.32 2005/04/20 18:19:17 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.logging.adapter.Adapter;
import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IStatus;
import org.eclipse.hyades.logging.adapter.config.outputters.AdapterLogOutputter;
import org.eclipse.hyades.logging.adapter.impl.SensorStatus;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.core.SerializationException;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.logging.parsers.ParserConstants;
import org.eclipse.hyades.logging.parsers.ParserUtilities;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class is used as in interface between the UI and the generic adapter.
 * The UI side is expecting an implementor of IParser and will call setUserInput() and parse(log).
 * This class will implement the plumbing to map the two methods above to those in the generic adapter.
 */
public class StaticParserWrapper extends Parser {
	private Adapter adapter;
	private String parserName;
	private String agentName;
	private String key;
	private String originalConfigFile = null;
	
	protected String newDirectory = null;
	protected String newFileName = null;
	protected String currentPlugin;

	// Hash table to store all active loggers
	private static Hashtable activeWrappers = new Hashtable();
	
	public StaticParserWrapper() {
		// Generate a key for this instance
		key = String.valueOf(System.currentTimeMillis());
		adapter = new Adapter();
		parserName = new String(Messages.getString("HyadesGAUnknownNameTag"));
		agentName = new String(Messages.getString("HyadesGAUnknownNameTag"));

		currentPlugin = "org.eclipse.hyades.logging.adapter.config";
	}

	public static Log getLogInstance(String logKey) {
		return (Log)activeWrappers.remove(logKey);
	}
	
	/**
	 * @see org.eclipse.hyades.logging.parsers.IParser.parse(org.apache.commons.logging.Log)
	 */
	public void parse(Log argLogger) throws LogParserException {
		try {
			activeWrappers.put(key, argLogger);
			/* bugzilla 61256 start
			 * Check if there is an internal logger for the parser
			 */
			
			Log parserLogger = getParserLogger();
			if (parserLogger != null) {
				/* Create an outputter for the Adapter context that will log the Adapter messages to
                 * the internal logger.
                 */ 
				AdapterLogOutputter logOutputter = new AdapterLogOutputter();
				logOutputter.setOutLogger(parserLogger);
				adapter.setLogOutputter(logOutputter);
			}
			
			// Start the adapter to parse the log file
			adapter.start(false,false);
			
			/* Get the Adapter status */
            IStatus status = adapter.getStatus();
            
            if (status != null) {
            	
            	/* Check the Adapter status to determine if the Adatper logged any messages */
        		IStatus [] contexts = status.getChildrenStatus();
        		
        		if (contexts != null && contexts.length > 0) {
        			for (int i=0; i < contexts.length; i++) {
        				if (contexts[i] != null && 
        					contexts[i].getName().equals(Messages.getString("HyadesGAAdapterContextName"))) {
							if (contexts[i].getItemsProcessedCount() > 0) {
								/* If the Adapter logged some messages Then throw an exception 
								 * to let the Log Import wizard know some errors occurred 
								 */
								throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_PARSING_ERROR_", file_path));
								
							}
							else {
								break;
							}
        				}
        			}
        		}
            }
            /* bugzilla 61256 end */
		} catch (AdapterException e) {
			// Return the error to the log import wizard - bugzilla 61256
		 	// Get the exception message and change it to a more meaningful one for the log import user if necessary - bugzilla 61256
            String errMessage = e.getLocalizedMessage();

            if (errMessage.indexOf("IWAT0375E") != -1 || errMessage.indexOf("IWAT0376E") != -1) {
            	throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_", file_path));
            }
            else {
            	throw new LogParserException(e);
            }
		}
	}

	public void setUserInput(Hashtable table) throws LogParserException {
		// Pass the instance key to the parser
		table.put("loggerKey", key);
		
		// Create a new adapter configuration file incorporating the user input from the Log Import wizard
		String newConfigFile = null;
		try {
			newConfigFile = getNewConfigFile(table);
		}
		catch (LogParserException e) {
			throw e;
		}
		catch (Throwable e) {
			String excMessage = e.getMessage();
			if (excMessage == null || excMessage.length() == 0) {
				excMessage = e.toString();
			}
			throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_", excMessage));
		}
		
		adapter.setContextConfigPath(newConfigFile);
		adapter.setComponentConfigPath(newConfigFile);
	}
	
	/**
	 * Since this is not a real parser we don't need to implement the getName() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.getName()
	 */
	public String getName() {

		return agentName;
	}

	// 
	/**
	 * Since this is not a real parser we don't need to implement the parseNext() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.parseNext()
	 */
	public CommonBaseEvent[] parseNext() throws LogParserException {
		return null;
	}

	// Since this is not a real parser we don't need to implement the getVersion() method
	public String getVersion() {
		return null;
	}

	/**
	 * Method to get an input stream to the config file
	 * @param fileName - config file name
	 * @param table - table of user input parameters
	 * @return InputStream
	 * @throws LogParserException 
	 */	
	private InputStream getConfigInputStream(String fileName, Hashtable table) throws LogParserException
	{
		String fileSeparator = System.getProperty("file.separator");
		// Get input stream for config file relative to the current directory
		try {
			// If we have the wrong file separator in the config file path then
			if (fileName.indexOf('/') != -1 && fileSeparator != "/") {
				// Replace file separators
				fileName = fileName.replace('/', fileSeparator.charAt(0));
			}
			// Remove ./
			if (fileName.startsWith("." + fileSeparator)) {
				fileName = fileName.substring(2);
			}
			
			return new FileInputStream(fileName);
		} catch (FileNotFoundException exc) {
			
			// The user may have specified a location for the config file so use it to try to find the config file
			String fileName2 = (String) table.get(Messages.getString("HyadesGAConfigFileRootAttributeName")) + fileSeparator + fileName;
			
			try {
				// Since we have a new config file path change the global variable
				originalConfigFile = fileName2;
				return new FileInputStream(fileName2);
			}  catch (FileNotFoundException exc2) {
				
				//
				// Bug 62317
				// Try to resolve the config file using the command line list of directories
				//
				String config_paths = (String)table.get("config_path"); // get the paths containing the adapter files

				// Check to ensure config_path was set in the input table
				// If it is not then throw an appropriate exception.
				if (config_paths != null && config_paths.length() != 0) {
					StringTokenizer strtok = new StringTokenizer(config_paths, File.pathSeparator, false); // tokenize the path
					while(strtok.hasMoreTokens()) {
						String config_path = strtok.nextToken();
						
						File f = new File(config_path, fileName); // look for the adapter file using the path
						if(f.exists()) {
							try {
								table.remove("config_path"); // This is no longer required
								/* reset the originalConfigFile value because it has changed - bugzilla 61256 */
								originalConfigFile = f.getAbsolutePath();
								return new FileInputStream(f.getAbsolutePath());
							} catch (FileNotFoundException e) {
								// Should not hit here since we've already checked the existance
							}
						}
					}
				}
				// Bug 62317 ends

				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_", fileName));
			} catch (SecurityException exc2) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", fileName));
			}
		} catch (SecurityException exc) {
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", fileName));
		}
		catch (Throwable e) {
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", fileName),e);
		}
	}

	/**
	 * Method to create a new config file based on an existing config file and 
	 * a hash table of configuration parameter values
	 * @param table - hash table of configuration parameters
	 * @return String - name of new config file
	 * @throws LogParserException 
	 */		
	private String getNewConfigFile(Hashtable table) throws LogParserException
	{
		Document doc = null;

		// Resolve the version number of the log in order to load the correct config file
		String version = (String) (table.get(Messages.getString("HyadesGALogVersionTagName")));
		String config_file = null;
		
		// If no version attribute given, use default
		if(version == null)
		{
			version = Messages.getString("HyadesGADefaultLogVersionTagName");
		}

		// Get the config file defined for this version
		config_file =  (String) (table.get(version));
		
		if(config_file == null)
		{
			// Cannot find the corresponding config file, use default
			config_file = (String) (table.get(Messages.getString("HyadesGADefaultLogVersionTagName")));
			
			// If the default (which should be) is not defined, throw an exception
			if(config_file == null) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_"));
			}
		}

		// Get the input stream for the config file

		InputStream inStream = null;
		String configFileDirectory = null;
		File adapterFile;
		originalConfigFile = config_file;  // set original config file
		
		try {
			// Try to get the input stream using the plugin if we're running in eclipse

			Boolean append = Boolean.TRUE;

			Class pathClass = Class.forName("org.eclipse.core.runtime.Path");
			Constructor pathConstructor = pathClass.getConstructor(new Class[] {config_file.getClass()});
			Object path = pathConstructor.newInstance(new Object[] {config_file});
			
			Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");			
			Class pluginClass = Class.forName("org.eclipse.core.runtime.Plugin");
			
			Object plugin = platformClass.getMethod("getPlugin", new Class[] { currentPlugin.getClass()}).invoke(null, new Object[] { currentPlugin });
			
			inStream = (InputStream)(pluginClass.getMethod("openStream", new Class[] {Class.forName("org.eclipse.core.runtime.IPath"), boolean.class}).invoke(plugin, new Object[] {path, append}));
			
			/* Get the directory of the config file */
			/* bugzilla 70110 - Use Plaform.asLocalURL() to resolve the relative URL returned by Plugin.find() */
			URL url = (URL)(pluginClass.getMethod("find", new Class[] {Class.forName("org.eclipse.core.runtime.IPath")}).invoke(plugin, new Object[] {path})) ;
			adapterFile = new File(((URL)(platformClass.getMethod("asLocalURL", new Class[] { url.getClass()}).invoke(null, new Object[] { url }))).getFile()) ;
			configFileDirectory = adapterFile.getParent();			
		} catch	(ClassNotFoundException e) {
			// We must be running outside of eclipse so try to find the config
			// file relative to the current directory

			inStream = getConfigInputStream(config_file, table);
			
			// getConfigInputStream may modify originalConfigFile due to a user input parameter so use it to get the config file directory bugzilla 61256
			adapterFile = new File(originalConfigFile);
			
			// bugzilla 79014
			// Get the directory where this adapter file is located.
			// getAbsoluteFile().getParent() is used instead of getParentFile().getAbsolutePath()
			// to handle the case where the file name specified for adapterFile 
			// does not have an absolute path.  In that case getParentFile() returns null.
			configFileDirectory = adapterFile.getAbsoluteFile().getParent();

		} catch	(InvocationTargetException e) {
			// We may be running outside of eclipse but the eclipse classes may be in
			// the path so try to find the config file relative to the current directory
			// To Do:  The exception should be logged, however, in case we are running inside
			// eclipse and there was an error from the invoked method calls.
			
			inStream = getConfigInputStream(config_file, table);
			
			// getConfigInputStream may modify originalConfigFile due to a user input parameter so use it to get the config file directory
			adapterFile = new File(originalConfigFile);
			
			// bugzilla 79014
			// Get the directory where this adapter file is located.
			// getAbsoluteFile().getParent() is used instead of getParentFile().getAbsolutePath()
			// to handle the case where the file name specified for adapterFile 
			// does not have an absolute path.  In that case getParentFile() returns null.
			configFileDirectory = adapterFile.getAbsoluteFile().getParent();
	
		} catch (Throwable e) {

			// Some other exception occurred - throw a LogParserException
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", config_file), e);
		}
		
		file_path = ((String)(table.get(Messages.getString("HyadesGAInputLogFilenameTag"))));
	    File logFile = null;
	    String logFileName = null;
	    String logFileDir = null;

	    if((file_path != null) && (file_path.trim().length() > 0)){
	    	/* bugzilla 91612 - parse the file name specified as a regular expression
	    	 * within double quotes if the last element of the file path is within double quotes
	    	 */
	    	if (file_path.endsWith("\"")) {
	    		int quoteIndex = file_path.substring(0,file_path.length()-1).lastIndexOf('"');
	    		// If there is a matching double quote and the whole file path is not quoted then 
	    		// parse the file name and directory from the path
	    		if (quoteIndex > 0) {
	    			logFileName = file_path.substring(quoteIndex+1,file_path.length()-1);
	    			logFileDir = file_path.substring(0,quoteIndex);
	    			file_path = logFileDir + logFileName;
	    		}
	    	}
		    logFile = new File(file_path.trim());
		    
		    // If there were no double quotes in the file path then
		    // use the File object to parse the file name and directory from the file path.
		    if (logFileName == null) {
		    	logFileName = logFile.getName();
		    }
		    if (logFileDir == null) {
		    	logFileDir = logFile.getParent();
		    }
	    }
	    
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(inStream);
		} catch (SAXException e) {
			// bugzilla 61256 - throw an exception so user knows what happened
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", adapterFile.getAbsolutePath()), e);
		} catch (IOException e) {
			// bugzilla 61256 - throw an exception so user knows what happened
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", adapterFile.getAbsolutePath()), e);
		} catch (ParserConfigurationException e) {
			// bugzilla 61256 - throw an exception so user knows what happened
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", adapterFile.getAbsolutePath()), e);
		} catch (Exception e) {
			// bugzilla 61256 - throw an exception so user knows what happened
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", adapterFile.getAbsolutePath()), e);
		}

		/* bugzilla 88115 start */
		/**
		 * Get the context instance config - there should only be one context instance - and fill in
		 * the properties and attributes with the values from the input hash table
		 */
		String isoLanguage = (String)table.get(Messages.getString("HyadesGAISOLanguageCodeAttributeName"));
		String isoCountry = (String)table.get(Messages.getString("HyadesGAISOCountryCodeAttributeName"));
		
		if ( (isoLanguage != null && isoLanguage.length() > 0) || (isoCountry != null && isoCountry.length() > 0)) {
			NodeList contextInstanceList = doc.getElementsByTagName(Messages.getString("HyadesGAContextInstanceElementTagName"));
			int cICount = contextInstanceList.getLength();
			for (int i = 0; i < cICount; ++i)
			{
				Element cIElement = (Element) contextInstanceList.item(i);
				
				// We should always have a sensor instance but check in case an invalid
				// config file is used.
				if (cIElement != null) {
					// Get the context instance attributes
					Node isoAttribute = cIElement.getAttributeNode(Messages.getString("HyadesGAISOLanguageCodeAttributeName"));
					
					if (isoLanguage != null && isoLanguage.length() > 0) {
						if (isoAttribute != null) {
							isoAttribute.setNodeValue(isoLanguage);
						}
						else {
							cIElement.setAttribute(Messages.getString("HyadesGAISOLanguageCodeAttributeName"), isoLanguage);
						}
					}
		
					isoAttribute = cIElement.getAttributeNode(Messages.getString("HyadesGAISOCountryCodeAttributeName"));
					
					if (isoCountry != null && isoCountry.length() > 0) {
						if (isoAttribute != null) {
							isoAttribute.setNodeValue(isoCountry);
						}
						else {
							cIElement.setAttribute(Messages.getString("HyadesGAISOCountryCodeAttributeName"), isoCountry);
						}
					}
				}
			}
		}
		/* bugzilla 88115 end */
		
		/**
		 * Get the sensor config - there should only be one sensor - and fill in
		 * the properties and attributes with the values from the input hash table
		 */
		NodeList sensorList = doc.getElementsByTagName(Messages.getString("HyadesGASensorTagName"));
		int sensorCount = sensorList.getLength();
		for (int i = 0; i < sensorCount; ++i)
		{
			Element sensorElement = (Element) sensorList.item(i);
			
			// We should always have a sensor instance but check in case an invalid
			// config file is used.
			if (sensorElement != null) {
				Element sensorNode;
				Element sensorPropertyElement;
				Element sensorTypeInstance = null;
				String propValue;
				String propName;
				String newConverterCmd = null;

				Element directoryPropertyElement = null;
				Element fileNamePropertyElement = null;
				
				// Get the sensor children (properties or sensor type instances)
				NodeList sensorNodes = sensorElement.getChildNodes();
				for (int k = 0; k < sensorNodes.getLength(); k++) {
					if (sensorNodes.item(k).getNodeType() == Node.ELEMENT_NODE) {
						sensorNode = (Element) sensorNodes.item(k);
						// Process the sensor property
						if (sensorNode.getTagName().equals(Messages.getString("HyadesGAPropertyElementTagName"))) {
							sensorPropertyElement = sensorNode;

							propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"));
							propValue = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"));
							
							if (propName.equals(Messages.getString("HyadesGAdirectoryAttributeName"))) {
								// save property element so it can be set after converter processing
								directoryPropertyElement = sensorPropertyElement;
								/* There might not be a log file if the converter command generates a file.
								 * If there is a log file then set the directory property
								 */
								if (logFileDir != null) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), logFileDir);
								}									
							}
							else if (propName.equals(Messages.getString("HyadesGAfileNameAttributeName"))) {
								// save property element so it can be set after converter processing
								fileNamePropertyElement = sensorPropertyElement;
								/* There might not be a log file if the converter command generates a file.
								 * If there is a log file then set the fileName property
								 */
								if (logFileName != null) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), logFileName);
								}									
							}
							else if (propName.equals(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
								if (propValue != null && propValue.length() > 0) {
									if (newConverterCmd == null) {
										newConverterCmd = modifyConverter(propValue, configFileDirectory, table, logFile);
										// ensure the new converter command is not null
										if (newConverterCmd == null) {
											newConverterCmd = "";
										}
									}
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), newConverterCmd);
								}
							}
							else if (propName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName"))) {
								parserName = propValue;							
							}
							else {
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), propValue.trim());
								}
							}
						}
						// Process the sensor type instance
						else if (sensorNode.getTagName().equals(Messages.getString("HyadesGAStaticParserSensorTagName")) ||
								 sensorNode.getTagName().equals(Messages.getString("HyadesGASingleFileSensorTagName"))) {
							sensorTypeInstance = sensorNode;
							// Get the sensor properties for this sensor type instance
							NodeList sensorPropertyList = sensorTypeInstance.getElementsByTagName(Messages.getString("HyadesGASensorPropertyElementTagName"));
							int sensorPropertyCount = sensorPropertyList.getLength();
							for(int j = 0; j < sensorPropertyCount; ++j)
							{
								sensorPropertyElement = (Element) sensorPropertyList.item(j);
								propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGASensorPropertyNameAttributeName"));
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), propValue.trim());
								}
							}

							// Get the sensor type attributes
							NamedNodeMap sensorTypeAttributeList = sensorTypeInstance.getAttributes();
							int sensorTypeAttributeCount = sensorTypeAttributeList.getLength();
				
							for(int j = 0; j < sensorTypeAttributeCount; ++j)
							{
								Node sensorTypeAttribute = sensorTypeAttributeList.item(j);
								String attrName = sensorTypeAttribute.getNodeName();
								// Modify the attribute based on user input
								if(attrName.equals(Messages.getString("HyadesGAdirectoryAttributeName")))
								{
									/* There might not be a log file if the converter command generates a file.
									 * If there is a log file then set the directory attribute
									 */
									if (logFileDir != null) {
										sensorTypeAttribute.setNodeValue(logFileDir);
									}									
								}
								else if(attrName.equals(Messages.getString("HyadesGAfileNameAttributeName")))
								{
									/* There might not be a log file if the converter command generates a file.
									 * If there is a log file then set the fileName attribute
									 */
									if (logFileName != null) {
										sensorTypeAttribute.setNodeValue(logFileName);
									}
								}
								else if(attrName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName")))
								{
									parserName = sensorTypeAttribute.getNodeValue();
								}
								else if(attrName.equals(Messages.getString("HyadesGAconverterCmdAttributeName")))
								{
									if (newConverterCmd == null) {
									   newConverterCmd = modifyConverter(sensorTypeAttribute.getNodeValue(), configFileDirectory, table, logFile);
									   // ensure the new converter command is not null
									   if (newConverterCmd == null) {
									   		newConverterCmd = "";
									   }
									}
									sensorTypeAttribute.setNodeValue(newConverterCmd);
								}
								else
								{
									propValue = (String)table.get(attrName);
									if(propValue != null)
									{
										sensorTypeAttribute.setNodeValue(propValue.trim());
									}
								}
							}
						}
					}
				}

				// Get the sensor attributes
				NamedNodeMap sensorAttributeList = sensorElement.getAttributes();
				int sensorTypeAttributeCount = sensorAttributeList.getLength();
				
				for(int k = 0; k < sensorTypeAttributeCount; ++k)
				{
					Node sensorTypeAttribute = sensorAttributeList.item(k);
					String attrName = sensorTypeAttribute.getNodeName();
					
					// Modify the attribute based on user input
					propValue = (String)table.get(attrName);
					if(propValue != null && propValue.length() > 0)
					{
						sensorTypeAttribute.setNodeValue(propValue.trim());
					}
				}

				// Check if the directory of the log file to be parsed was updated by modifyConverter 				
				if (newDirectory != null) {
					// If so update it in the config file
					if (directoryPropertyElement != null) {
						directoryPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), newDirectory);
					}
					if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAdirectoryAttributeName"));
						sensorAttribute.setNodeValue(newDirectory);					
					}
				}
				
				// Check if the name of the log file to be parsed was updated by modifyConverter 				
				if (newFileName != null) {
					// If so update it in the config file
					if (fileNamePropertyElement != null) {
						fileNamePropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), newFileName);
					}
					if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAfileNameAttributeName"));
						sensorAttribute.setNodeValue(newFileName);					
					}	
				}
			}
		}

		/**
		 * Add the logger key to the outputter so the the outputter can write the the UI
		 * logger directly - there should only be one outputter.
		 */
		NodeList outputterList = doc.getElementsByTagName(Messages.getString("HyadesGAOutputterTagName"));
		int outputterCount = outputterList.getLength();
		for (int i = 0; i < outputterCount; ++i) {
			Element outputterElement = (Element) outputterList.item(i);
			Element outputterProperty = doc.createElement(Messages.getString("HyadesGAPropertyElementTagName"));
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), key);
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"),Messages.getString("HyadesGAStaticParserLoggerKeyName"));
			// Insert the Property node as the first child of the Outputter element otherwise we get a schema validation error
			outputterElement.insertBefore(outputterProperty,outputterElement.getFirstChild());
		}

		// Save the resulting xml

		// Create a temporary file for the new adapter configuration
		File newFile = null;
		
		try {
			
			// Try to create a temporary file in the plugin's workspace directory (e.g. <workspace>/.metadata/plugins/org.eclipse.hyades.logging.adapter.config) 
			// for the new adapter configuration

			Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");
			
			Method getPluginStateLocationMethod = platformClass.getMethod("getPluginStateLocation",new Class[]{Class.forName("org.eclipse.core.runtime.Plugin")});
			
			Object iPathObject =  getPluginStateLocationMethod.invoke(null,new Object[]{AdapterConfigPlugin.getPlugin()});
			
			Class iPathClass = iPathObject.getClass();
			
			Method toOSStringMethod = iPathClass.getMethod("toOSString",null);
			
			/* bugzilla 84701
			 * Changing the creation of a new GLA file to use File.createTempFile
			 * which is more threadsafe than generating the file name with System.currentTimeMillis().
			 */
			File dirpath = new File((String)(toOSStringMethod.invoke(iPathObject,null)));
			newFile = File.createTempFile("GLA", ".adapter", dirpath);
		}
		catch (Exception e) {
			// We are likely not running in an Eclipse environment so try to create the
			// temporary file in the system's temp directory
			try {		
				newFile = File.createTempFile("GLA", ".adapter");
			}
			catch (Throwable e2){
				// If a temporary file cannot be generated in the OS tmp directory
				// try creating it in the same directory as the current config file.
				try {
					File oldFile = new File(config_file);
					String tmpFileName = new String("GLA" + System.currentTimeMillis() + ".adapter");
					newFile = new File(oldFile.getParent(), tmpFileName);
				}
				catch (Throwable e3) {
					throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Save_ERROR_","GLA.adapter", e3.toString()),e3);
				}
			}
		}
		catch (Throwable e) {
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Save_ERROR_","GLA.adapter", e.toString()),e);
		}
		
		// Ensure the temporary file gets deleted
		newFile.deleteOnExit();
		
		//Save the configuration in the temporary file:
		try {
			XmlUtility.serialize(doc, newFile);
        } 
		catch (SerializationException s) {
		    throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Save_ERROR_",newFile.getAbsolutePath(), s.getMessage()),s);
        }

		return newFile.getAbsolutePath();
	}

	/**
	 * Method to perform any modifications to the converter string based on user
	 * input from the UI
	 * @param converter  - converter string to be updated
	 * @param configFileDirectory  - the absolute path to the directory of the config file 
	 * @param table  - of user input values
	 * @param logFile - log file to be parsed
	 * @return String - new converter string
	 */
	protected String modifyConverter(String converter, String configFileDirectory, Hashtable table, File logfile) {
		String tempWindowsFile = "windows.txt";
		String newConverter = converter;
        // If this is the Windows application/security/system event log parser then create the converter command
        // Use the config file directory to determine if this is a Windows application/security/system event log parser
        if (configFileDirectory.indexOf("Windows") != -1 && ((configFileDirectory.indexOf("application") != -1) || (configFileDirectory.indexOf("security") != -1) || (configFileDirectory.indexOf("system") != -1))) {

            // Create a temporary file to hold the output of the converter.
            File tempFile;
            try {
                tempFile = File.createTempFile("WindowsLogParser", ".tmp");
            }
            catch (Exception e) {
                // If a temporary file cannot be generated, create one in the same directory as the
                // config file.
                tempFile = new File(configFileDirectory + ParserConstants.FILE_SEPARATOR + tempWindowsFile);
            }
            try {
                // Ensure the temporary files gets deleted
                tempFile.deleteOnExit();
            }
            catch (Exception e) {
                // Ignore this exception
            }

            String windowsEventLog = "application";

            if (configFileDirectory.indexOf("security") != -1) {
                windowsEventLog = "security";
            }
            else if (configFileDirectory.indexOf("system") != -1) {
                windowsEventLog = "system";
            }

            // Create the new converter command		
            newConverter = "\"".concat(configFileDirectory).concat(ParserConstants.FILE_SEPARATOR).concat("eventlogreader.exe\" \"").concat(windowsEventLog).concat("\" \"").concat(tempFile.getAbsolutePath()).concat("\"");

            // set the new file to parse the text file that will be the output of the converter command
            newFileName = tempFile.getName();
            newDirectory = tempFile.getParent();
        }
		return newConverter;
	}
	
	/**
	 * Stop the parsing by stopping the adapter instance.
	 */
	public void stop() {
		adapter.stop();
	}
	
	/**
	 * Get the parsing status
	 * @return String - status string in the form "total=nnnnn processed=nnnnn"
	 *
	 */
	public String getStatus() {
		String totalSize = "0";
		String totalProcessed = "0";
		
		// Get the adapter status
        IStatus status = adapter.getStatus();
        
        if (status != null) {
        	
        	// Get the context statuses
    		IStatus [] contexts = status.getChildrenStatus();
    		
    		if (contexts != null && contexts.length > 1) {
    			// There should be two contexts, the adapter logging context and the real parsing context
    			// The staus of the real parsing context is required so we'll skip the adapter logging context
    			// which is first in the array
    			for (int i=1; i < contexts.length; i++) {
    				if (contexts[i] != null) {
        				IStatus cstatus = contexts[i];
        				
        				// Get the component statuses 
    					IStatus [] components = cstatus.getChildrenStatus();
    					if (components != null && components.length > 0) {
    						// Loop through the components until we find the sensor component 
    						// - it should be the first one
    						for (int j=0; j < components.length; j++) {
    							IStatus cpstatus = components[j];

    							// If this is the sensor status then set the status values
    							if (cpstatus instanceof SensorStatus) {
    								if (cpstatus.isActive()) {
    									totalSize = Long.toString(((SensorStatus)cpstatus).getTotalSize());
    									totalProcessed = Long.toString(((SensorStatus)cpstatus).getProcessedSize());
    								}
    								else {
    									// If the sensor is done set total processed size = total size to signal parsing is finished
    									totalSize = Long.toString(((SensorStatus)cpstatus).getTotalSize());
    									totalProcessed = totalSize;
    								}
    								break;
    							}
    						}
    					}
    				}
    			}
    		}
        	
        }
        
        // return the status string
		return new String("total=" + totalSize + " processed=" + totalProcessed);
	}
}
