/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestSuiteTreeModel.java,v 1.3 2005/02/16 22:21:48 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.manual.runner.ui.tree;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.JTree;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import org.eclipse.hyades.test.manual.runner.model.Action;
import org.eclipse.hyades.test.manual.runner.model.IActionOwner;
import org.eclipse.hyades.test.manual.runner.model.IVerdictListener;
import org.eclipse.hyades.test.manual.runner.model.NamedElement;
import org.eclipse.hyades.test.manual.runner.model.TestInvocation;
import org.eclipse.hyades.test.manual.runner.model.TestSuite;
import org.eclipse.hyades.test.manual.runner.model.util.ModelUtil;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class TestSuiteTreeModel 
implements TreeModel, IVerdictListener
{
	private List modelListeners;
	private JTree tree;
	
	public TestSuiteTreeModel()
	{
		TestInvocation.addListener(this);
		modelListeners = new ArrayList();
	}
	
	public void dispose()
	{
		modelListeners.clear();
		tree = null;
	}
	
	public void handleVerdict(TestInvocation testInvocation)
	{
		fireTreeNodesChanged(testInvocation, getTreePath(testInvocation).getPath(), null, null);
		select(ModelUtil.DEFAULT_EXECUTION_MANAGER.next());
	}
	
	public void select(NamedElement namedElement)
	{
		if(namedElement == null)
			namedElement = ModelUtil.DEFAULT_EXECUTION_MANAGER.getRoot();
			
		TreePath path = TestSuiteTreeModel.getTreePath(namedElement);
		getTree().setSelectionPath(path);
		getTree().scrollPathToVisible(path);
	}

	public static TreePath getTreePath(NamedElement namedElement)
	{
		if(namedElement == null)
			return null;
			
		List objectPath = new ArrayList();
		getTreePath(objectPath, namedElement);
		
		return new TreePath(objectPath.toArray());
	}
	
	private static void getTreePath(List objectPath, Object child)
	{
		if(child == null)
			return;
			
		objectPath.add(0, child);
		if(child instanceof Action)
			getTreePath(objectPath, ((Action)child).getOwner());
		
		else if(child instanceof TestSuite)
			getTreePath(objectPath, ((TestSuite)child).getTestInvocation());		
	}
		
	/**
	 * @see TreeModel#addTreeModelListener(TreeModelListener)
	 */
	public void addTreeModelListener(TreeModelListener l)
	{
		if((l != null) && (!modelListeners.contains(l)))
			modelListeners.add(l);
	}

	/**
	 * @see TreeModel#removeTreeModelListener(TreeModelListener)
	 */
	public void removeTreeModelListener(TreeModelListener l)
	{
		modelListeners.remove(l);
	}

	/**
	 * @see TreeModel#getChild(Object, int)
	 */
	public Object getChild(Object parent, int index)
	{
		if(parent instanceof IActionOwner)
			return ((IActionOwner)parent).getActions().get(index);
			
		if(parent instanceof TestInvocation)
		{
			if(index == 0)
			{
				TestInvocation testInvocation = (TestInvocation)parent;
				if((testInvocation.getTest() != null) && (testInvocation.getTest() instanceof TestSuite))
					return testInvocation.getTest();
			}
		}
		
		return null;
	}

	/**
	 * @see TreeModel#getChildCount(Object)
	 */
	public int getChildCount(Object parent)
	{
		if(parent instanceof IActionOwner)
			return ((IActionOwner)parent).getActions().size();

		if(parent instanceof TestInvocation)
		{
			TestInvocation testInvocation = (TestInvocation)parent;
			if((testInvocation.getTest() != null) && (testInvocation.getTest() instanceof TestSuite))
				return 1;
		}
			
		return 0;
	}

	/**
	 * @see TreeModel#getIndexOfChild(Object, Object)
	 */
	public int getIndexOfChild(Object parent, Object child)
	{
		if(parent instanceof IActionOwner)
			return ((IActionOwner)parent).getActions().indexOf(child);

		if(parent instanceof TestInvocation)
		{
			TestInvocation testInvocation = (TestInvocation)parent;
			if(testInvocation.getTest() == child)
				return 0;
		}

		return -1;
	}

	/**
	 * @see TreeModel#getRoot()
	 */
	public Object getRoot()
	{
		return ModelUtil.DEFAULT_EXECUTION_MANAGER.getRoot();
	}

	/**
	 * @see TreeModel#isLeaf(Object)
	 */
	public boolean isLeaf(Object node)
	{
		return (getChildCount(node) == 0);
	}

	/**
	 * @see TreeModel#valueForPathChanged(TreePath, Object)
	 */
	public void valueForPathChanged(TreePath path, Object newValue)
	{
	}

	protected void fireTreeNodesChanged(Object source, Object[] path, int[] childIndices, Object[] children)
	{
		TreeModelEvent e = new TreeModelEvent(source, path, childIndices, children);
		for(Iterator i = modelListeners.iterator(); i.hasNext();)
		{
			TreeModelListener modelListener = (TreeModelListener) i.next();
			modelListener.treeNodesChanged(e);
		}
	}

	/**
	 * Gets the tree.
	 * @return Returns a JTree
	 */
	JTree getTree()
	{
		return tree;
	}

	/**
	 * Sets the tree.
	 * @param tree The tree to set
	 */
	void setTree(JTree tree)
	{
		this.tree = tree;
		tree.setModel(this);
	}
}