/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGReport.java,v 1.7 2005/04/22 19:10:04 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.sample.report.svg;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;

import org.eclipse.emf.common.util.EList;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.util.ExecutionUtil;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.hyades.ui.report.IReportGenerator;
import org.eclipse.hyades.ui.sample.HyadesSamplePlugin;
import org.eclipse.hyades.ui.sample.report.svg.DayHelper.DataDay;
import org.eclipse.hyades.ui.sample.report.svg.DayHelper.GraphDay;
import org.eclipse.hyades.ui.sample.svg.generator.GraphicDocumentProperties;
import org.eclipse.hyades.ui.sample.svg.generator.SVGLineChart;

/**
 * This class generates an SVG report for a set of test case execution results.
 * 
 * <p>For the selected test cases, this class will generate a report that
 * presents a graph with the number of execution attempts, and how many succeeded, failed or 
 * soft failed, ordered by day.
 * 
 * <p>To use the report, you must integrate the generator into Component Test. A quick way to do this is by
 * registering it in the <b>Local report definition</b> preference page.  Please refer to the 
 * documentation for the full details on how to do this.  The class name of this report is 
 * org.eclipse.hyades.ui.sample.report.svg.SVGReport and the classpath is the full path - relative  
 * to the file system - of the <b>bin</b> directory  of this project.
 * 
 * <p><b>IMPORTANT:</b> Read the comments for the method getPreferencePageDirectory() if
 * you are experiencing problems trying to open the SVG preference page.
 */

public class SVGReport implements IReportGenerator
{
	private String usedClasspath = null;
	
	/**
	 * If the report is registered into the Component Test tool by the preference page
	 * it will be difficult for the report generator to figure it out where it is 
	 * located in the file system.  So, in this case, this method will be invoked
	 * 
	 * @param String The classpath entered by the user in the preference page.
	 */
	public void setUsedClasspath(String fullPath)
	{
		this.usedClasspath = fullPath;
	}

	/**
	 * Before the generate method invocation, the Component Test will execute this
	 * method in order to retrieve the extension of the resource that will be created
	 * for the report.
	 * 
	 * @param ISelection The selection
	 * @return String The resource extension
	 */
	public String getReportFileExtension(ISelection selection)
	{
		return "html";
	}

	/**
	 * Check if the report generator is available to the given selection.
	 * 
	 * @param selection The selection.
	 * @return true if the report generator is applicable to the given selection.
	 */
	public boolean isAvailable(ISelection selection)
	{
		// This report is available only if more than 1 testcases are selected
		if (selection instanceof IStructuredSelection){
			int numberOfTestSuites = 0;
			Iterator iter = ((IStructuredSelection)selection).iterator();
			while (iter.hasNext()) {
				Object s = iter.next();
				if (s instanceof TPFTestSuite || s instanceof ITestSuiteProxyNode) {
					numberOfTestSuites += 1;
				}

				if (numberOfTestSuites >= 2) {
					return true;
				}
				continue;
			}
		}
		return false;
	}


	/**
	 * This method will be invoked to generate the report for the given selection.
	 * 
	 * <p>The <code>reportFile</code> argument is the Eclipse resource that 
	 * <b>will</b> be created by Component Test using the InputStream returned by the method.  
	 * The implementors of this method should NOT create any resource with this file name.
	 * 
	 * @param IFile The report file that <b>will</b> be created by the Component Test Plugin
	 * @param ISelection The selection
	 * @return InputStream The content of the report
	 * @exception Exception Any exception thrown by this method will be treated.
	 */
	public InputStream generate(IFile reportFile, ISelection selection) throws Exception
	{
		if (selection == null) {
			return null;
		}

		// Find the execution results of the selected testcases
		Map executionResults = ExecutionUtil.findExecutionResults(((IStructuredSelection)selection).toList());
		
		// Resets the GraphDay static information
		GraphDay.reset();
		
		// Retrieve the test status per day
		ArrayList dataDays = analyseTestcaseInstances(executionResults);
		if(dataDays.isEmpty())
			return null;
		
		// Sort the days, fills the missing days with a previous information and 
		// carries the results from previous days
		GraphDay[] graphDays = createGraphDays(dataDays);

		// Loads the statistics from file with the expected execution values
		int numberOfTestcaseInstances = loadExpectedExecutionValues(reportFile.getParent(), graphDays);
		
		// Counts the number of testcase instances with data for the graph
		int numberOfTestcaseInstancesWithData = countTestcaseWithData(executionResults);
		if(numberOfTestcaseInstances < numberOfTestcaseInstancesWithData)
			numberOfTestcaseInstances = numberOfTestcaseInstancesWithData;
		
		// Returns the report content as an InputStream and creates the
		// svg and css files
		return generate(reportFile, graphDays, numberOfTestcaseInstances);
	}
	

	//
	// Methods for collecting the testcase instances data 
	//	
		
		
	/**
	 * Analyses the test case instances array collecting the daily status.
	 * 
	 * <p>Note: Only the last execution of a test case instance of a particular day will
	 * be considered.
	 * 
	 * @param TestcaseInstance[] the TestcaseInstances to be analysed.
	 * @return ArrayList List of DataDay - the testcase instances status per day;
	 */
	protected ArrayList analyseTestcaseInstances(Map executionResults)
	{
		Hashtable dataDayByDay = new Hashtable();

		if (executionResults != null) {
			Iterator i = executionResults.values().iterator();
			while(i.hasNext()) {
				List executionResultsList = ((List)(i.next()));
				Hashtable lastExecutionByDay = getLastExecutionByDay(executionResultsList);
				for(Iterator iter = lastExecutionByDay.keySet().iterator(); iter.hasNext();)
				{
					String day = (String)iter.next();
					TPFExecutionResult executionResult = (TPFExecutionResult)lastExecutionByDay.get(day);

					DataDay dataDay = (DataDay)dataDayByDay.get(day);
					if(dataDay == null)
					{
						dataDay = new DataDay(day);
						dataDayByDay.put(day, dataDay);
					}
					dataDay.addVerdict((TPFTestSuite)executionResult.getTest(), executionResult.getVerdict());
				}
			}
		}

		return new ArrayList(dataDayByDay.values());
	}

	/**
	 * 	Gets the day from the double value of the ExecutionAttempt timestampt (yyyyMMddHHmmss)
	 * 
	 * @param long Timestamp
	 * @return String The day and time in the yyyyMMddHHmmss format
	 */
	protected String getDayTime(long timestamp)
	{
		return new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(timestamp));
	}
	
	protected Hashtable getLastExecutionByDay(List executionResultsList)
	{
		Hashtable executionByDayTime = new Hashtable();
		Hashtable latestTimeByDay = new Hashtable();

        if (executionResultsList != null) {
			Iterator i = executionResultsList.iterator();
			while(i.hasNext()) {
				TPFExecutionResult executionResult = (TPFExecutionResult)i.next();
				if (executionResult == null) {
					continue;					} 
				TPFExecutionHistory executionHistory = executionResult.getExecutionHistory();
				if (executionHistory == null) {
					continue;
				}
				EList events = executionHistory.getExecutionEvents();
				if ((events == null) || events.isEmpty()) {
					continue;
				}
				String dayTime = getDayTime(((TPFExecutionEvent)events.get(0)).getTimestamp());
				
				if(dayTime == null)
					continue;
					
				String day = dayTime.substring(0, "yyyyMMdd".length());
				String time = dayTime.substring("yyyyMMdd".length());			
	
				executionByDayTime.put(dayTime, executionResult);
				
				String latestTime = (String)latestTimeByDay.get(day);
				if((latestTime == null) || (time.compareTo(latestTime) > 0)) {
					latestTimeByDay.put(day, time);
				}
			}
		}

		Hashtable lastExecutionByDay = new Hashtable();
		for (Iterator iter = latestTimeByDay.keySet().iterator(); iter.hasNext();)
		{
			String day = (String)iter.next();
			String latestTime = (String)latestTimeByDay.get(day);
			lastExecutionByDay.put(day, executionByDayTime.get(day+latestTime));
		}
		
		return lastExecutionByDay;
	}
	


	/**
	 * This methods executes 3 operations:
	 * <br>1.Sorts the execution days
	 * <br>2.Keep the last status of a testcase in order to repeat it on the next
	 * day if a testcase instance was not executed.
	 * <br>3.Navigates the ArrayList looking for missing days in the key collection
	 * adding the previous status value
	 * 
	 * @param Collection the testcase instance status per day.
	 * @param GraphDay[] the sorted daily status
	 */
	protected GraphDay[] createGraphDays(ArrayList dataDays)
	{
		DataDay[] dataDayArray = (DataDay[])dataDays.toArray(new DataDay[dataDays.size()]);
		Arrays.sort(dataDayArray);
		
		ArrayList graphDays = new ArrayList();
		Hashtable lastStatusByTestcaseInstance = new Hashtable();

		for(int i=0,max=dataDayArray.length; i<max; i++)
		{
			for(Iterator iter = dataDayArray[i].testcaseIterator(); iter.hasNext();)
			{
				TPFTestSuite testSuite = (TPFTestSuite)iter.next();
				lastStatusByTestcaseInstance.put(testSuite, dataDayArray[i].getVerdict(testSuite));
			}
			
			GraphDay graphDay = new GraphDay(dataDayArray[i].getDay());
			graphDays.add(graphDay);
			for(Iterator iter = lastStatusByTestcaseInstance.values().iterator(); iter.hasNext();)
			{
				TPFVerdict executionVerdict = (TPFVerdict)iter.next();
				graphDay.addStatistic(GraphDay.ATTEMPT, 1);
				if (executionVerdict == TPFVerdict.ERROR_LITERAL) {
					graphDay.addStatistic(GraphDay.ERROR, 1);
				}
				else if (executionVerdict == TPFVerdict.PASS_LITERAL) {
					graphDay.addStatistic(GraphDay.PASS, 1);
				}
				else if (executionVerdict == TPFVerdict.FAIL_LITERAL) {
					graphDay.addStatistic(GraphDay.FAIL, 1);
				}
				else if (executionVerdict == TPFVerdict.INCONCLUSIVE_LITERAL) {
					graphDay.addStatistic(GraphDay.INCONCLUSIVE, 1);
				}
			}
			
			if(i < max-1)
			{
				GraphDay repeatedGraphDay = ((GraphDay)graphDay.clone()).incrementDay();
				
				//Repeats the current GraphDay for all the days that don't have
				//data
				while(repeatedGraphDay.getDay().compareTo(dataDayArray[i+1].getDay()) < 0)
				{
					graphDays.add(repeatedGraphDay.clone());
					repeatedGraphDay.incrementDay();
				}
			}
		}
		
		return (GraphDay[])graphDays.toArray(new GraphDay[graphDays.size()]);
	}
		
	/**
	 * Loads the expected results for the testcase instances execution from the 
	 * <b>expected_execution.xml</b> file in the specified container (project or folder).
	 * 
	 * <p>Expected Result File format:
	 * 
	 * 	<ComponentTest_ExpectedExecution>
	 * 		<day index="1" attempt="3" pass="2" fail="1" error="0" inconclusive="3"/>
	 * 		<day index="2" attempt="5" pass="2" fail="2" error="1" inconclusive="2"/>
	 * 		<day index="4" attempt="8" pass="5" fail="2" error="1" inconclusive="0"/>
	 * 	</ComponentTest_ExpectedExecution>
	 * 
	 * <b>Each day entry specifies the values for a particular day.  
	 * In the example above the day 3 will have the same information of the day 2.
	 * 
	 * @param IContainer the container (project or folder)
	 * @param ArrayList the list of dailyStatistics where the exepected results will be loaded.
	 * @return int the greater number of testcase instances executions in one day;
	 */
	protected int loadExpectedExecutionValues(IContainer container, GraphDay[] graphDays)
	{
		IFile file = container.getFile(new Path("expected_execution.xml"));
		if((file == null) || (!file.exists()))
			return 0;
			
		Element topElement = null;
		Document document = XMLUtil.getXmlDom(file.getLocation().makeAbsolute().toOSString());
		NodeList list = document.getElementsByTagName("ComponentTest_ExpectedExecution");
		if((list.getLength() != 1) || (!(list.item(0) instanceof Element)))
			return 0;
		
		int maxTestcaseInstaceCount = 0;
		
		int attempt = 0;
		int error = 0;
		int pass = 0;
		int fail = 0;
		int inconclusive = 0;
		int oldIndex = 0;
		
		topElement = (Element)list.item(0);
		list = topElement.getElementsByTagName("day");
		for(int i=0, max=list.getLength(); i<max; i++)
		{
			Object item = list.item(i);
			if(item instanceof Element)
			{
				Element dayElement = (Element)item;
				int index = XMLUtil.getIntValue(dayElement, "index");
				if(index <= 0)
					continue;
				
				index--;	
				if(index >= graphDays.length)
					break;
					
				if(index < graphDays.length)
				{
					for(int j=oldIndex; j<index; j++)
					{
						graphDays[j].addStatistic(GraphDay.EXPECTED_ATTEMPT, attempt);
						graphDays[j].addStatistic(GraphDay.EXPECTED_ERROR, error);
						graphDays[j].addStatistic(GraphDay.EXPECTED_PASS, pass);
						graphDays[j].addStatistic(GraphDay.EXPECTED_FAIL, fail);
						graphDays[j].addStatistic(GraphDay.EXPECTED_INCONCLUSIVE, inconclusive);
					}

					attempt = XMLUtil.getIntValue(dayElement, "attempt");
					if(maxTestcaseInstaceCount < attempt)
						maxTestcaseInstaceCount = attempt;
					
					pass = XMLUtil.getIntValue(dayElement, "error");
					if(maxTestcaseInstaceCount < error)
						maxTestcaseInstaceCount = error;

					pass = XMLUtil.getIntValue(dayElement, "pass");
					if(maxTestcaseInstaceCount < pass)
						maxTestcaseInstaceCount = pass;

					fail = XMLUtil.getIntValue(dayElement, "fail");
					if(maxTestcaseInstaceCount < fail)
						maxTestcaseInstaceCount = fail;

					inconclusive = XMLUtil.getIntValue(dayElement, "inconclusive");
					if(maxTestcaseInstaceCount < inconclusive)
						maxTestcaseInstaceCount = inconclusive;
					
					oldIndex = index;
				}
			}
		}			

		for(int j=oldIndex; j<graphDays.length; j++)
		{
			graphDays[j].addStatistic(GraphDay.EXPECTED_ATTEMPT, attempt);
			graphDays[j].addStatistic(GraphDay.EXPECTED_ERROR, error);
			graphDays[j].addStatistic(GraphDay.EXPECTED_PASS, pass);
			graphDays[j].addStatistic(GraphDay.EXPECTED_FAIL, fail);
			graphDays[j].addStatistic(GraphDay.EXPECTED_INCONCLUSIVE, inconclusive);
		}
		
		return maxTestcaseInstaceCount;
	} 
		
	protected int countTestcaseWithData(Map executionResults)
	{
		HashSet set = new HashSet();			
		Iterator i = executionResults.values().iterator();
		while(i.hasNext()) {
			Iterator ii = ((List)(i.next())).iterator();
			while(ii.hasNext()) {
				TPFExecutionResult executionResult = (TPFExecutionResult)ii.next();
				Object testSuite = executionResult.getTest();
				if (testSuite instanceof TPFTestSuite) {
					set.add(testSuite);
				}
			}
		}
		return set.size();
	}

		
	//
	// Methods for generating the report content and extra files
	//	
		

	/**
	 * Generates the report content and the required files - "report".svg and "report".css
	 * where "report" is the report file full path minus the extension.
	 * 
	 * @param IFile The report file that <b>will</b> be created by the Component Test Plugin
	 * @param ArrayList the sorted daily statistics
	 * @return InputStream the report content
	 */ 
	protected InputStream generate(IFile reportFile, GraphDay[] graphDays, int numberOfTestcaseInstances)
	throws Exception
	{
		// these are not the best values
		String width = "750";
		String height = "500";

		// Defines a new color palette and initializes it with white in all the positions
		ArrayList colourPalette = new ArrayList(); 

		// creates a Document by collecting information from the Component Test objects
		Document newDoc = generateDataDocument(graphDays, numberOfTestcaseInstances, colourPalette);
		if(newDoc == null)
			return null;

		// turn the DOM into an SVG chart
		GraphicDocumentProperties properties = new GraphicDocumentProperties();
		properties.setDataDocument(newDoc);
		properties.setGraphicWidth(width);
		properties.setGraphicHeight(height);

		String preferenceDir = getSVGPreferencePageDirectory();
		if (preferenceDir != null)
			properties.setResourcesFolder("file://" + preferenceDir.replace('\\', '/') + "/");

		properties.addGraphicColorPalette("Test Status", (String[])colourPalette.toArray(new String[colourPalette.size()]));
		properties.setGraphicColorPaletteName("Test Status");

		// create the line chart SVG
		SVGLineChart svgLineChart = new SVGLineChart();
		Document generatedDocument = svgLineChart.generateGraphicDocument(properties);

		// write the SVG to a file
		String svgFileName = createFile(reportFile, "svg", svgLineChart.serializeGeneratedDocumentToString(generatedDocument)).getName();

		// create the HTML to return to caller
		StringBuffer sb = new StringBuffer();
		sb.append("<html>\n").append("<head>\n").append("<title>").append(getUserText("chart.Title")).append("</title>").append("</head>\n");
		sb.append("<body>\n").append("<object type=\"image/svg+xml\" data=\"");
		sb.append(svgFileName).append("\" ").append("width=\"").append(width).append("\" ");
		sb.append("height=\"").append(height).append("\" ");
		sb.append("name=\"mySvgDocument\"");
		sb.append("/>\n");
		sb.append("</body>").append("</html>");

		return new ByteArrayInputStream(sb.toString().getBytes());
	}
	
	/**
	 * Generates the Document that will be used as the input for the
	 * SVG widgets API to produce the report graph.
	 * 
	 * @param GraphDay[] the sorted daily statistics
	 * @param int the number of Testcase Instances
	 * @param ArrayList the colour palette
	 * @return Document the graph input document
	 */
	protected Document generateDataDocument(GraphDay[] graphDays, int numberOfTestcaseInstances, ArrayList colourPalette)
	throws Exception
	{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document document = builder.newDocument();

		// create the root element, i.e. configuration
		Element root = document.createElement("configuration");
		document.appendChild(root);
		
		root.setAttribute("title", getUserText("chart.Title"));
		root.setAttribute("legendtitle", getUserText("chart.Legend"));
		root.setAttribute("timestampPrefix", getUserText("chart.timestampPrefix"));
		

		// Create the element dataUpdate
		Element dataUpdateElement = document.createElement("dataUpdate");
		root.appendChild(dataUpdateElement);
		Date dateNow = new Date();
		SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");	
		dataUpdateElement.setAttribute("timestamp", dateFormatter.format(dateNow));

		/* Create a list of dataset elements (each dataset correponds to a line in the graph) inside the 
		 * dataUpdate element.  If the line is added then the colourPalette will be changed so it can be
		 * drawn in a specific color,
		 */
		int position = 0;			
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.expectedAttempt"), GraphDay.EXPECTED_ATTEMPT, graphDays))
		{
			colourPalette.add("#D3D3D3"); //grey
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.expectedPass"), GraphDay.EXPECTED_PASS, graphDays))
		{
			colourPalette.add("#ADFF2F"); //light green
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.expectedFail"), GraphDay.EXPECTED_FAIL, graphDays))
		{
			colourPalette.add("#DB7093"); //light red
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.expectedError"), GraphDay.EXPECTED_ERROR, graphDays))
		{
			colourPalette.add("#DB9993"); //light orange
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.expectedInconclusive"), GraphDay.EXPECTED_INCONCLUSIVE, graphDays))
		{
			colourPalette.add("#FAFAD2"); //light yellow
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.attempt"), GraphDay.ATTEMPT, graphDays))
		{
			colourPalette.add("#000000"); //black
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.pass"), GraphDay.PASS, graphDays))
		{
			colourPalette.add("#008000"); //green
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.fail"), GraphDay.FAIL, graphDays))
		{
			colourPalette.add("#FF0000"); //red
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.error"), GraphDay.ERROR, graphDays))
		{
			colourPalette.add("#FFAA00"); //orange
			position++;
		}
		if(addLineData(document, dataUpdateElement, position, getUserText("chart.inconclusive"), GraphDay.INCONCLUSIVE, graphDays))
		{
			colourPalette.add("#FFFF00"); //yellow
			position++;
		}
		
		// Create the categorization element which contains the information
		// related X axis
		createXData(document, dataUpdateElement, graphDays);
		
		// Create the categorization element which contains the information
		// related Y axis
		createYData(document, dataUpdateElement, numberOfTestcaseInstances);

		return document;
	}
	
	/**
	 * Adds a line data to the document.  The line will represent one of the values
	 * of the execution statistics.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param int the position of the line
	 * @param String the name of the line
	 * @param int the type of statistic value will be represented on the line
	 * @param DayStatistic[] the sorted daily statistics
	 * @return boolean if the line data was added
	 */
	protected boolean addLineData(Document document, Element dataUpdateElement, int position, String name, int statisticFlag, GraphDay[] graphDays)
	{
		if(!GraphDay.hasStatistic(statisticFlag))
			return false;
			
		Element dataset = document.createElement("dataSet");
		dataUpdateElement.appendChild(dataset);
		dataset.setAttribute("label",name);
		dataset.setAttribute("position", new Integer(position).toString());
		dataset.setAttribute("dataRange1Ref", "day");
		dataset.setAttribute("dataRange2Ref", "count");
		
		Element data = document.createElement("dataPoint");
		dataset.appendChild(data);	
		data.setAttribute("value1", "0");
		data.setAttribute("value2", "0");

		for(int i = 0; i < graphDays.length; i++)
		{
			data = document.createElement("dataPoint");
			dataset.appendChild(data);
			
			data.setAttribute("value1", Integer.toString(i+1));
			data.setAttribute("value2", Integer.toString(graphDays[i].getStatistic(statisticFlag)));
		}
		
		return true;
	}
	
	/**
	 * Creates the data that is presented at the X axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param DayStatistic[] the sorted daily statistics
	 */
	protected void createXData(Document document, Element dataUpdateElement, GraphDay[] graphDays)
	{
		Element dataRangeElement = document.createElement("dataRange");
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Title information
		StringBuffer xTitle = new StringBuffer();
		xTitle.append(getUserText("chart.Xtitle"));
		xTitle.append(" ").append(graphDays[0].getFormattedDay());
		dataRangeElement.setAttribute("id", "day");
		dataRangeElement.setAttribute("location", "S");
		dataRangeElement.setAttribute("label", xTitle.toString());
		
		// Adds a "0"  day
		Element segmentMarkerElement = document.createElement("segmentMarker");
		dataRangeElement.appendChild(segmentMarkerElement);
		segmentMarkerElement.setAttribute("value", "0");
		segmentMarkerElement.setAttribute("label", "0");

		// Add the X axis points
		for(int i = 0; i < graphDays.length; i++)
		{
			segmentMarkerElement = document.createElement("segmentMarker");
			dataRangeElement.appendChild(segmentMarkerElement);
			String day = graphDays[i].getDayOfMonth();

			segmentMarkerElement.setAttribute("value", Integer.toString(i+1));
			segmentMarkerElement.setAttribute("label", day);
		}
	}

	/**
	 * Creates the data that is presented at the Y axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 */
	protected void createYData(Document document, Element dataUpdateElement, int numberOfTestcaseInstances)
	{
		Element dataRangeElement = document.createElement("dataRange");
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Define the Y axis range
		dataRangeElement.setAttribute("id", "count");
		dataRangeElement.setAttribute("location", "W");
		dataRangeElement.setAttribute("label", getUserText("chart.Ytitle"));

		// Add the Y axis points
		for (int i = 0; i <= numberOfTestcaseInstances; i++)
		{
			Element segmentMarkerElement = document.createElement("segmentMarker");
			dataRangeElement.appendChild(segmentMarkerElement);
			
			segmentMarkerElement.setAttribute("value", Integer.toString(i));
			segmentMarkerElement.setAttribute("label", Integer.toString(i));
		}
	}
		
	/**
	 * 	Save all stylesheet content in a css file, the css file name is the same as
	 *  the html name, e.g. if the html file name is test1.html, the css name is 
	 *  test1.css
	 * 
	 * @param	file	a html file 
	 * @return a css file name
	 */
	protected String createStyleSheet(IFile file) throws CoreException
	{
		// Generate a string buffer in which all stylesheet content resides
		StringBuffer sb = new StringBuffer();

		sb.append("rect.outline{stroke-width:0.75pt; stroke:#666666; fill:#ffffff; }\n");
		sb.append("polyline.keyline{stroke-width:0.75pt; stroke:#666666;}\n");
		sb.append("text.title{font-family:Arial; font-style:bold; font-size:11; text-anchor:left;}\n");
		sb.append("text.xtitle{font-family:Arial; font-style:regular; font-size:10; text-anchor:middle;}\n");
		sb.append("text.ytitle{font-family:Arial; font-style:regular; font-size:10; text-anchor:start; writing-mode:tb;}\n");
		sb.append("text.axislabel{font-family:Arial; font-style:regular; font-size:8; text-anchor:middle;}\n");
		sb.append("text.ydata{font-family:Arial; font-style:regular; font-size:8; text-anchor:middle;}\n");
		sb.append("text.legendtitle{font-family:Arial; font-style:regular; font-size:9; text-anchor:start;}\n");
		sb.append("text.legendlabel{font-family:Arial; font-style:regular; font-size:8; text-anchor:start;}\n");
		sb.append("polyline.grid{stroke-width:0.75pt; stroke:#000000;}\n");
		sb.append("polyline.gridline{stroke-width:0.75pt; stroke:#cccccc;}\n");
		sb.append("polyline.griddashline{stroke-width:0.75pt; stroke:#cccccc; stroke-dasharray:2; fill:none;}\n");
		sb.append("polyline.dataset0{stroke-width:0.75pt; stroke:#003399;}\n");
		sb.append("polyline.dataset1{stroke-width:0.75pt; stroke:#0066CC;}\n");
		sb.append("polyline.dataset2{stroke-width:0.75pt; stroke:#3399FF;}\n");
		sb.append("polyline.dataset3{stroke-width:0.75pt; stroke:#66CCFF;}\n");
		sb.append(".shape0{stroke:black; fill:#003399;}\n");
		sb.append(".shape1{stroke:black; fill:#0066CC;}\n");
		sb.append(".shape2{stroke:black; fill:#3399FF;}\n");
		sb.append(".shape3{stroke:black; fill:#66CCFF;}\n");

		return createFile(file, "css", sb.toString()).getName();
	}

	/**
	 * Returns the directory where the preference page for the SVG graph is
	 * stored.
	 * 
	 * @return String The absolute path of the preference page directory.
	 */
	protected String getSVGPreferencePageDirectory()
	{
		File directory = getPluginDirectory();
		if(directory != null)
		{
			directory = new File(directory, "preferences");
		}
		else
		{	
			if(usedClasspath != null)
			{
				String classpath = usedClasspath;
				int index = classpath.indexOf(File.pathSeparator);
				if(index >= 0)
					classpath = classpath.substring(0, index);
					
				index = classpath.lastIndexOf(File.separator);
				if(index >= 0)
					classpath = classpath.substring(0, index);

				directory = new File(classpath + "/preferences");
			}
		}
		
		if((directory != null) && directory.exists() && directory.isDirectory())
			return directory.getAbsolutePath();
		return null;		
	}
	
	/**
	 * Returns a {@link File} handle to the directory of a plugin or 
	 * <code>null</code> if the directory was not found.
	 * @param pluginId
	 * @return File
	 */
	protected File getPluginDirectory()
	{
		File pluginDir = null;
		
		try
		{
			URL pluginURL = new URL(HyadesSamplePlugin.getInstance().getBundle().getEntry("/"), "plugin.xml");			
			File manifestFile = new File(Platform.asLocalURL(pluginURL).getFile()).getAbsoluteFile();
			if(manifestFile.exists())
				pluginDir = manifestFile.getParentFile();
		}
		catch(Throwable e)
		{
		}
		
		return pluginDir;
	}	

	/**
	 * Creates a file in the same location and name of the <i>originalFile</i> but with a different
	 * extension.
	 * 
	 * @param IFile originalFile The file that will be used to define the location and name of the new file
	 * @param String extension The extension of the new file
	 * @param String content The content of the new file.
	 * @return IFile The created file;
	 */
	protected IFile createFile(IFile originalFile, String extension, String content) throws CoreException
	{
		// Generate the file handle where the svg file will be saved
		IPath newPath = originalFile.getProjectRelativePath().removeFileExtension().addFileExtension(extension);
		IFile newFile = originalFile.getProject().getFile(newPath);

		// Deletes any existing file with the same path
		if (newFile.exists())
			newFile.delete(true, true, null);

		// Creates the svg file in the path
		newFile.create(new ByteArrayInputStream(content.getBytes()), true, null);
		return newFile;
	}
	
	/**
	 * Returns the values that will be presented for the user.
	 * 
	 * @param String key The key for the value
	 * @return String the value
	 */
	protected String getUserText(String key)
	{
		if("chart.Title".equals(key))
			return "Test progress over time in days";
		
		if("chart.Legend".equals(key))
			return "Status";
			
		if("chart.timestampPrefix".equals(key))
			return "Last update:";
		
		if("chart.Xtitle".equals(key))	
			return "Date starting at:";
			
		if("chart.Ytitle".equals(key))
			return "Tests";


		if("chart.attempt".equals(key))
			return "Attempt";
			
		if("chart.error".equals(key))
			return "Error";
		
		if("chart.pass".equals(key))
			return "Pass";
		
		if("chart.fail".equals(key))	
			return "Fail";
			
		if("chart.inconclusive".equals(key))
			return "Inconclusive";


		if("chart.expectedAttempt".equals(key))
			return "Expected Attempt";
			
		if("chart.expectedError".equals(key))
			return "Expected Error";
		
		if("chart.expectedPass".equals(key))
			return "Expected Pass";
		
		if("chart.expectedFail".equals(key))	
			return "Expected Fail";
			
		if("chart.expectedInconclusive".equals(key))
			return "Expected Inconclusive";
		
		return key;
	}
}