package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GraphicAttributes.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Locale;

import org.w3c.dom.Document;
/**
 * Convenience class for maintaining the chart attributes.
 * 
 * @version 1.22
 */
public class GraphicAttributes implements IGraphicDocumentStyle, IUserPreferencesConstants, IGraphicDocumentDefaults, Serializable {
	
	/**
	 * Browser cookie name
	 */
	private String browserCookieName;
	
	/**
	 * Configuration document
	 */
	private Document configDocument;
	
	/**
	 * Data document
	 */	
	private Document dataDocument;
		
	/**
	 * Graphic type
	 */
	private String graphicType;	
		
	/**
	 * Resource folder location
	 */
	private String resourcesFolder;
	
	/**
	 * Graphic title
	 */
	private String graphicTitle;
		
	/**
	 * Timestamp prefix
	 */
	private String graphicTimestampPrefix;
		
	/**
	 * Legend display indicator
	 */
	private boolean legendSuppressed = false;
	
	/**
	 * Legend title
	 */
	private String graphicLegendTitle;
		
	/**
	 * Graphic font
	 */
	private String graphicFont = DEFAULT_FONT;
		
	/**
	 * Default colour palette
	 */
	private String graphicColorPaletteName = DEFAULT_PALETTE;
		
	/**
	 * Set of available colour palettes
	 */
	private HashMap graphicColorPalette;
		
	/**
	 * Graphic width
	 */
	private short graphicWidth = 0;
	
	/**
	 * Graphic height
	 */
	private short graphicHeight = 0;
	
	/**
	 * Locale for formatting. Default is the default on machine
	 */
	Locale locale = Locale.getDefault();
	
	/**
	 * Graphic outline suppression indicator
	 */
	private boolean outlineSuppressed = false;
	
	/**
	 * User preference page
	 */
	private String preferencesPage;
	
	/**
	 * User preference page pixel width, resolvable to <code>int</code>.
	 * Specific to the preference page.
	 */
	private String preferencesPageWidth;
	
	/**
	 * User preference page pixel height, resolvable to <code>int</code>.
	 * Specific to the preference page.
	 */
	private String preferencesPageHeight;
	
	/**
	 * User preference page icon hover text.
	 */
	private String preferencesIconText;
	
	/**
	 * Graphic will be SVG format only
	 */
	private boolean svgFormatOnly = true;
	
	/**
	 * Timestamp suppression indicator
	 */
	private boolean timestampSuppressed = false;
	
	/**
	 * Title bar suppression indicator
	 */
	private boolean titlebarSuppressed = false;
	
	/**
	 * User preferences 
	 */
	private String userPreferences;
	
	/**
	 * User preference suppression indicator
	 */
	private boolean userPreferencesSuppressed = false;
	
	/**
	 * User preference update script variable name
	 */
	private String userPreferencesUpdateVariableName;
	
	
	/**
	 * Number formatter for small numbers
	 */
	private DecimalFormat numberFormatter = null;
	
	/**
	 * Number formatter for big numbers
	 */
	private DecimalFormat bigNumberFormatter = null;
	
	/**
	 * Percent formatter
	 */
	private NumberFormat percentFormatter = null;
	
	/**
	 * Date formatter
	 */
	private DateFormat dateFormatter = null;
	
	/**
	 * Sole constructor.
	 */
	public GraphicAttributes() {		
		setGraphicWidth(DEFAULT_WIDTH);
		setGraphicHeight(DEFAULT_HEIGHT);				
		setFormatter();
	}	
	
    /**
     * Initializes this object using values from the input bean.
     * 
     * @param input the input bean
     */
    public void setAttributesFrom(GraphicDocumentProperties input) {
	    configDocument = input.getConfigDocument();	
		dataDocument = input.getDataDocument();	
		
		setBrowserCookieName(input.getBrowserCookieName());
		setResourcesFolder(input.getResourcesFolder());
		setGraphicTitle(input.getGraphicTitle());		
		setGraphicTimestampPrefix(input.getGraphicTimestampPrefix());		
		setLegendSuppressed(input.getLegendSuppressed());	
		setGraphicLegendTitle(input.getGraphicLegendTitle());		
		setGraphicFont(input.getGraphicFont());						
		setPreferencesPage(input.getPreferencesPage());
		setPreferencesPageWidth(input.getPreferencesPageWidth());	
		setPreferencesPageHeight(input.getPreferencesPageHeight());	
		setPreferencesIconText(input.getPreferencesIconText());
		setSvgFormatOnly(input.getSvgFormatOnly());
		setUserPreferences(input.getUserPreferences());  
		setUserPreferencesSuppressed(input.getUserPreferencesSuppressed());
		setUserPreferencesUpdateVariableName(input.getUserPreferencesUpdateVariableName());
		
		setTitlebarSuppressed(input.getTitlebarSuppressed());
		setTimestampSuppressed(input.getTimestampSuppressed());
		setOutlineSuppressed(input.getOutlineSuppressed());
		setGraphicHeight(input.getGraphicHeight()); 	
		setGraphicWidth(input.getGraphicWidth());
		setLocale(input.getLocale());
			
		createGraphicColorPalettes(input.getGraphicColorPalettes());
		setGraphicColorPaletteName(input.getGraphicColorPaletteName());

		/* Test 
		System.out.println("BrowserCookieName: " + input.getBrowserCookieName()); 
		System.out.println("ResourceFolder: " + input.getResourcesFolder());
   		System.out.println("Graphic title: " + input.getGraphicTitle());		
   		System.out.println("Timestamp prefix: " + input.getGraphicTimestampPrefix());		
   		System.out.println("Legend title:  " + input.getGraphicLegendTitle());		
   		System.out.println("Font: " + input.getGraphicFont());						
		System.out.println("Preferences Page: " + input.getPreferencesPage());
   		System.out.println("Preference page width: " + input.getPreferencesPageWidth());	
   		System.out.println("Preference page height: " + input.getPreferencesPageHeight());	
		System.out.println("Preferences Icon Text: " + input.getPreferencesIconText());
   		System.out.println("Graphic Height: " + input.getGraphicHeight()); 	
   		System.out.println("Graphic Width: " + input.getGraphicWidth()); 	
   		System.out.println("Legend Suppressed: " + input.getLegendSuppressed());			
		System.out.println("User Preferences: " + input.getUserPreferences());  
		System.out.println("User Preferences Suppressed: " + input.getUserPreferencesSuppressed());  
   		System.out.println("Outline Suppressed: " + input.getOutlineSuppressed());
   		System.out.println("Title Bar Suppressed: " + input.getTitlebarSuppressed());   		
   		System.out.println("Timestamp Suppressed: " + input.getTimestampSuppressed()); 
 	   		
 		try {
			HashMap colourMap = input.getGraphicColorPalettes();
			if (!colourMap.isEmpty())
			{
				Set colourSet = colourMap.keySet();
				for (Iterator i = colourSet.iterator(); i.hasNext();)
				{
					String key = (String)i.next();
					System.out.println("Graphic Colour Palette Name " + key);
					String[] colourArr = (String[])colourMap.get(key);
					System.out.print("values: ");
					for (int j = 0; j < colourArr.length; j++) {
						System.out.print(colourArr[j] + ", ");
					}
					System.out.println("");
				} 
			}
		} catch (Exception e) {}		
		System.out.println("Graphic Colour Palette Name chosen: " + input.getGraphicColorPaletteName());
		/* Test */

    }	
    
    /**
	 * Returns the browserCookieName.
	 * @return String
	 */
	public String getBrowserCookieName() {
		return browserCookieName;
	}
		

	/**
	 * Returns the configDocument.
	 * @return Document
	 */
	public Document getConfigDocument() {
		return configDocument;
	}

	/**
	 * Returns the dataDocument.
	 * @return Document
	 */
	public Document getDataDocument() {
		return dataDocument;
	}
		    
	/**
	 * Returns the graphicType.
	 * @return String
	 */
	public String getGraphicType() {
		return graphicType;
	}
		
	/**
	 * Returns the graphicHeight.
	 * @return short
	 */
	public short getGraphicHeight() {		
		return graphicHeight;
	}

	/**
	 * Returns the graphicWidth.
	 * @return short
	 */
	public short getGraphicWidth() {		
		return graphicWidth;
	}


	/**
	 * Returns the graphicFont.
	 * @return String
	 */
	public String getGraphicFont() {
		return graphicFont;
	}
	
	/**
	 * Returns the graphicColorPaletteName.
	 * @return String
	 */
	public String getGraphicColorPaletteName() {
		return graphicColorPaletteName;
	}
	
	/**
	  * Returns the graphicColorPalette.
	 * @return HashMap
	 */
	public HashMap getGraphicColorPalette() {
		return graphicColorPalette;
	}
	
	/**
	 * Returns the graphicTitle.
	 * @return String
	 */
	public String getGraphicTitle() {
		return graphicTitle;
	}
			
	/**
	 * Returns the legendSuppressed.
	 * @return boolean
	 */
	public boolean isLegendSuppressed() {
		return legendSuppressed;
	}	
	
	/**
	 * Returns the graphicTimestampPrefix.
	 * @return String
	 */
	public String getGraphicTimestampPrefix() {
		return graphicTimestampPrefix;
	}		
	
	/**
	 * Returns the graphicLegendTitle.
	 * @return String
	 */
	public String getGraphicLegendTitle() {
		return graphicLegendTitle;
	}
	
	/**
	 * Returns the outlineSuppressed.
	 * @return boolean
	 */
	public boolean isOutlineSuppressed() {
		return outlineSuppressed;
	}

	/**
	 * Returns the locale.
	 * @return Locale
	 */
	public Locale getLocale() {		
		return locale;
	}
		
	/**
	 * Returns the preferencesPage.
	 * @return String
	 */
	public String getPreferencesPage() {
		return preferencesPage;
	}
	
	/**
	 * Returns the preferencesPageWidth.
	 * @return String
	 */
	public String getPreferencesPageWidth() {
		return preferencesPageWidth;
	}

	/**
	 * Returns the preferencesPageHeight.
	 * @return String
	 */
	public String getPreferencesPageHeight() {
		return preferencesPageHeight;
	}
	
	/**
	 * Returns the preferencesIconText.
	 * @return String
	 */
	public String getPreferencesIconText() {
		return preferencesIconText;
	}

	/**
	 * Returns the svgFormatOnly.
	 * @return boolean
	 */
	public boolean getSvgFormatOnly() {
		return svgFormatOnly;
	}
	
	/**
	 * Returns the timestampSuppressed.
	 * @return boolean
	 */
	public boolean isTimestampSuppressed() {
		return timestampSuppressed;
	}
	
	/**
	 * Returns the titlebarSuppressed.
	 * @return boolean
	 */
	public boolean isTitlebarSuppressed() {
		return titlebarSuppressed;
	}	
		
	/**
	 * Returns the userPreferencesSuppressed.
	 * @return boolean
	 */
	public boolean isUserPreferencesSuppressed() {
		return userPreferencesSuppressed;
	}	
		
	/**
	 * Returns the userPreferencesUpdateVariableName.
	 * @return boolean
	 */
	public String getUserPreferencesUpdateVariableName() {
		return userPreferencesUpdateVariableName;
	}	
				
	/**
	 * Returns the userPreferences.
	 * @return String
	 */
	public String getUserPreferences() {
		return userPreferences;
	}
	 
	 /**
	 * Returns the resourcesFolder.
	 * @return String
	 */
	public String getResourcesFolder() {
		return resourcesFolder;
	}
		
	/**
	 * Sets the browserCookieName.
	 * @param newBrowserCookieName the new browser cookie name
	 */
	public void setBrowserCookieName(String newBrowserCookieName) {
		if (newBrowserCookieName != null) {
			browserCookieName = newBrowserCookieName.trim();
		}
	}
		
	/**
	 * Sets the configuration Document.
	 * @param Document the document containing configuration settings
	 */
	public void setConfigDocument(Document newConfigDocument) throws DataInputProcessingException {
		configDocument = newConfigDocument;
	}
	
	/**
	 * Sets the data update Document.
	 * @param Document the document containing data updates
	 */
	public void setDataDocument(Document newDataDocument) throws DataInputProcessingException {
		dataDocument = newDataDocument;
	}

	/**
	 * Sets the graphicHeight.
	 * @param String the new graphic height
	 */
	public void setGraphicHeight(String newGraphicHeight) {
		try {
			graphicHeight = Short.parseShort(newGraphicHeight);			
		} catch (Exception e) {}	
	}

	/**
	 * Sets the graphicWidth.
	 * @param String the new graphic width
	 */
	public void setGraphicWidth(String newGraphicWidth) {		
		try {
			graphicWidth = Short.parseShort(newGraphicWidth);			
		} catch (Exception e) {}						
	}

	/**
	 * Sets the graphicTitle.
	 * @param String the new graphic title
	 */
	public void setGraphicTitle(String newGraphicTitle) {
		if (newGraphicTitle != null) {
			graphicTitle = newGraphicTitle.trim();
		}
	}
	
	/**
	 * Sets the legendSuppressed.
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setLegendSuppressed(String newLegendSuppressed) {
		if (newLegendSuppressed != null) {			
			legendSuppressed = Boolean.valueOf(newLegendSuppressed).booleanValue();			
		}	
	}	
	
	/**
	 * Sets the graphicTimestampPrefix.
	 * @param String the new graphic timestamp prefix
	 */
	public void setGraphicTimestampPrefix(String newGraphicTimestampPrefix) {
		if (newGraphicTimestampPrefix != null) {
			graphicTimestampPrefix = newGraphicTimestampPrefix.trim();;
		}
	}		
	
	/**
	 * Sets the graphicLegendTitle.
	 * @param String the new graphic legend title
	 */
	public void setGraphicLegendTitle(String newGraphicLegendTitle) {
		if (newGraphicLegendTitle != null) {
			graphicLegendTitle = newGraphicLegendTitle.trim();
		}
	}
	
	/**
	 * Sets the graphicFont.
	 * @param String the new graphic font
	 */
	public void setGraphicFont(String newGraphicFont) {		
		if (newGraphicFont != null) {
			graphicFont = newGraphicFont + ", " + DEFAULT_FONT;
		}		
	}
	
	/**
 	 * Sets the graphicColorPaletteName.
	 * @param String the new graphic color palette
	 */
	public boolean setGraphicColorPaletteName(String newGraphicColorPaletteName) {
		boolean validName = false;
		if (newGraphicColorPaletteName != null) {			
			String name = newGraphicColorPaletteName.trim();
			validName = graphicColorPalette.containsKey(name);
			if (validName) {
				graphicColorPaletteName = name;
			}
		}
		return validName;				
	}
	
	/**
	 * Sets the outlineSuppressed.
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setOutlineSuppressed(String newOutlineSuppressed) {
		if (newOutlineSuppressed != null) {			
			outlineSuppressed = Boolean.valueOf(newOutlineSuppressed).booleanValue();			
		}			

	}


    /**
     * Set the number and date formatters
     */
    private void setFormatter()
    {
        NumberFormat nf;
        nf = NumberFormat.getInstance(locale);
        if (nf instanceof DecimalFormat) {
        	numberFormatter = (DecimalFormat) nf;
        	numberFormatter.applyPattern("#,##0.##");
        }
        nf = NumberFormat.getInstance(locale);
        if (nf instanceof DecimalFormat) {
        	bigNumberFormatter = (DecimalFormat) nf;
        	bigNumberFormatter.applyPattern("0.##E0");
        }
        
        percentFormatter = NumberFormat.getPercentInstance(locale);
        dateFormatter = DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG, locale);
    }
    	
	/** 
	 * Sets the locale.
	 * @param Locale locale to be used for date, time, number formatting
	 */
	public void setLocale(Locale newLocale) {
        if (newLocale == null) {
		    locale = Locale.getDefault();
        }
        else {
		    locale = newLocale;
        }
        setFormatter();
	}
	
	/** 
	 * Sets the preferencesPage.
	 * @param String the new preferences page
	 */
	public void setPreferencesPage(String newPreferencesPage) {
		if (newPreferencesPage != null) {
			preferencesPage = newPreferencesPage.trim();
		}
	}
	
	/**
	 * Sets the preferencesPageWidth.
	 * @param String the new preferences page width
	 */
	public void setPreferencesPageWidth(String newPreferencesPageWidth) {
		if (newPreferencesPageWidth != null) {
			preferencesPageWidth = newPreferencesPageWidth;
		}
	}
	
	/**
	 * Sets the preferencesPageHeight.
	 * @param String the new preferences page height
	 */
	public void setPreferencesPageHeight(String newPreferencesPageHeight) {
		if (newPreferencesPageHeight != null) {
			preferencesPageHeight = newPreferencesPageHeight;
		}
	}
	
	/**
	 * Sets the preferencesIconText.
	 * @param String the new preferences icon hover text
	 */
	public void setPreferencesIconText(String newPreferencesIconText) {
		if (newPreferencesIconText != null) {
			preferencesIconText = newPreferencesIconText.trim();
		}
	}
	
	/**
	 * Sets the resourcesFolder.
	 * @param String the new resources folder
	 */
	public void setResourcesFolder(String newResourcesFolder) {
		if (newResourcesFolder != null) {
			resourcesFolder= newResourcesFolder.trim();
		}
	}
	
	/**
	 * Sets the svgFormatOnly to indicate that only the SVG format
	 * will be used. In this case, code is generated which works 
	 * properly in SVG but may produce unpredictable results in other
	 * formats, for example, JPG.
	 * 
	 * @param String The svgFormatOnly to set
	 */
	public void setSvgFormatOnly(String newSvgFormatOnly) {
		if (newSvgFormatOnly != null) {
			this.svgFormatOnly = Boolean.valueOf(newSvgFormatOnly).booleanValue();
		}
	}
	
	/**
	 * Sets the timestampSuppressed.	
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setTimestampSuppressed(String newTimestampSuppressed) {
		if (newTimestampSuppressed !=null){			
			timestampSuppressed = Boolean.valueOf(newTimestampSuppressed).booleanValue();			
		}
	}
	
	/**
	 * Sets the titlebarSuppressed.
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setTitlebarSuppressed(String newTitlebarSuppressed) {
		if (newTitlebarSuppressed != null) {			
			titlebarSuppressed = Boolean.valueOf(newTitlebarSuppressed).booleanValue();			
		}	
	}
	
	/**
	 * Sets the userPreferences.
	 * @param String The default user preferences
	 */
	public void setUserPreferences(String newUserPreferences) {
		if (newUserPreferences != null) {			
			userPreferences = newUserPreferences;			
		}	
	}
	
	/**
	 * Sets the userPreferencesSuppressed.
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setUserPreferencesSuppressed(String newUserPreferencesSuppressed) {
		if (newUserPreferencesSuppressed != null) {			
			userPreferencesSuppressed = Boolean.valueOf(newUserPreferencesSuppressed).booleanValue();			
		}	
	}
	
	/**
	 * Sets the userPreferencesUpdateVariableName.
	 * @param String For <code>true</code>, must be "true". Otherwise, set to <code>false</code>.
	 */
	public void setUserPreferencesUpdateVariableName(String newUserPreferencesUpdateVariableName) {
		if (newUserPreferencesUpdateVariableName != null) {			
			userPreferencesUpdateVariableName = newUserPreferencesUpdateVariableName;			
		}	
	}
			
	/**
	 * Creates the set of available color palettes. Adds both the
	 * user-specified palettes and the documented set of default 
	 * palettes. 
	 *
	 * @param HashMap the user-specified palettes 
	 * @see IGraphicDocumentStyle
	 */
	private void createGraphicColorPalettes(HashMap userPalettes) {	
		if (userPalettes != null) {
			graphicColorPalette = new HashMap(userPalettes);
		}
		else {
			graphicColorPalette = new HashMap();
		}
			
		graphicColorPalette.put(SEASIDE, SEASIDE_PALETTE);
		graphicColorPalette.put(SUMMER, SUMMER_PALETTE);
		graphicColorPalette.put(EARTH, EARTH_PALETTE);
		graphicColorPalette.put(SPRING, SPRING_PALETTE);
		graphicColorPalette.put(HARVEST, HARVEST_PALETTE);
		graphicColorPalette.put(PASTEL, PASTEL_PALETTE);				
	}
	
	/**
	 * Format a number in the format for the locale set in this GraphicAttributes object
	 */
	public String formatNumber(double d) {
		DecimalFormat df;
		df = ((d > 100000) || (d < -100000)) ? bigNumberFormatter : numberFormatter;
		return df.format(d);
	}
	
	/**
	 * Format a number in the percent format for the locale set in this GraphicAttributes object
	 */
	public String formatPercentage(double d) {
		return percentFormatter.format(d);
	}
	
	/**
	 * Format a date in the format for the locale set in this GraphicAttributes object
	 */
	public String formatDate(Date d) {
		return dateFormatter.format(d);
	}
	
	/**
	 * Parse a date in ISO8601 format CCYY-MM-DDThh:mm:ss
	 * @param s date in ISO8601 format
	 * @return null if the format is invalid
	 */
	static public Date parseISO8601Date(String s) {
		if (s == null) {
			return null;
		}
		
		int year = 0, month = 0, day = 0, hrs = 0, min = 0, sec = 0;
		try {
            int len = s.length();
            if (len >= 4) {
                year = Integer.parseInt(s.substring(0,4));
            }
            if (len >= 7) {
                month = Integer.parseInt(s.substring(5,7))-1;
            }
            if (len >= 10) {
                day = Integer.parseInt(s.substring(8,10));
            }
            if (len >= 13) {
                hrs = Integer.parseInt(s.substring(11,13));
            }
            if (len >= 16) {
                min = Integer.parseInt(s.substring(14,16));
            }
            if (len >= 19) {
                sec = Integer.parseInt(s.substring(17,19));
            }
		    return new GregorianCalendar(year, month, day, hrs, min, sec).getTime();
		}
		catch(Exception e) {
			return null;
		}
	}

}
