package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GraphicDocumentProperties.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.HashMap;
import org.w3c.dom.Document;
import java.util.Locale;

/**
 * Data bean for setting the configurable properties of a GraphicDocumentGenerator object.
 * 
 * @version 1.13
 */
public class GraphicDocumentProperties implements Serializable {
	/**
	 * Browser cookie name
	 */
	private String browserCookieName;
	
	/**
	 * Configuration document
	 */
	private Document configDocument;
	
	/**
	 * Data document
	 */	
	private Document dataDocument;		
		
	/**
	 * Resource folder location
	 */
	private String resourcesFolder;
		
	/**
	 * Graphic title
	 */
	private String graphicTitle;
		
	/**
	 * Timestamp prefix
	 */
	private String graphicTimestampPrefix;
		
	/**
	 * Legend suppression indicator
	 */
	private String legendSuppressed;
	
	/**
	 * Locale 
	 */
	private Locale locale;
	
	/**
	 * Legend title
	 */
	private String graphicLegendTitle;
		
	/**
	 * Graphic font
	 */
	private String graphicFont;
		
	/**
	 * Default color palette
	 */
	private String graphicColorPaletteName;
				
	/**
	 * Graphic width
	 */
	private String graphicWidth;
	
	/**
	 * Graphic height
	 */
	private String graphicHeight;
	
	/**
	 * Graphic outline suppression indicator
	 */
	private String outlineSuppressed;
	
	/**
	 * User preference page
	 */
	private String preferencesPage;
	
	/**
	 * User preference page pixel width
	 * Specific to the preference page.
	 */
	private String preferencesPageWidth;
	
	/**
	 * User preference page pixel height
	 * Specific to the preference page.
	 */
	private String preferencesPageHeight;
	
	/**
	 * User preference page icon hover text.
	 */
	private String preferencesIconText;
	
	/**
	 * Graphic will be SVG format only
	 */
	private String svgFormatOnly;
	
	/**
	 * Timestamp suppression indicator
	 */	
	private String timestampSuppressed;
	
	/**
	 * Title bar suppression indicator
	 */
	private String titlebarSuppressed;
	
	/**
	 * User preferences
	 */
	private String userPreferences;
		
	/**
	 * User preferences suppression indicator.
	 */
	private String userPreferencesSuppressed;

	/**
	 * User preference retrieval JavaScript variable.
	 */
	private String userPreferencesUpdateVariableName;

	/**
	 * Set of color palettes to add to the user preferences
	 */	
	private HashMap graphicColorPalettes;
	
	/**
	 * Returns the browserCookieName.
	 * 
	 * @return String
	 */
	public String getBrowserCookieName() {
		return browserCookieName;
	}

	/**
	 * Returns the configDocument.
	 * 
	 * @return Document
	 */
	public Document getConfigDocument() {
		return configDocument;
	}
	
	/**
	 * Returns the dataDocument.
	 * 
	 * @return Document
	 */
	public Document getDataDocument() {
		return dataDocument;
	}

	/**
	 * Returns the graphicColorPaletteName.
	 * 
	 * @return String
	 */
	public String getGraphicColorPaletteName() {
		return graphicColorPaletteName;
	}

	/**
	 * Returns the set of color palettes.
	 * 
	 * @return HashMap
	 */
	public HashMap getGraphicColorPalettes() {
		return graphicColorPalettes;
	}
	
	/**
	 * Returns the graphicFont.
	 * 
	 * @return String
	 */
	public String getGraphicFont() {
		return graphicFont;
	}

	/**
	 * Returns the graphicHeight.
	 * 
	 * @return String
	 */
	public String getGraphicHeight() {
		return graphicHeight;
	}

	/**
	 * Returns the graphicLegendTitle.
	 * 
	 * @return String
	 */
	public String getGraphicLegendTitle() {
		return graphicLegendTitle;
	}

	/**
	 * Returns the graphicTimestampPrefix.
	 * 
	 * @return String
	 */
	public String getGraphicTimestampPrefix() {
		return graphicTimestampPrefix;
	}

	/**
	 * Returns the graphicTitle.
	 * 
	 * @return String
	 */
	public String getGraphicTitle() {
		return graphicTitle;
	}

	/**
	 * Returns the graphicWidth.
	 * 
	 * @return String
	 */
	public String getGraphicWidth() {
		return graphicWidth;
	}
	
	/**
	 * Returns the legendSuppressed.
	 * 
	 * @return String
	 */
	public String getLegendSuppressed() {
		return legendSuppressed;
	}

	/**
	 * Returns the locale.
	 * 
	 * @return Locale
	 */
	public Locale getLocale() {
		return locale;
	}
	
	/**
	 * Returns the outlineSuppressed.
	 * @return String
	 */
	public String getOutlineSuppressed() {
		return outlineSuppressed;
	}

	/**
	 * Returns the preferencesIconText.
	 * 
	 * @return String
	 */
	public String getPreferencesIconText() {
		return preferencesIconText;
	}

	/**
	 * Returns the preferencesPage.
	 * 
	 * @return String
	 */
	public String getPreferencesPage() {
		return preferencesPage;
	}

	/**
	 * Returns the preferencesPageHeight.
	 * 
	 * @return String
	 */
	public String getPreferencesPageHeight() {
		return preferencesPageHeight;
	}

	/**
	 * Returns the preferencesPageWidth.
	 * 
	 * @return String
	 */
	public String getPreferencesPageWidth() {
		return preferencesPageWidth;
	}

	/**
	 * Returns the resourcesFolder.
	 * 
	 * @return String
	 */
	public String getResourcesFolder() {
		return resourcesFolder;
	}
	
	
	/**
	 * Returns the svgFormatOnly.
	 * @return String
	 */
	public String getSvgFormatOnly() {
		return svgFormatOnly;
	}
	
	/**
	 * Returns the timestampSuppressed.
	 * @return String
	 */
	public String getTimestampSuppressed() {
		return timestampSuppressed;
	}
	
	/**
	 * Returns the titlebarSuppressed.
	 * 
	 * @return String
	 */
	public String getTitlebarSuppressed() {
		return titlebarSuppressed;
	}

	/**
	 * Returns the userPreferences passed to the <code>setUserPreferences</code> method.
	 * 
	 * @return String Format is a comma-separated list of 6 preference values:
	 *  1. palette identifer, e.g., "ZZ1"
	 *	2. font size, e.g., "8"
	 *	3. font, e.g., "font-weight:normal; font-style:normal; font-family:arial; fill:#000000;"
	 *	4. use shapes, "1" OR "0"
	 *	5. show values, "1" OR "0" 
	 *	6. show percents, "1" OR "0" 	 
	 */	
	public String getUserPreferences() {
		return userPreferences;
	}
	
	/**
	 * Returns the userPreferencesSuppressed.
	 * 
	 * @return String 
	 */
	public String getUserPreferencesSuppressed() {
		return userPreferencesSuppressed;
	}
	
	/**
	 * Returns the userPreferencesUpdateVariableName.
	 * 
	 * @return String 
	 */
	public String getUserPreferencesUpdateVariableName() {
		return userPreferencesUpdateVariableName;
	}
	
	
	/**
	 * Sets the browser cookie name. This method should be used only
	 * if user preferences are to be saved to a browser cookie on the
	 * client machine.
	 * 
	 * @param browserCookieName 
	 */
	public void setBrowserCookieName(String browserCookieName) {
		this.browserCookieName = browserCookieName;
	}
	
	
	/**
	 * Sets the configDocument.
	 * 
	 * @param configDocument The configuration which conforms to
	 * the documented configuration schema. 
	 */
	public void setConfigDocument(Document configDocument) {
		this.configDocument = configDocument;
	}

	/**
	 * Sets the dataDocument.
	 * 
	 * @param dataDocument The data update which conforms to the 
	 * documented data update schema.
	 */
	public void setDataDocument(Document dataDocument) {
		this.dataDocument = dataDocument;
	}

	/**
	 * Sets the graphicColorPaletteName. Use to make one of added the custom palettes
	 * the default. 
	 * 
	 * @param graphicColorPaletteName The default color palette to use
	 */
	public void setGraphicColorPaletteName(String graphicColorPaletteName) {
		this.graphicColorPaletteName = graphicColorPaletteName;
	}

	/**
	 * Sets the graphicFont. Caution that it is set to a font which is available
	 * to the target application (e..g, a browser).
	 * 
	 * @param graphicFont The default text font to use.
	 */
	public void setGraphicFont(String graphicFont) {
		this.graphicFont = graphicFont;
	}

	/**
	 * Sets the graphicHeight.
	 * 
	 * @param graphicHeight The height which is assumed in pixel units unless
	 * otherwise specified.
	 */
	public void setGraphicHeight(String graphicHeight) {
		this.graphicHeight = graphicHeight;
	}

	/**
	 * Sets the graphicLegendTitle.
	 * 
	 * @param graphicLegendTitle The title which displays over the legend.
	 */
	public void setGraphicLegendTitle(String graphicLegendTitle) {
		this.graphicLegendTitle = graphicLegendTitle;
	}

	/**
	 * Sets the graphicTimestampPrefix.
	 * 
	 * @param graphicTimestampPrefix The text which is prepended to the timestamp.
	 */
	public void setGraphicTimestampPrefix(String graphicTimestampPrefix) {
		this.graphicTimestampPrefix = graphicTimestampPrefix;
	}

	/**
	 * Sets the graphicTitle.
	 * 
	 * @param graphicTitle The title which displays at the top of the graph.
	 */
	public void setGraphicTitle(String graphicTitle) {
		this.graphicTitle = graphicTitle;
	}

	/**
	 * Sets the graphicWidth.
	 * 
	 * @param graphicWidth The width which is assumed in pixel units unless
	 * otherwise specified.
	 */
	public void setGraphicWidth(String graphicWidth) {
		this.graphicWidth = graphicWidth;
	}
	
	/**
	 * Sets the legendSuppressed indicator. By default, legends are not
	 * suppressed.
	 * 
	 * @param legendSuppressed One of {"true", "false"} 
	 */
	public void setLegendSuppressed(String legendSuppressed) {
		this.legendSuppressed = legendSuppressed;
	}

	/**
	 * Sets the locale. The default locale is the default locale of the machine
	 * on which the generator is running.
	 * 
	 * @param locale The locale which is to be used to generate date, time, number formats
	 */
	public void setLocale(Locale locale) {
		this.locale = locale;
	}
	
	/**
	 * Sets the outlineSuppressed. By default, the outline is not
	 * suppressed.
	 * 
	 * @param outlineSuppressed One of {"true", "false"} 
	 */
	public void setOutlineSuppressed(String outlineSuppressed) {
		this.outlineSuppressed = outlineSuppressed;
	}

	/**
	 * Sets the preferencesIconText.
	 * 
	 * @param preferencesIconText The flyover text appearing over the
	 * preferences icon. 
	 */
	public void setPreferencesIconText(String preferencesIconText) {
		this.preferencesIconText = preferencesIconText;
	}

	/**
	 * Sets the preferencesPage.
	 * 
	 * @param preferencesPage A path- or file-name which when appended to
	 * the value set for the resources folder locates the preference page.	 
	 */
	public void setPreferencesPage(String preferencesPage) {
		this.preferencesPage = preferencesPage;
	}

	/**
	 * Sets the preferencesPageHeight.
	 * @param preferencesPageHeight The pixel height to set
	 */
	public void setPreferencesPageHeight(String preferencesPageHeight) {
		this.preferencesPageHeight = preferencesPageHeight;
	}

	/**
	 * Sets the preferencesPageWidth.
	 * 
	 * @param preferencesPageWidth The pixel width to set
	 */
	public void setPreferencesPageWidth(String preferencesPageWidth) {
		this.preferencesPageWidth = preferencesPageWidth;
	}

	/**
	 * Sets the resourcesFolder. Warning: this value must be set (and to the correct location)
	 * to prevent the generator from throwing an exception during generation.
	 * 
	 * @param resourcesFolder A URI which points to where required resources are located.
	 */
	public void setResourcesFolder(String resourcesFolder) {
		this.resourcesFolder = resourcesFolder;
	}
	
	/**
	 * Sets the svgFormatOnly to indicate whether the generated SVG will
	 * be transcoded to other formats, for example, JPG. When this is
	 * set to <code>false</code>, some script code and the preferences icon
	 * are suppressed during generation. The default value is <code>true</code>.
	 * 
	 * @param svgFormatOnly The svgFormatOnly to set
	 */
	public void setSvgFormatOnly(String svgFormatOnly) {
		this.svgFormatOnly = svgFormatOnly;
	}

	/**
	 * Sets the timestampSuppressed indicator. By default, timestamps are not
	 * suppressed.
	 * 
	 * @param timestampSuppressed One of {"true", "false"} 
	 */
	public void setTimestampSuppressed(String timestampSuppressed) {
		this.timestampSuppressed = timestampSuppressed;
	}
	
	/**
	 * Sets the titleBarSuppressed indicator. By default, title bars are not
	 * suppressed.
	 * 
	 * @param titlebarSuppressed One of {"true", "false"} 
	 */
	public void setTitlebarSuppressed(String titlebarSuppressed) {
		this.titlebarSuppressed = titlebarSuppressed;
	}

	/**
	 * Sets the userPreferences. Warning: this method should be used only in conjunction with
	 * the client-side ECMAScript function <code>getUserPreferences</code>. The parameter must
	 * be set to an unmolested return value of that function. Any other use of this
	 * method is very likely to cause unpredictable and undesirable results.
	 * 
	 * @param userPreferences A value previously returned by the ECMAScript function <code>getUsePreferences</code>. 
	 */
	public void setUserPreferences(String userPreferences) {
		this.userPreferences = userPreferences;
	}
	
	/**
	 * Sets the userPreferencesSuppressed indicator. By default, user preferences
	 * are not suppressed.
	 * 
	 * @param userPreferencesSuppressed One of {"true", "false"}
	 */
	public void setUserPreferencesSuppressed(String userPreferencesSuppressed) {
		this.userPreferencesSuppressed = userPreferencesSuppressed;
	}
	
	/**
	 * Sets the userPreferencesUpdateVariableName to be updated when the user
	 * preferences are updated. This name is generated into the output and called
	 * by the generated code. Note that an error will result if the variable does not
	 * exist in the embedding document.
	 * 
	 * @param userPreferencesUpdateVariableName JavaScript variable name to be used
	 */
	public void setUserPreferencesUpdateVariableName(String userPreferencesUpdateVariableName) {
		this.userPreferencesUpdateVariableName = userPreferencesUpdateVariableName;
	}
	
	/**
	 * Adds the specified color palette to the color palette set. If the palette 
	 * name already exists, the palette is not added. No checking is done to ensure 
	 * the palette is valid or that the specified colors are supported in the target 
	 * environment. Note that adding a palette does not select that palette as the 
	 * default palette.
	 * 
	 * @param paletteName the name of the palette
	 * @param palette the colors (hexadecimal codes)
	 * @return boolean <code>true</code>, if the operation is successful	
	 */
	public boolean addGraphicColorPalette(String paletteName, String [] palette) throws NullPointerException {		
		boolean nameAvailable = false;
		if (paletteName != null && palette != null) {
			if (graphicColorPalettes == null) {
				graphicColorPalettes = new HashMap();
			}	
			String name = paletteName.trim();
			nameAvailable = !graphicColorPalettes.containsKey(name);
			if (nameAvailable) {				
				graphicColorPalettes.put(name, palette);				
			}		
		}
		return nameAvailable;	
	}	
	
	
}
