package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGAreaChart.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Date;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <code>SVGAreaChart</code> generates a Area chart graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.17
 */ 
public class SVGAreaChart extends SVGLineChart implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, Serializable {
	/**
	 * Sole constructor
	 */ 
	public SVGAreaChart() {
		super();
		graphicType = IGraphicTypeConstants.AREA_CHART;
	}
	
		/**
     * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
     */       
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {	
		// get the input
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);	
		Document newDataDocument = graphicAttrs.getDataDocument();		
		int numberOfDatasets = 0;
		String [] legendFlyOverLabels = null;		
		// make sure there is some data, or exit
		try {
			numberOfDatasets = dataRetriever.getNumberOfDatasets(newDataDocument);
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;		
		}		
		
		String rangeMaxValue = null;
		int gridXOffset = 60;
		int halfGridXOffset = gridXOffset/2;
		int gridYOffset = 50;
		int xAxisLength = 0;
		// grid     
		try {
			xAxis = dataRetriever.getDataRange(newDataDocument, "S");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		try {
			yAxis = dataRetriever.getDataRange(newDataDocument, "W");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		if ((xAxis == null) || (yAxis == null)) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}
		if ((xAxis.getType() != DataRange.CONTINUUM) || (yAxis.getType() != DataRange.CONTINUUM)) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		xMarkers = xAxis.getSegmentMarkers();
		yMarkers = yAxis.getSegmentMarkers();

		if (yMarkers != null && yMarkers.length > 0) { 
			rangeMaxValue = (String)yMarkers[yMarkers.length - 1].getValueString();
		}
		
		// The datasets
		dataSets = new DataSet[numberOfDatasets];
		for(int i=0; i < numberOfDatasets; i++) {
            dataSets[i] = dataRetriever.getDatasetWithPosition(newDataDocument, Integer.toString(i));
		}
		//get the Fly over text for the legend item.				
		legendFlyOverLabels = getLegendFlyOverLabels(dataSets);

		//get the max size of Ymarker Width.		
		double maxLengthOfYmarkerWidth = getMaxLengthOfYmarker(yMarkers, graphicAttrs);
	    double maxAllowableYMarkerWidth = graphicAttrs.getGraphicWidth() / 4;
	    if (maxLengthOfYmarkerWidth > maxAllowableYMarkerWidth) {
			maxLengthOfYmarkerWidth = maxAllowableYMarkerWidth;
		}
		
		//get the max size of Legent Lable Width.		
		double maxLengthOfLegendLableWidth = getMaxLengthOfLegendLabel(graphicAttrs, dataSets);
	    double maxAlloableLegendWidth = graphicAttrs.getGraphicWidth() / 3;
   		if (maxLengthOfLegendLableWidth > maxAlloableLegendWidth) {
			maxLengthOfLegendLableWidth = maxAlloableLegendWidth;
		}
		
		// Start calculations
        if (graphicAttrs.isLegendSuppressed()) {
			maxLengthOfLegendLableWidth = 0;
        }
		xAxisLength = (int)(graphicAttrs.getGraphicWidth() - (maxLengthOfLegendLableWidth + maxLengthOfYmarkerWidth + gridXOffset));
   		int xLabelRowCount = getNumberOfAxisLabels(graphicAttrs, xAxisLength, xAxis);									
		int yAxisLength = (int)(graphicAttrs.getGraphicHeight() - (gridYOffset * 2.5) - (xLabelRowCount * 10));			

		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs, numberOfDatasets);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, numberOfDatasets, gridXOffset, gridYOffset, xAxisLength, yAxisLength);
 		
 		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 		
		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}
			
		// title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		// need to adjust graphic vertical position
 		else {
 			gridYOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicHeight(), yAxisLength, 3.0);	
 		}	
 		
		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
		// the actual x offset
		gridXOffset = (halfGridXOffset) + (int)maxLengthOfYmarkerWidth;

		// border, tick marks etc.
        addGrid(generatedDocument, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);

		// x and y axis labels
        addAxisLabels(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
                
        // axis titles
        addTitles(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, xLabelRowCount, rangeMaxValue);

        // data lines and area
		Element gTextBoxElement = generatedDocument.createElement("g");        
        addDataSets(generatedDocument, graphicAttrs, svgRoot, gTextBoxElement, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, dataSets); 

       // legend
        if (!graphicAttrs.isLegendSuppressed()) {      		
	  		try {
	       		String [] legendLabels = dataRetriever.getLegendLabels(newDataDocument);		       
				addLegend(generatedDocument, graphicAttrs, gTextBoxElement, legendLabels, legendFlyOverLabels, (int)maxLengthOfYmarkerWidth + xAxisLength + (halfGridXOffset), gridYOffset); 								

	  		} catch (DataInputProcessingException e) {
	  			e.setWidgetType(graphicType);
	  			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());	  			
	  			throw e;
	  		} 		
    	}
		svgRoot.appendChild(gTextBoxElement);
    	return generatedDocument;
	}
	
	/**
     * @see SVGDocumentGenerator#addLegendItemGraphic()
     */ 
		protected Element addLegendItemGraphic(Document generatedDocument, Element parent, int index, int fontSize, int x, int y) {
		final String[] legendShapeType = {"#legendcircleshape", "#legendsquareshape", "#legendhorizontalrectshape", "#legendverticalrectshape", "#legendtriangleshape", "#legenddiamondshape"};
		Element  gColourElement, useElementShape;
		useElementShape = generatedDocument.createElement("use");
		useElementShape.setAttribute("xlink:href", "#legendsquareshapeWithoutLine");
		useElementShape.setAttribute("x", Integer.toString(x));
		// rectangles are placed using upper left corner
		useElementShape.setAttribute("y", Integer.toString(y + fontSize - 9));    
			
		// add the element to its color group for user preference changes
		gColourElement = generatedDocument.createElement("g");
		gColourElement.setAttribute("id", "legendsquareshapeWithoutLine" + index);
		gColourElement.setAttribute("class", "shape" + index);		
		if (graphicType.equals(IGraphicTypeConstants.AREA_CHART)) {
			gColourElement.setAttribute("onclick","toggleVisibility('dataColour" + index + "');");		
			gColourElement.setAttribute("onmouseover", "ChangeOpacity('poly" + index + "', '0.6');");
			gColourElement.setAttribute("onmouseout", "ChangeOpacity('poly" + index + "', '0.1');");			
		}
		gColourElement.appendChild(useElementShape);
		parent.appendChild(gColourElement);	
		
 		useElementShape = generatedDocument.createElement("use");
		useElementShape.setAttribute("xlink:href", legendShapeType[index % (legendShapeType.length)]);
		useElementShape.setAttribute("x", Integer.toString(x));
		// rectangles are placed using upper left corner
		useElementShape.setAttribute("y", Integer.toString(y + fontSize - 8));    
		
		// add the element to its color group for user preference changes
		gColourElement = generatedDocument.createElement("g");
		gColourElement.setAttribute("id", "shape" + index);		
		gColourElement.setAttribute("onclick","toggleVisibility('dataColour" + index + "');");
		gColourElement.setAttribute("onmouseover", "ChangeOpacity('poly" + index + "', '0.6');");
		gColourElement.setAttribute("onmouseout", "ChangeOpacity('poly" + index + "', '0.1');");			
		gColourElement.setAttribute("class", "shape" + index);	
		gColourElement.appendChild(useElementShape);
		return gColourElement;
	}

	// specific to area chart 
	private void addDataSets(Document generatedDocument, GraphicAttributes attrs, Element parent, Element gTextBoxElement, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, DataRange xAxis, DataRange yAxis, DataSet[] dataSets)
		throws DataInputProcessingException {

		// add lines and area
		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		double yMax = yAxis.getMaxValue();

		if ((xMin == xMax) || (yMin == yMax)) return;

		for (int i = 0; i < dataSets.length; i++) {
			DataSet dataset = dataSets[i];
			DataPoint[] dps = dataset.getDataPoints();
			double[] yPositions = new double[dps.length];
			double[] xPositions = new double[dps.length];
			
			for(int j=0; j < dps.length; j++) {
				xPositions[j] = xAxisLength * (dps[j].getValue1() - xMin) / (xMax - xMin);
				yPositions[j] = yAxisLength * (yMax - dps[j].getValue2()) / (yMax - yMin);
			}
			
		 	// add the line	and area	
	  	 	addArea(generatedDocument, parent, dps, i, xPositions, yPositions, xAxisLength, yAxisLength, gridXOffset, gridYOffset);

	  	 	// add the texbox
			String stylesheetClass = "dataValues anchorAtMiddle";		
			Element gToggleElement = generatedDocument.createElement("g");
			gToggleElement.setAttribute("id","textbox" + i);
			gToggleElement.setAttribute("class", stylesheetClass);
			gToggleElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
			//gToggleElement.setAttribute("font-family", font);
			gToggleElement.setAttribute("visibility", "hidden");
			gToggleElement.setAttribute("onclick", "toggleVisibility('textbox" + i + "');");
			gTextBoxElement.appendChild(gToggleElement);	
			addTextBoxes(generatedDocument, attrs, gToggleElement, i, xPositions, yPositions, dps, xAxisLength, yAxisLength);
		}			
	}


	// Add a set of polyline and polygon elements for each dataset
	private void addArea(Document generatedDocument, Element parent, DataPoint[] dps, int linesetNumber, double[] xPositions, double[] yPositions, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset) {

		final String[] shapeType = {"#circleshape", "#squareshape", "#horizontalrectshape", "#verticalrectshape", "#triangleshape", "#diamondshape"};
		final double[] shapeWidth = {5.6, 5.6, 5.6, 2.8, 5.6, 5.6};
		final double[] shapeHeight = {5.6, 5.6, 2.8, 5.6, 4.85, 5.6};
		final double BOX_HEIGHT = 18.0;

		int length = xPositions.length;
		double xPrevious = 0;
		double yPrevious = 0;
		double xNext;
		double yNext;
		
		// group for data colour changes and toggling exact values visibility
		Element gColourElement = generatedDocument.createElement("g");
		String pointAttribute = xPositions[0] + " " + yAxisLength;
		for (int i = 0; i < length; i++) 
		{
			xNext = xPositions[i];
			yNext = yPositions[i];
			pointAttribute += ", " + xNext + " " + yNext;
		}
		pointAttribute += ", " + xPositions[length-1] + " " + yAxisLength;
		// Draw area for the line chart
		Element polygonElement = generatedDocument.createElement("polygon");
		polygonElement.setAttribute("id", "poly" + linesetNumber);
		polygonElement.setAttribute("points", pointAttribute);
		polygonElement.setAttribute("style", "fill-opacity: 0.1; stroke: black; stroke-width: 0;");
		polygonElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gColourElement.appendChild(polygonElement);

		gColourElement.setAttribute("id", "dataColour" + linesetNumber);
		gColourElement.setAttribute("class", "dataset" + linesetNumber);
		gColourElement.setAttribute("visibility", "visible");
		gColourElement.setAttribute("onclick", "toggleVisibility('textbox" + linesetNumber + "');");
		
		parent.appendChild(gColourElement);
		
		// draw a polyline with data points
		int shapeNumber = linesetNumber % shapeType.length;
		for (int i = 0; i < length; i++) {
			xNext = xPositions[i];
			yNext = yPositions[i];
			
			if (i > 0) {
				Element newElement = generatedDocument.createElement("polyline");
				String points = xPrevious + " " + yPrevious + "," + xNext + " " + yNext;
				newElement.setAttribute("points", points);
				if (dps[i].getType() == DataPoint.HOLE || dps[i-1].getType() == DataPoint.HOLE)
				{
					newElement.setAttribute("style", "stroke-dasharray: 9, 5; stroke-width: 2;");
				}	
				newElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
				// add new polyline to group element	
				gColourElement.appendChild(newElement);
			}
			
			// Add shapes				
			double shapeX = xNext - shapeWidth[shapeNumber] / 2.0;
			if (shapeX <= 0) { // adjust shape x position if it crosses the left boundary
				shapeX = 0;	
			} else if (shapeX + shapeWidth[shapeNumber] > xAxisLength) { // adjust shape x position if it crosses the right boundary
				shapeX = xAxisLength - shapeWidth[shapeNumber];
			}
			
			double shapeY = yNext - shapeHeight[shapeNumber] / 2.0;
			if (shapeY + shapeHeight[shapeNumber] > yAxisLength) {  // adjust the shape y position if it crosses the x-axis
				shapeY = yAxisLength - shapeHeight[shapeNumber];	
			}
			
			Element shapeElement = generatedDocument.createElement("use");
			if (dps[i].getType() == DataPoint.HOLE)
			{
				shapeElement.setAttribute("xlink:href", "#holeshape");
			}else
			{
				shapeElement.setAttribute("xlink:href", shapeType[shapeNumber]);
			}			
			shapeElement.setAttribute("transform", "translate(" + (gridXOffset + shapeX) + "," + (gridYOffset + shapeY) + ")");
			//shapeElement.setAttribute("x", Double.toString(gridXOffset + shapeX));
			//shapeElement.setAttribute("y", Double.toString(gridYOffset + shapeY));
			shapeElement.setAttribute("visibility", "visible");
			gColourElement.appendChild(shapeElement);
						
			// add shapes on the data point
			xPrevious = xNext;
			yPrevious = yNext;	
		}		
	}	


	/**
	 * Initialize the Scatter chart attributes
	 */
	protected GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(AREA_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.AREA_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.AREA_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
	
}
