package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGBarChart.java,v 1.4 2005/02/16 22:24:05 qiyanli Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Date;

import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
/**
 * <code>SVGBarChart</code> generates a bar chart graphic DOM using Scalable Vector Graphics (SVG).
 * 
 * @version 1.56.1.23
 */
public class SVGBarChart extends SVGXYChart implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, Serializable {
	/**
	 * Sole constructor.
	 */
	public SVGBarChart() {
		super();
		graphicType = IGraphicTypeConstants.BAR_CHART;			
	}
	
	/**
	 * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
	 */	
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {		
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);
		Document newDataDocument = graphicAttrs.getDataDocument();	
        		
		// clean this up ???					
		String rangeMaxValue = null;
		int gridXOffset = 60;
		int gridYOffset = 50;
		DataRange xAxis = null;
		DataRange yAxis = null;	
		SegmentMarker[] xMarkers = null;
		SegmentMarker[] yMarkers = null;
		
		// The axes
		try {
			xAxis = dataRetriever.getDataRange(newDataDocument, "S");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		try {
			yAxis = dataRetriever.getDataRange(newDataDocument, "W");

		} catch (DataInputProcessingException e) {
			e.setWidgetType(getGraphicType());
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}

		if ((xAxis == null) || (yAxis == null)) {
			throw new DataInputProcessingException("", NO_RANGE_FOUND, DATARANGE, getGraphicType());
		}
		if ((xAxis.getType() != DataRange.CATEGORIZATION) || (yAxis.getType() != DataRange.CONTINUUM)) {
			throw new DataInputProcessingException("", UNSUPPORTED_DATA_RANGE_TYPE, DATARANGE, getGraphicType());
		}

		xMarkers = xAxis.getSegmentMarkers();
		yMarkers = yAxis.getSegmentMarkers();

		if (yMarkers != null && yMarkers.length > 0) { 
			rangeMaxValue = (String)yMarkers[yMarkers.length - 1].getValueString();
		}
		
		// The dataset
		DataSet dataSet = dataRetriever.getDatasetWithPosition(newDataDocument, "0");
		
		
		// Start the calculations
		int xAxisLength = (int)(graphicAttrs.getGraphicWidth() - (gridXOffset * 2.25));
   		int xLabelRowCount = getNumberOfAxisLabels(graphicAttrs, xAxisLength, xAxis);									
		int yAxisLength = (int)(graphicAttrs.getGraphicHeight() - (gridYOffset * 2.5) - (xLabelRowCount * 10));			
		
		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		addDefinitions(generatedDocument, graphicAttrs, dataSet.getDataPoints().length);
		registerEventHandler(generatedDocument.getDocumentElement(),"onload", "init(evt)");				
		addJavaScriptFunctions(generatedDocument, graphicAttrs, dataSet.getDataPoints().length, gridXOffset, gridYOffset, xAxisLength, yAxisLength); 		
	   
	    // graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		

		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {		
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}	
		 		 	
	   // title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}
 		// need to adjust graphic vertical position
 		else {
 			gridYOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicHeight(), yAxisLength, 3.0);	
 		}	

		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
				 		  
		gridXOffset = super.resetGridOffsetForCenter(graphicAttrs.getGraphicWidth(), xAxisLength, 2.0);		
		if ((rangeMaxValue != null) && (rangeMaxValue.length() > 2)) {
			gridXOffset += (rangeMaxValue.length() - 2) * 7.5;    // newly added
		}
		
		// work out where to put the data horizontally
		evenlyDistributeSegmentMarkers(xAxis);

		// border, tick marks etc.
        addGrid(generatedDocument, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);

		// x and y axis labels
        addAxisLabels(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis);
                
        // axis titles
        addTitles(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, xLabelRowCount, rangeMaxValue);

        // data lines
        addDataSet(generatedDocument, graphicAttrs, svgRoot, xAxisLength, yAxisLength, gridXOffset, gridYOffset, xAxis, yAxis, dataSet); 

    	return generatedDocument;
	}

	// add definitions
	private void addDefinitions(Document generatedDocument, GraphicAttributes attrs, int numberOfDatasets) {
		Element defsElement = super.addDefinitions(generatedDocument, attrs);        
		addDefaultInternalStylesheet(generatedDocument, attrs, defsElement, numberOfDatasets);			
	}	
			
	// add default style sheet
	private void addDefaultInternalStylesheet(Document generatedDocument, GraphicAttributes attrs, Element parent, int sets) {
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		styleRules.append(" ");
		styleRules.append(XYCHART_STYLE);		
				
		String [] palette = (String [])(attrs.getGraphicColorPalette()).get(attrs.getGraphicColorPaletteName());
		// palette has limited number of colours, so loop if more sets
		int paletteLength = palette.length;		
		for (int i = 0, j = 0; i < sets; i++) {
		  j = (i + 1) % paletteLength;
		  j = (j == 0) ? paletteLength - 1 : j-1;			
		  styleRules.append(" .dataset" + i + "{stroke-width:0.75pt; stroke:" + palette[j] + ";}"); 
		  styleRules.append(" .shape" + i + "{stroke:none; fill:" + palette[j] + ";}");
		}		
		CDATASection newCDATASection = generatedDocument.createCDATASection(styleRules.toString());
		
		Element newElement = generatedDocument.createElement("style");
		newElement.setAttribute("id", "defaultStyleRules");
		newElement.setAttribute("type", "text/css");
		newElement.appendChild(newCDATASection);
		parent.appendChild(newElement);
	}	
	
	// add the data value-related elements
	private void addDataSet(Document generatedDocument, GraphicAttributes attrs, Element parent, int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, DataRange xAxis, DataRange yAxis, DataSet dataSet) throws DataInputProcessingException {
		int numberOfSegments = xAxis.getSegmentMarkers().length;
		double barWidth = (numberOfSegments > 0) ? xAxisLength / numberOfSegments / 2 : 0;

		Element gToggleElement = generatedDocument.createElement("g");
		String stylesheetClass = "dataValues anchorAtMiddle";		
		gToggleElement.setAttribute("id","exactValues");
		gToggleElement.setAttribute("class", stylesheetClass);		
		gToggleElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gToggleElement.setAttribute("visibility", "hidden");
		gToggleElement.setAttribute("onclick","toggleVisibility(\"exactValues\")");
		parent.appendChild(gToggleElement);
		
		Element gDataSetElement = generatedDocument.createElement("g");
		gDataSetElement.setAttribute("id","dataset");
		gDataSetElement.setAttribute("onclick","toggleVisibility(\"exactValues\")");
		gDataSetElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		parent.appendChild(gDataSetElement);
		
		// need to compute y values for actual datapoints based on the number of pixels 
		// between each potential datapoint along y-axis
		// NumberFormatException not possible here
		//double yLabelsMaxValue = Double.parseDouble(yMax);			
		addBars(generatedDocument, attrs, gDataSetElement, gToggleElement, 
				xAxisLength, yAxisLength, xAxis, yAxis, barWidth, dataSet);
	}
	
	// add the data value bars	
	private void addBars(Document generatedDocument, GraphicAttributes attrs, Element parent, Element toggleParent, 
							int xAxisLength, int yAxisLength, DataRange xAxis, DataRange yAxis, 
							double barWidth, DataSet dataSet) throws DataInputProcessingException  {

		SegmentMarker[] xsms = xAxis.getSegmentMarkers();
		double yMax = yAxis.getMaxValue();
		double yMin = yAxis.getMinValue();
		
		for (int i = 0; i< xsms.length; i++) {
			SegmentMarker sm = xsms[i];
			DataPoint dp = dataSet.findDataPoint(sm.getValue());
			if (dp == null) continue;
			if (dp.getType() == DataPoint.HOLE) continue;
			
			//leftmost x
			double x = (sm.getPosition() * xAxisLength) - (barWidth / 2);
			//topmost y
			double yValue = dp.getValue2();
			if (yValue < yMin) yValue = yMin;
			if (yValue > yMax) yValue = yMax;
			double y = yAxisLength * (yMax - yValue)/(yMax - yMin);

			Element newElement = generatedDocument.createElement("rect");

			newElement.setAttribute("id", "dataColour" + i);  // for updating colour change
			newElement.setAttribute("class", "shape" + i);
			newElement.setAttribute("x", Double.toString(x));
			newElement.setAttribute("y", Double.toString(y));
			newElement.setAttribute("width", Double.toString(barWidth));
			newElement.setAttribute("height", Double.toString(yAxisLength - y - 0.9));
			newElement.setAttribute("stroke-width", "0.75pt");
			newElement.setAttribute("stroke", "#666666");			
			parent.appendChild(newElement);
			
			// Set up the position of the text over a bar			
			double textX = x + barWidth / 2.0;
			double textY = y - 5.0;   // text y position
			
			// Add values over the bar
			String label = dp.getLabel();
			if (label == null) {
				label = attrs.formatNumber(dp.getValue2());
			}
			addExactValue(generatedDocument, attrs, toggleParent, textX, textY, label);
		}			
	}
			
	// add category total values over the bar
	private void addExactValue(Document generatedDocument, GraphicAttributes attrs, Element parent, double x, double y, String value) {				
		Element gElement = generatedDocument.createElement("g");
		gElement.setAttribute("id","text1");		
		addLabel(generatedDocument, attrs, gElement, value, "dataValues anchorAtMiddle", x, y, 0);
		parent.appendChild(gElement);	
	}	
	
	/**
	 * Initialize the bar chart attributes
	 */
	private GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults		
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.BAR_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.BAR_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
}
