package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractAccessLogParser.java,v 1.36 2005/03/23 08:08:48 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.StringTokenizer;

import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;

/** 
 * <code>AbstractAccessLogParser</code> is the abstract super class for the <code>ApacheAccessLogParser</code>.
 * <p>
 * For each access log record, this class parses the client IP address, user ID, time 
 * stamp, method, file name, return code and file size.  This parsed information and 
 * information about the local machine (machine where the parse is performed, but not 
 * necessarily the machine that produced the access log) is used to produce a
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent</code> object for each record.
 * 
 * This parser is based on the Apache Common Log Format (CLF) or "%h %l %u %t \"%r\" %>s %b".  
 * For more information, see the 
 * <a href="http://httpd.apache.org/docs-2.0/logs.html#accesslog">Apache HTTP Server Version 2.0 specification (Access Log File)</a>.
 *  
 * 
 * @author  Paul Slauenwhite
 * @author  Gary Dudley
 * @version	November 15, 2004
 * @see		org.eclipse.hyades.logging.parsers.Parser
 * @see		org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 */
public abstract class AbstractAccessLogParser extends MonitoringParser {

    //Variables to hold values parsed from the access log record:  

    /**
     * Parsed value of the client's IP address field. 
     */
    protected String clientIP = null;

    /**
     * Parsed value of the method field. 
     */
    protected String method = null;

    /**
     * Parsed value of the file name field. 
     */
    protected String fileName = null;

    /**
     * Parsed value of the return code field. 
     */
    protected String returnCode = null;

    /**
     * Parsed value of the file size field. 
     */
    protected String fileSize = null;

    /**
     * Parsed value of the userid field. 
     */
    protected String userID = null;

    /**
     * Parsed value of the User-agent field. 
     */
    protected String userAgent = null;

    /**
     * Parsed value of the Referer field. 
     */
    protected String referer = null;

    /**
     * Parsed value of the message portion of the access log record.  The message includes
     * the method, file name, protocol, return code, and file size fields. 
     */
    protected String message = null;

    /**
     * Parsed value of the time stamp field converted to the XML dateTime format. 
     */
    protected StringBuffer currentTimeStamp = null;

    /**
     * CGI debugging output. 
     * NOTE:  Must be initialized to an empty string buffer.
     */
    protected StringBuffer cgiDebuggingOutput = new StringBuffer();

    //User-supplied variables:

    /**
     * User-supplied value of the version of the web server product.  Instantiated in 
     * the setUserInput method of the subclass.
     */
    protected String productVersion = null;

    /**
     * Web server product name and version supplied by the subclass.   
     */
    protected String sourceID = null;

    //Private variables:

    /**
     * Parsed value of the time stamp field converted to the XML dateTime format from previous access log record. 
     * NOTE:  Must be initialized to an empty string for comparison with the first log record.
     * ASSUMPTION:  All access log records appear in chronological order within the log file.
     */
    private String previousTimeStamp = "";

    /**
     * Count of records with identical time stamp values. 
     * NOTE:  Must be initialized to 0 since no log records have been parsed.
     * ASSUMPTION:  All access log records appear in chronological order within the log file.
     */
    private long duplicateTimeStampCounter = 0;

	/**
	 * Initialize this parser.
	 */
	public void preParse() throws LogParserException {
		super.preParse();
	}
	
	/**
	 * De-constructs this parser (e.g. closes the log file handle).
	 */
	public void postParse() throws LogParserException {
		super.postParse();
	}
	
	/**
	 * Parses each access log record and produces a Common Base Event (CBE) object that is 
	 * returned as a member of a CBE array.
	 *
	 * @return CommonBaseEvent[] array of CBE's representing parsed records.  
	 * @exception LogParserException thrown if the parser is unable to parse the access log
	 */
	public CommonBaseEvent[] parseNext() throws LogParserException {
		CommonBaseEvent[] temp = null;
		
		curLine = readALine();
		arrayIndex = 0;

		try {

			//Iterate all the lines (e.g. until EOF/null line) within the file:
			while (curLine != null) {

				//Trim leading/trailing white-space from the current line:
				curLine = curLine.trim();
				
				//Only parse non-empty (e.g. lines with one or more non-whitespace characters) lines:
				if (curLine.length() > 0) {

					//Parse the log record:
					//ASSUMPTION:  Access log records do NOT span multiple lines.  Each log record is contained on one line.
					if (parseLogRecord()) {

						if (cgiDebuggingOutput.length() > 0) {

                            createCGIDebuggingOutputCBE();

                            arrayIndex++;

                            if (arrayIndex == MessageArraySize) {
                                // Increase the message array size by one
                                increaseMsgArraySize();
                            }

                            recordCount++;
                        }

						//(Re)initialize the CBE:
						reinitializeCBE();

						//Increment the CBE's sequence number and running counter when duplicate time stamp values are detected. 
						//ASSUMPTION:  All access log records appear in chronological order within the log file.
						if (previousTimeStamp.equals(currentTimeStamp.toString()))
							messages[arrayIndex].setSequenceNumber(++duplicateTimeStampCounter);

						//Otherwise persist the current time stamp as the new previous time stamp and reset the running duplicate time stamp counter:
						else {
							previousTimeStamp = currentTimeStamp.toString();
							duplicateTimeStampCounter = 0;
						}

						//Add information from the access log record to the CBE:
						messages[arrayIndex].setCreationTime(currentTimeStamp.toString());
						if (message.length() > 1024) {
							messages[arrayIndex].setMsg(message.substring(0, 1024));
							messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.MESSAGE_EXTENDED_DATA_ELEMENT, message));
						} else {
							messages[arrayIndex].setMsg(message);
						}

						determineSeverity(messages[arrayIndex]);

						//Add remaining information from the access log record as extended data elements to the CBE:
						if (method != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("method", method));
						if (clientIP != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.CLIENT, clientIP));
						if (fileName != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.FILE, fileName));
						if (returnCode != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("return_code", returnCode));
						if (fileSize != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("file_size", fileSize));
						if (userID != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("userid", userID));
						if (referer != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("referer", referer));
						if (userAgent != null)
							messages[arrayIndex].addExtendedDataElement(createStringEDE("userAgent", userAgent));

						arrayIndex++;

						if (arrayIndex == MessageArraySize) {
							arrayIndex = 0;
							recordCount++;
							reset();
							return messages;
						}
						
						recordCount++;

						//Reset the local properties of a log record:
						reset();
					}
					else  {

						cgiDebuggingOutput.append(curLine);
			            cgiDebuggingOutput.append(ParserConstants.LINE_SEPARATOR);
					}
				}

				//Read the next line in the file:
				curLine = readALine();
			}

			if (cgiDebuggingOutput.length() > 0) {

                //Invalid log record:
                if (recordCount == 0)
                    throw new LogParserException(ParserUtilities.getResourceString("INVALID_ACCESS_LOG_ERROR_", file_path));

                else {

                    createCGIDebuggingOutputCBE();

                    arrayIndex++;

                    if (arrayIndex == MessageArraySize) {
                        arrayIndex = 0;
                        recordCount++;
                        reset();
                        return messages;
                    }
                    else {
                    	/* Bugzilla 69409 - Reset the cgi debugging output string buffer */
                    	cgiDebuggingOutput = new StringBuffer();
                    }

                    recordCount++;
                }
            }

			// If we are not logging the message then null the array elements that weren't set on this call
			if (arrayIndex == 0) {
				temp = null;
				setEndOfFile();
			}
			else {
			
				for (int i=arrayIndex; i < MessageArraySize; i++) {
					messages[i] = null;
				}
				temp = messages;
			}

			
			//Throw an exception if no valid access log records are parsed/logged:
			if (recordCount == 0) {
				throw new LogParserException(ParserUtilities.getResourceString("NO_ACCESS_LOG_RECORDS_ERROR_", file_path));
			}

		}
		catch (LogParserException l) {
		    throw l;
		}
		catch (Throwable t) {
			ParserUtilities.exceptionHandler(t, curLineNumber, curLine, ParserUtilities.getResourceString("ACCESS_LOG_PARSER_ERROR_"));
		}

		return temp;
	}

    private void reinitializeCBE() {

        //(Re)initialize the CBE:
        messages[arrayIndex].init();
        
        //Set the event's globalInstanceId property with a new GUID:
        messages[arrayIndex].setGlobalInstanceId(Guid.generate());

        //(Re)initialize the CBE's source component ID, set various properties and add to the CBE:
        ComponentIdentification sourceComponentID = eventFactory.createComponentIdentification();
        sourceComponentID.setLocation(localHostId);
        sourceComponentID.setLocationType(localHostIdFormat);
        sourceComponentID.setComponent(sourceID);
        sourceComponentID.setSubComponent(ParserConstants.UNKNOWN);
        //New for CBE 1.0.1:
        sourceComponentID.setComponentType(ParserConstants.APACHE_COMPONENT_TYPE);
		sourceComponentID.setComponentIdType(ParserConstants.APACHE_COMPONENT_ID_TYPE);

		messages[arrayIndex].setSourceComponentId(sourceComponentID);

        messages[arrayIndex].setSituation(createSituation());
    }

    /**
     * Generates the appropriate situation for a CBE using the return code information. 
     * 
     * @return ISituation The generic situation for a CBE. 
     * @since CBE 1.0.1 
     */
    private Situation createSituation() {

        //Initialize the CBE's situation and set various properties:
        Situation cbeSituation = eventFactory.createSituation();
        
        boolean defaultSituation = true;

        if (returnCode != null && returnCode.trim().length() > 0) 
        {
    			char firstChr = returnCode.charAt(0);
    			
    			if(returnCode.equals("202") || firstChr == '3' || firstChr == '1')  
    			{
    				/*
    			 	* 1xx,3xx,202 ==> REQUEST,SUCCESSFUL,REQUEST INITIATED,HTTP,EXTERNAL
    			 	*/
    				RequestSituation cbeRequestSituation = eventFactory.createRequestSituation() ;
    				cbeRequestSituation.setReasoningScope(ParserConstants.EXTERNAL_REASONING_SCOPE) ;
    				cbeRequestSituation.setSuccessDisposition(ParserConstants.SUCCESSFUL_SUCCESS_DISPOSITION) ;		
    				cbeRequestSituation.setSituationQualifier(ParserConstants.REQUEST_INITIATED_SITUATION_QUALIFIER) ;
    				cbeSituation.setCategoryName(ParserConstants.REQUEST_SITUATION_CATEGORY_NAME);
    				cbeSituation.setSituationType(cbeRequestSituation);
    				defaultSituation = false;
    			}
    			else  if(firstChr == '2') 
    			{
    				/*
    			 	* 2xx except 202==> REQUEST,SUCCESSFUL,REQUEST COMPLETE,HTTP,EXTERNAL
    			 	*/
    				RequestSituation cbeRequestSituation = eventFactory.createRequestSituation() ;
    				cbeRequestSituation.setReasoningScope(ParserConstants.EXTERNAL_REASONING_SCOPE) ;
    				cbeRequestSituation.setSuccessDisposition(ParserConstants.SUCCESSFUL_SUCCESS_DISPOSITION) ;		
    				cbeRequestSituation.setSituationQualifier(ParserConstants.REQUEST_COMPLETED_SITUATION_QUALIFIER) ;
    				cbeSituation.setCategoryName(ParserConstants.REQUEST_SITUATION_CATEGORY_NAME);
    				cbeSituation.setSituationType(cbeRequestSituation);
    				defaultSituation = false;
    			}
    			else  if(firstChr == '4' || firstChr == '5')
    			{
    				/*
    			 	* 4xx, 5xx ==> REQUEST,UNSUCCESSFUL,REQUEST INITIATED,HTTP,EXTERNAL
    			 	*/
    				RequestSituation cbeRequestSituation = eventFactory.createRequestSituation() ;
    				cbeRequestSituation.setReasoningScope(ParserConstants.EXTERNAL_REASONING_SCOPE) ;
    				cbeRequestSituation.setSuccessDisposition(ParserConstants.UNSUCCESSFUL_SUCCESS_DISPOSITION) ;		
    				cbeRequestSituation.setSituationQualifier(ParserConstants.REQUEST_INITIATED_SITUATION_QUALIFIER) ;
    				cbeSituation.setCategoryName(ParserConstants.REQUEST_SITUATION_CATEGORY_NAME);
    				cbeSituation.setSituationType(cbeRequestSituation);
    				defaultSituation = false;
    			}
        }

        if(defaultSituation) 
    	{
    	        //Unknown situation therefore use the generic:
    	        //Initialize the CBE's situation type, set various properties and add to the situation:
    	        ReportSituation cbeReportSituation = eventFactory.createReportSituation();
    	        cbeReportSituation.setReasoningScope(ParserConstants.INTERNAL_REASONING_SCOPE);
    	        cbeReportSituation.setReportCategory(ParserConstants.LOG_REPORT_CATEGORY);

    	        cbeSituation.setCategoryName(ParserConstants.REPORT_SITUATION_CATEGORY_NAME);
    	        cbeSituation.setSituationType(cbeReportSituation);
    	}

        return cbeSituation;
    }
    
    /**
     * Uses the return code of the message to determine the severity.
     */ 
    protected void determineSeverity(CommonBaseEvent message) {
    	
		if (returnCode != null && returnCode.trim().length() > 0) {
			char firstChr = returnCode.charAt(0);
			switch(firstChr) {
			
				case '4':
					message.setSeverity(ParserConstants.CBE_SEVERITY_MINOR);
					break;
					
				case '5':
					message.setSeverity(ParserConstants.CBE_SEVERITY_FATAL);
					break;
					
				default:
					message.setSeverity(ParserConstants.CBE_SEVERITY_3);
					
			}
    	        }
		else {
		
	        	message.setSeverity(ParserConstants.CBE_SEVERITY_3);
		}
    }
    
    /**
     * Parses a time stamp from an access log record, produces a Date object from the
     * parsed information, and converts the Date object into a XML DateTime String.
     *
     * @param startIndex Starting index of time stamp in curLine. 
     * @param endIndex Ending index of time stamp in curLine. 
     * @return true if time stamp is valid; otherwise false. 
     */
    protected boolean parseDate(int startIndex, int endIndex) {

        //Running example: [28/May/2003:16:36:39 -0400] --> 2003-05-28T16:36:39.000000-04:00

        //Verify that the time stamp is enclosed with square brackets:
        if ((curLine.charAt(startIndex) == '[') && (curLine.charAt(endIndex) == ']')) {

            //Find the time zone offset (e.g. -0400):
            int timeZoneIndex = curLine.indexOf('-', startIndex);

            //Check if the time zone offset is -GMT:
            if ((timeZoneIndex == -1) || (timeZoneIndex > endIndex)) {

                //Find the time zone offset (e.g. +GMT):
                timeZoneIndex = curLine.indexOf('+', startIndex);

                //Check if the time zone offset is +GMT:
				if ((timeZoneIndex == -1) || (timeZoneIndex > endIndex)) {
                    return false;
				}
            }

            //Parse the access log's time stamp excluding the time zone offset and square brackets (e.g. 28/May/2003:16:36:39) to a java.util.Date object:
            SimpleDateFormat formatter = new SimpleDateFormat(ParserConstants.APACHE_ACCESS_TIME_STAMP_FORMAT, Locale.US);
            Date creationDate = formatter.parse(curLine.substring((startIndex + 1), timeZoneIndex).trim(), new ParsePosition(0));

            //If the access log's time stamp is valid (e.g. non-null java.util.Date object), convert to its XML dateTime format:
            if (creationDate != null) {

                //Format the java.util.Date object to its XML dateTime format (e.g. "yyyy-MM-dd HH:mm:ss"):
                formatter = new SimpleDateFormat(ParserConstants.XML_DATETIME_FORMAT);
                currentTimeStamp = new StringBuffer(formatter.format(creationDate).trim());

                //Replace the first space with "T":
                currentTimeStamp.replace(10, 11, "T");

                //Add the fractional second value (e.g. .000000):
                currentTimeStamp.append(ParserConstants.SIX_ZERO);

                //NOTE:  Time Zone is in the form of [+/-]hh:mm.
                //Add the time zone offset sign (e.g. "+" or "-"):
                currentTimeStamp.append(curLine.charAt(timeZoneIndex));

                //Derive the time zone offset (e.g. hours or 04 and minutes or 00) and add it to the XML dateTime string delimited by a colon:
                currentTimeStamp.append(curLine.substring((timeZoneIndex + 1), (timeZoneIndex + 3)));
                currentTimeStamp.append(":");
                currentTimeStamp.append(curLine.substring((timeZoneIndex + 3), (timeZoneIndex + 5)));

                return true;
            }
        }

        return false;
    }

    /**
     * Main parsing routine for an access log record.
     *
     * @return   true if the access log record is successfully parsed; otherwise false. 
     */
    protected boolean parseLogRecord() {

    	/*A typical access log record looks like this example: 9.26.157.24 - - [28/May/2003:16:36:39 -0400] "GET / HTTP/1.1" 200 4757
     	*/
    	
     		StringTokenizer st = new StringTokenizer(curLine, " ");
    		int count = st.countTokens();
    		int fldCount = count;
    		if(count-- >= 1) {
    			clientIP = st.nextToken();
    		}	
            
            //Return false if the client's IP address is not a valid IPv4 address since it is an invalid log record: 
            if (!ParserUtilities.isValidIPAddress(clientIP))
                return false;

            //NOTE: Skip the identity of client (e.g. - or RFC1413).
            if(count-- >= 1) {
            	st.nextToken();
            }
			
            if(count-- >= 1) {
            	userID = st.nextToken();
                if(userID.equals("-")) {
                	userID = null;
                }
            }
            
            
            //Find the first '[' character in the log record:
            //Example:  9.26.157.24 - -* *|[|28/May/2003:16:36:39 -0400] "GET / HTTP/1.1" 200 4757
            int currentIndex = curLine.indexOf("[");

            //Return false if the '[' character cannot be found since it is an invalid log record: 
            if (currentIndex == -1)
                return false;

            //Advance the previous pointer to the current pointer: 
            //Example:  9.26.157.24 - - *|[|*28/May/2003:16:36:39 -0400] "GET / HTTP/1.1" 200 4757
            int previousIndex = currentIndex;

            //Find the first ']' character in the log record:
            //Example:  9.26.157.24 - - *[*28/May/2003:16:36:39 -0400|]| "GET / HTTP/1.1" 200 4757
            currentIndex = curLine.indexOf("]", previousIndex);

            //Return false if the ']' character cannot be found since it is an invalid log record: 
            if (currentIndex == -1)
                return false;

            //Parse the time stamp (e.g. [28/May/2003:16:36:39 -0400]):
            if (!parseDate(previousIndex, currentIndex))
                return false;

            if(count-- >= 1) {
            	st.nextToken();
            }	

            if(count-- >= 1) {
            	st.nextToken();
            }	
            
            if(count-- >= 1) {
            	method = st.nextToken();
            }	
            
            if(method != null && method.charAt(0) == '"') {
            	
            	method = method.substring(1);
            	if(method.length() == 2 && method.charAt(0) == '-') {
            		method = null;

            		message = curLine.substring(currentIndex + 1).trim();
            		 
            		if(count-- >= 1) {
                		returnCode = st.nextToken();
                	}	
                
            		if(returnCode.equals("-")) {
            			returnCode = null;
            		}
                
            		if(count-- >= 1) {
            			fileSize = st.nextToken();
            			if(fileSize.equals("-")) {
            				fileSize = null;
            			}
            		}	
                
            		if(count-- >= 1) {
            			userAgent = st.nextToken();
                	
            			if(userAgent.charAt(0) == '"') {
            				userAgent = userAgent.substring(1);
            			}
            			
            			int userAgentLength = userAgent.length();
                    
            			if(userAgent.charAt(userAgentLength - 1) == '"') {
                    	
            				userAgent = userAgent.substring(0, userAgentLength - 1);
            			}
            			
            			if(userAgent.equals("-") == true) {
            				userAgent = null;
            			}
            			
            		}
            		
            		if(count-- >= 1) {
                        	referer = st.nextToken();

                        	if(referer.charAt(0) == '"') {
                        		referer = referer.substring(1);
                        	}

                        	int refererLength = referer.length();
                            
                            if(referer.charAt(refererLength - 1) == '"') {
                            	
                            	referer = referer.substring(0, refererLength - 1);
                            }

                            if(referer.equals("-") == true) {
                				referer = null;
                			}

                    }

            		return true;
            	}
            }
            
            message = curLine.substring(currentIndex + 1).trim();
            
            String tmpFileName = message.substring(message.indexOf(' '),message.lastIndexOf('"') - 1);
            if(tmpFileName.substring(tmpFileName.lastIndexOf(' ')).indexOf("HTTP") > -1)
    		{
            	fileName = tmpFileName.substring(0,tmpFileName.lastIndexOf(' '));
    		}
            else
            {
            	fileName = tmpFileName;
            }
            
            /*
             * The below code has been commented and the above code has been added to support spaces in the filePath:
             * Ex : 127.127.0.1 - - [28/May/2003:16:36:39 -0400] "GET /Tom and Jerry.gif HTTP/1.1" 200 4757
             */
            
         /* 
            if(count-- >= 1){
            	
            	fileName = st.nextToken();
                int fileNameLength = fileName.length();
                
                if(fileName.charAt(fileNameLength - 1) == '"') {
                	
                	fileName = fileName.substring(0, fileNameLength - 1);
                }

            }
          */

            String temp = null;

            if(count-- >= 1) {
            	temp = st.nextToken();
            }
            
            /*
             * This code has been added to support spaces in the filePath:
             * Ex : 9.26.157.51 - - [28/May/2003:16:36:39 -0400] "GET /Tom and Jerry.gif HTTP/1.1" 200 4757
             * 
             * The condition temp.indexOf("\\\"") > -1 is added to support the \" sequence
             * Ex : 9.26.64.34 - - [15/Jan/2003:11:52:42 -0500] "GET /cgi-bin/htsearch?Exclude=\"%60/etc/passwd%60\" HTTP/1.0" 404 210
             */
            
            while((!(temp.endsWith("\"") || (temp.indexOf('"') > -1))) || 
            		temp.indexOf("\\\"") > -1)
            {
            	if(count-- >= 1) {
                	temp = st.nextToken();
                }
            	else
            	{
            		break;
            	}
            }
            
            
            /*
            if(temp != null && temp.indexOf("HTTP") == -1) {
            	
            	returnCode = temp;
            }
            else {
	        	if(count-- >= 1) {
	        		returnCode = st.nextToken();
	        	}	
            }
            */
            
        	if(count-- >= 1) {
        		returnCode = st.nextToken();
        	}
        	
            if(returnCode.equals("-")) {
            	returnCode = null;
            }
            
            if(count-- >= 1) {
            	fileSize = st.nextToken();
                if(fileSize.equals("-")) {
                	fileSize = null;
                }
            }	
            
            if(count-- >= 1) {
            	userAgent = st.nextToken();
            	
            	if(userAgent.charAt(0) == '"') {
            		userAgent = userAgent.substring(1);
            	}

            	int userAgentLength = userAgent.length();
                
                if(userAgent.charAt(userAgentLength - 1) == '"') {
                	
                	userAgent = userAgent.substring(0, userAgentLength - 1);
                }
    			if(userAgent.equals("-") == true) {
    				userAgent = null;
    			}

            }
        
            if(count-- >= 1) {
            	referer = st.nextToken();

            	if(referer.charAt(0) == '"') {
            		referer = referer.substring(1);
            	}

            	int refererLength = referer.length();
                
                if(referer.charAt(refererLength - 1) == '"') {
                	
                	referer = referer.substring(0, refererLength - 1);
                }

                if(referer.equals("-") == true) {
    				referer = null;
    			}

            }
            return true;
    }

    private void createCGIDebuggingOutputCBE() {

        //Reinitialize the CBE:
        reinitializeCBE();

        //Add information from the access log record to the CBE:
        messages[arrayIndex].setSequenceNumber(++duplicateTimeStampCounter);
        messages[arrayIndex].setCreationTime(previousTimeStamp);
        messages[arrayIndex].setMsg(cgiDebuggingOutput.toString().trim());
        messages[arrayIndex].setSeverity(ParserConstants.CBE_SEVERITY_3);
        messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.CGI, cgiDebuggingOutput.toString().trim()));
    }

    /**
     * Resets the parsed values of a log record before next record is parsed.
     */
    protected void reset() {
        userID = null;
        method = null;
        returnCode = null;
        clientIP = null;
        fileName = null;
        fileSize = null;
        referer = null;
        userAgent = null;
		message = null;
        currentTimeStamp = null;
        cgiDebuggingOutput = new StringBuffer();
        
    }
}