package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogParserException.java,v 1.6 2005/03/23 08:08:48 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Subclass of java.lang.Exception.  Parser classes in the org.eclipse.hyades.logging.parsers
 * package throw exceptions of this class when a log cannot be parsed.  A LogParserException
 * can wrap another exception caught during the parse process.
 */
public class LogParserException extends Exception {

    /**
     * Variable to hold wrapped existing exception.
     */
    private Throwable exception;

    /**
     * Create a new LogParserException with no message.
     */
    public LogParserException() {
        super();
        this.exception = null;
    }

    /**
     * Create a new LogParserException.
     *
     * @param message The error or warning message.
     */
    public LogParserException(String message) {
   		super(message);
 
        this.exception = null;
    }

    /**
     * Create a new LogParserException wrapping an existing exception.
     *
     * <p>The existing exception will be embedded in the new
     * one.</p>
     *
     * @param e The exception to be wrapped in a LogParserException.
     */
    public LogParserException(Throwable e) {
        super();
        this.exception = e;
    }

    /**
     * Create a new LogParserException from an existing exception.
     *
     * <p>The existing exception will be embedded in the new
     * one, but the new exception will have its own message.</p>
     *
     * @param message The detail message.
     * @param e The exception to be wrapped in a LogParserException.
     */
    public LogParserException(String message, Throwable e) {
        super(message);
        this.exception = e;
    }

    /**
     * Return a detail message for this exception.
     *
     * <p>If there is an embedded exception, and if the LogParserException
     * has no detail message of its own, this method will return
     * the detail message from the embedded exception.</p>
     * <p>If there is an embedded exception, and the LogParserException
     * has a detail message of its own, this method will append the
     * embedded exception message to the LogParserException message and
     * return it.</p>
     *
     * @return The error or warning message.
     */
    public String getMessage() {
        String message = super.getMessage();
        String excMessage = null;

        // If there is an embedded exception Then get its message
    	if (exception != null) {
    		excMessage = exception.getMessage();
    		if (excMessage == null || excMessage.length() == 0) {
    			excMessage = exception.toString();
    		}
    	}
    	// If three is no detail message then return the embedded exception 
    	// message if one exists.
        if (message == null || message.length() == 0) {
        	if (excMessage != null) {
        		return excMessage;
        	}
        	else {
        		return null;
        	}
        }
        // If there is a detail message and an embedded exception message then
        // concatenate them.
        else if (excMessage != null){
            return message + "  " + ParserUtilities.getResourceString("EXCEPTION") + " = " + excMessage;
        }
        // If there is only a detail message then return that
        else {
            return message;
        }
    }

    /**
     * Return the embedded exception, if any.
     *
     * @return The embedded exception, or null if there is none.
     */
    public Throwable getException() {
        return exception;
    }

    /**
     * Override this method in the superclass in order to output the stack trace
     * for the wrapped exception.
     */
    public void printStackTrace() {
        super.printStackTrace();
        if (exception != null) {
	        System.err.println(ParserUtilities.getResourceString("EMBEDDED_EXCEPTION"));
	        exception.printStackTrace();
		}
    }

    /**
     * Override this method in the superclass in order to output the stack trace
     * for the wrapped exception.
     *
     * @param s The stream to which the stack trace is written
     */
    public void printStackTrace(java.io.PrintStream s) {
        super.printStackTrace(s);
        if (exception != null) {
	        s.println(ParserUtilities.getResourceString("EMBEDDED_EXCEPTION"));
	        exception.printStackTrace(s);
		}
    }

    /**
     * Override this method in the superclass in order to output the stack trace
     * for the wrapped exception.
     *
     * @param s The writer to which the stack trace is written
     */
    public void printStackTrace(java.io.PrintWriter s) {
        super.printStackTrace(s);
        if (exception != null) {
	        s.println(ParserUtilities.getResourceString("EMBEDDED_EXCEPTION"));
	        exception.printStackTrace(s);
		}
    }

}
