package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RemoteLogParserLoader.java,v 1.13 2005/05/07 03:49:44 dnsmith Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.lang.reflect.InvocationTargetException;
import java.util.Hashtable;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.hyades.logging.commons.Logger;
import org.eclipse.hyades.logging.core.IPreemptedLogger;
import org.eclipse.hyades.logging.core.LoggingAgent;

/**
 * This class handles initialization (e.g. calling the setUserInput() API) 
 * and invocation (e.g. calling the parse() API) of a remote log
 * file parser based on the parameters passed to the main() API and 
 * logs all parsed log record(s) to a Logging Agent.
 * 
 * Generally, this class is invoked by a local instance of the Workbench 
 * via the Agent Controller running on a remote host.
 * This class is invoked by a local instance of the Workbench by executing 
 * the Agent Controller's RemoteLogParserLoader executable based on the following 
 * Agent Controller configuration entry:
 * 
 * 	 <Application configuration="default" executable="RemoteLogParserLoader" path="%JAVA_PATH%\java.exe" location="%JAVA_PATH%">
 * 	 	<Parameter value="org.eclipse.hyades.logging.parsers.RemoteLogParserLoader" position="prepend"/>
 * 		<Variable name="CLASSPATH" value="%RASERVER_HOME%\lib\logutil.jar" position="prepend"/>
 * 	 </Application>
 * 
 * The possible parameters which may be passed to the main() API are as follows:
 * 
 * [org.eclipse.hyades.logging.parsers.RemoteLogParserLoader] <parser package name>.<parser class name> [<key 1>=<value 1> ... <key n>=<value n>] 
 * 
 * where:
 *     <parser class name> the class name of the remote log file parser
 *     <key n>             the name of the nth configuration variable required by the remote log file parser such as 'file_path'
 *     <value n>           the value of the nth configuration variable required by the remote log file parser such as 'C:\logs\access.log'
 * 
 * The parsed log record(s) are returned to the local instance of the Workbench via a Logging Agent and the Agent Controller.
 * 
 * All errors and/or exceptions are caught and returned to the local instance of the Workbench via a Logging Agent and the Agent Controller. 
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version	August 27, 2004
 * @since   April 17, 2003
 */
public class RemoteLogParserLoader {

    /**
     * Handles the initialization (e.g. calling the setUserInput() API) and invocation 
     * (e.g. calling the parse() API) of a log file parser.  
     * 
     * The possible parameters which may be passed to the main() API are as follows:
     * 
     * [org.eclipse.hyades.logging.parsers.RemoteLogParserLoader] <parser package name>.<parser class name> [<key 1>=<value 1> ... <key n>=<value n>] 
     * 
     * where:
     *     <parser class name> the class name of the remote log file parser
     *     <key n>             the name of the nth configuration variable required by the remote log file parser such as 'file_path'
     *     <value n>           the value of the nth configuration variable required by the remote log file parser such as 'C:\logs\access.log'
     * 
     * The parsed log record(s) are returned to the local instance of the Workbench via a Logging Agent and the Agent Controller.
     * 
     * All errors and/or exceptions are caught and returned to the local instance of the Workbench via a Logging Agent and the Agent Controller. 
     * 
     * @param args the array of non-null String parameters for the initialization of a log file parser
     */
    public static void main(String[] args) {

    	// Parser status monitoring thread
    	Thread statusMonitorThread = null;
    	
    	// Log to log the parsed log file records to
        Log logger = null;
        
        // Log to log parser status to.  The status includes parsing progress information and error messages.
        Log statusLogger = null;
        
        long maxWaitTimeInMillis = 300000;
        long statusLoggerWaitTime = 10000;  /* bugzilla 92984 - increase wait time from 2000 to handle network delay issues */

        try {
            maxWaitTimeInMillis = Long.parseLong(System.getProperty("org.eclipse.hyades.logging.parsers.maxWaitTimeInMillis"));
        }
        catch (Exception e) {
        }

        try {

        	//Set Apache logger:
        	System.setProperty("org.apache.commons.logging.Log", "org.eclipse.hyades.logging.commons.Logger");

            logger = LogFactory.getLog(ParserConstants.REMOTE_LOG_LOADER_AGENT_NAME);

        	//Get a GLA Logger instance to log parser status to:

            LogFactory logFactory = new GLALoggerFactory();
            statusLogger = logFactory.getInstance(ParserConstants.REMOTE_LOG_LOADER_STATUS_AGENT_NAME);

            int parameterNum = args.length;

            //Throw an exception if no parameters have been passed:
            if (parameterNum == 0)
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_PARAMETER_ERROR_"));

            //NOTE: The first parameter could be 'org.eclipse.hyades.logging.parsers.RemoteLogParserLoader' if specified in the serviceconfig.xml file:
            int parameterIndex = 0;

            //If the first parameter is 'org.eclipse.hyades.logging.parsers.RemoteLogParserLoader', use the next parameter:
            if (args[parameterIndex].trim().equals(ParserConstants.REMOTE_LOG_LOADER_CLASS))
                parameterIndex++;

            String parserClassName = null;

            //Retrieve and set the name of the log parser class or throw an exception if no class name parameter has been passed:
            //NOTE:  The parser class name MUST be of the format 'org.eclipse.hyades.logging.parsers.<parser class name>':
            if (parameterNum > parameterIndex)
                parserClassName = args[parameterIndex];
            else
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CLASSNAME_PARAMETER_ERROR_"));
           
            //Use the next parameter:
            parameterIndex++;

            //Throw an exception if no initialization parameter(s) for the log parser class have been passed:
            if (parameterNum <= parameterIndex)
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));

            Hashtable parserParameters = new Hashtable();
            String keyValuePair = null;
            int equalsSignIndex = -1;

            //Retrieve and set the parameter(s) for the log parser class:
            while (parameterNum > parameterIndex) {

                keyValuePair = args[parameterIndex++];
                equalsSignIndex = keyValuePair.indexOf('=');

                if (equalsSignIndex != -1) {
                    parserParameters.put(keyValuePair.substring(0, equalsSignIndex), keyValuePair.substring(equalsSignIndex + 1));
                }
            }

            if (parserParameters.isEmpty())
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));

            //Resolve the client's language and country codes for locale-specific message resolution:
            Locale clientLocale = null;
            
            if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)){
                
                String clientLanguage = ((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)));
                
                if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_COUNTRY_KEY)){

                    String clientCountry = ((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_LANGUAGE_KEY)));

                    if(parserParameters.containsKey(ParserConstants.CLIENT_LOCALE_VARIANT_KEY)){
                        clientLocale = new Locale(clientLanguage,clientCountry,((String)(parserParameters.get(ParserConstants.CLIENT_LOCALE_VARIANT_KEY))));
                    }
                    else{
                        clientLocale = new Locale(clientLanguage,clientCountry);
                    }
                }
                else{
                    clientLocale = new Locale(clientLanguage);
                }
            }
            
            
            //Assume this is a PreemptedLogger and recover if necessary:
            if (logger instanceof IPreemptedLogger) {

                ((IPreemptedLogger) (logger)).waitUntilLogging(maxWaitTimeInMillis);

                if (!((IPreemptedLogger) (logger)).isLogging()) {
                    throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_MONITORING_ERROR_", clientLocale,parserClassName, String.valueOf(((double) (maxWaitTimeInMillis)) / 1000.0)));
                }
            }

            //Assume this is a PreemptedLogger and recover if necessary:
            // The client might be old and will not know about the status
            // logger so we won't wait as long.
            if (statusLogger instanceof IPreemptedLogger) {

                ((IPreemptedLogger) (statusLogger)).waitUntilLogging(statusLoggerWaitTime);

                if (!((IPreemptedLogger) (statusLogger)).isLogging()) {
                	// The client might be old and will not start monitoring the
                	// status logger.  Therefore clean up the statusLogger and continue
                	// without it.
                	((IPreemptedLogger) statusLogger).finalize();
                	statusLogger = null;
                	// throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_MONITORING_ERROR_", clientLocale,parserClassName, String.valueOf(((double) (maxWaitTimeInMillis)) / 1000.0)));
                }
            }

            try {

                if (logger instanceof Logger) {
                    ((Logger) logger).setLevel(Logger.TRACE_LEVEL);
                }

                if (statusLogger != null && statusLogger instanceof GLALogger) {
                    ((Logger) statusLogger).setLevel(Logger.TRACE_LEVEL);
                }
  
                Class parserClass = Class.forName(parserClassName);
                Object parser = parserClass.newInstance();
                               
                parserClass.getMethod("setUserInput", new Class[] { parserParameters.getClass()}).invoke(parser, new Object[] { parserParameters });
                
                if (statusLogger != null) {
	                // Pass the status logger to the parser
	                parserClass.getMethod("setParserLogger", new Class[] { Class.forName("org.apache.commons.logging.Log")}).invoke(parser, new Object[] { statusLogger });
	                
	                // Create the parser status monitoring thread
	                StatusMonitor statusMonitor = new StatusMonitor(parser, parserClass, statusLogger);
	                statusMonitorThread = new Thread(statusMonitor);
	                
	                // Start the parser status monitoring thread
	                statusMonitorThread.start();
                }
                
                // Parse the log file, passing the logger to send the parsed data to
                parserClass.getMethod("parse", new Class[] { Class.forName("org.apache.commons.logging.Log")}).invoke(parser, new Object[] { logger });
                
                if (statusLogger != null) {
                	// Parsing has completed so stop the status monitoring thread.
                	statusMonitorThread.interrupt();
                }
            }
            //If the called method throws a LogParserException:
            catch (InvocationTargetException i) {
            	// If one of the parser methods threw an exception then stop the status monitoring thread 
            	// if there is one.
            	if (statusMonitorThread != null) {
            		statusMonitorThread.interrupt();
            	}
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_EXECUTION_ERROR_", clientLocale, parserClassName, i.getTargetException().getLocalizedMessage()));
            }
            catch (Throwable t) {
            	// Stop the status monitoring thread if there is one.
            	if (statusMonitorThread != null) {
            		statusMonitorThread.interrupt();
            	}
                throw new LogParserException(ParserUtilities.getResourceString("REMOTE_LOG_PARSER_INITIALIZATION_ERROR_", clientLocale, parserClassName),t);
            }
        }
        catch (Throwable t) {
        	// If there was an error from parsing or that prevented parsing then return it to the
        	// client via a logging agent.
        	String errMessage = t.getMessage();
        	
        	if (errMessage == null) {
        		errMessage = t.toString();
        	}
        	
            LoggingAgent errorLogger = new LoggingAgent(ParserConstants.REMOTE_LOG_LOADER_ERROR_AGENT_NAME);

            errorLogger.waitUntilLogging(maxWaitTimeInMillis);

            if (errorLogger.isLogging()) {
                errorLogger.write(errMessage);
                // Sleep for a couple seconds to give the client time to 
                // receive the message before deregistering the agent.
                try {
                	Thread.sleep(2000);
                }
                catch (InterruptedException ie) {
                	// Ignore the interruption.
                }
            }
            else {
                System.err.println(errMessage);
            }
            errorLogger.deregister(); 
        }

        if ((logger != null) && (logger instanceof IPreemptedLogger)) {
            ((IPreemptedLogger) logger).finalize();
        }
        
        if ((statusLogger != null) && (statusLogger instanceof IPreemptedLogger)) {
            ((IPreemptedLogger) statusLogger).finalize();
        }

    }
}
