/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationClient.java,v 1.3 2005/05/20 21:38:57 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.client;

import org.eclipse.hyades.automation.client.strategies.AbstractExecutionStrategy;
import org.eclipse.hyades.automation.core.Service;

/**
 * @author Scott E. Schneider
 * 
 * The automation client is the client-side intrastructure that allows easy
 * automation of automatable published services and is the primary access-point
 * for all client adapters. A client adapter allows for the adaptation of a
 * particular client's invocation paradigm into the standard expected calls.
 * These adapted calls then trigger an appropriate invocation of the automation
 * server using the Eclipse headless mode feature from the command-line (or
 * other methods such as directly communicating with the Eclipse starter
 * platform class).
 * 
 * The automation client can launch another Java VM process that is independent
 * of the automation launcher's process. This Java VM has its own classpath and
 * is the heavyweight component of the two key automation components. The
 * automation launcher is the lightweight component of the pair having few
 * dependencies and abstracting the complexities and need to understand Eclipse
 * headless mode operation. It also shields clients from the actual components
 * servicing the automation requests and therefore also protects the automation
 * server's ability to change over time without clients having to change (given
 * there is still a mapping from client request to server functionality main-
 * tained).
 * 
 * The automation client can also sit in-process with the Eclipse instance that
 * hosts the automation server by constructing and configuring the automation
 * client appropriately.
 * 
 * The automation client is associated with one and only one root, to automate
 * instances of TPTP rooted at different locations create a new automation
 * client via the automation client factory. It will maintain exactly one
 * instance per root (and per execution strategy). An automation client also is
 * associated with only one execution strategy -- to establish a client context
 * that has a different execution strategy one must also create a new automation
 * client via the factory.
 * 
 * @provisional
 */
public interface AutomationClient extends Service.Discoverable, Service.Executable, Service.Requestable {

    /**
     * The default automation client
     * 
     * @author Scott E. Schneider
     */
    static class Default implements AutomationClient {

        /**
         * The Eclipse root/home directory
         */
        private String root;

        /**
         * The execution strategy to use when executing the automation command
         * with this client
         */
        private final Service.Executable strategy;

        /**
         * Constructs the default automation client
         * 
         * @param root
         *            the Eclipse root/home directory
         */
        private Default(String root) {
            this(root, AbstractExecutionStrategy.Factory.getInstance().createDefaultStrategy());
        }

        /**
         * Constructs the default automation client with the specified stategy
         * 
         * @param root
         *            the Eclipse root/home directory
         * @param strategy
         *            the strategy to use
         */
        private Default(String root, Service.Executable strategy) {
            this.root = root;
            this.strategy = strategy;
        }

        /**
         * Creates the service proxy, represents the service before one can be
         * constructed on the server-side (within Eclipse)
         * 
         * @param identifier
         *            the service identifier that identifies the service to
         *            request/execute
         * @return the service to execute
         */
        private Service createServiceProxy(String identifier) {
            return new ServiceProxy(this, this.strategy, identifier);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.automation.core.Service.Discoverable#discover(java.lang.String)
         */
        public boolean discover(String identifier) {
            return true;
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service)
         */
        public Object execute(Service service) {
            return service.execute();
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.automation.client.AutomationClient#getRoot()
         */
        public String getRoot() {
            return this.root;
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.hyades.automation.core.Service.Requestable#request(java.lang.String)
         */
        public Service request(String identifier) {
            return this.createServiceProxy(identifier);
        }

    }

    /**
     * @author Scott E. Schneider
     * 
     * The automation factory provides the public factory methods for all
     * automation-related elements that provide the ability to substitute the
     * actual concrete class to be used when creating instances.
     * 
     * This will be supported by setting the appropriate properties in a
     * configuration file. Downstream products can edit this file and provide an
     * additional classpath and qualified-class to be used for creating concrete
     * elements. All elements in the factory are returned and manipulated
     * through their interfaces. (this part is not implemented in this release)
     */
    final class Factory {

        /**
         * The automation factory
         */
        private static final Factory factory;

        static {
            factory = new Factory();
        }

        /**
         * Access the singleton factory
         * 
         * @return the singleton factory
         */
        public static final Factory getInstance() {
            return Factory.factory;
        }

        /**
         * Limit instantiation to this class
         */
        private Factory() {
        }

        /**
         * Creates the automation client given the following root
         * 
         * @param root
         *            the Eclipse root or home directory
         * @return the automation client to use
         */
        public AutomationClient createAutomationClient(String root) {
            return new AutomationClient.Default(root);
        }

        /**
         * Creates the automation client to use given the root directory of
         * Eclipse and the launching/execution strategy to use
         * 
         * @param root
         *            the Eclipse root directory
         * @param strategy
         *            the launch/execution strategy to use for this automation
         *            client
         * @return the automation client ready to use
         */
        public AutomationClient createAutomationClient(String root, AbstractExecutionStrategy strategy) {
            return new AutomationClient.Default(root, strategy);
        }

    }

    /**
     * The Eclipse root or home directory
     * 
     * @return this determines which Eclipse will be invoked
     */
    public String getRoot();

}