/********************************************************************** 
 * Copyright (c) 2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ServiceProxy.java,v 1.1 2005/05/17 03:49:04 sschneid Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.automation.core.AbstractService;
import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.Service.Memento;

/**
 * The service proxy classes services as a proxy between the calling request and
 * the actual class that implements the extension point called out by the
 * service identifier (the service implementation is a java class that
 * implements the service interface and the appropriate extension point and id).
 * 
 * @author Scott E. Schneider
 */
class ServiceProxy extends AbstractService {

    /**
     * The concrete service implementation given the requested service
     * identifier
     */
    private Service serviceImplementation;

    /**
     * Construct a service proxy given the specified state, this serves as a
     * proxy between the calling code and the concrete implementation that
     * implements the appropriate extension point, this allows the memento state
     * to be revived and the service identifier to be read without violating
     * encapsulation.
     * 
     * @param state
     *            the state of the service to revive
     */
    ServiceProxy(Memento state) {

        // Plug the memento state in hydrating service state
        super(state);

        // Find service implementation for this service identifier
        IExtension extension = Platform.getExtensionRegistry().getExtension("org.eclipse.hyades.execution.core.service",
                this.getIdentifier());

        if (extension != null) {

            // Retrieve the extension point implementors
            IConfigurationElement[] implementors = extension.getConfigurationElements();

            // Currently, we use the first returned service (future will have
            // execution strategies)
            try {
                Object executableExtension = implementors[0].createExecutableExtension("implementation");
                if (executableExtension instanceof Service) {
                    this.serviceImplementation = (Service) executableExtension;
                    this.serviceImplementation.setMemento(state);

                }
            } catch (CoreException e) {
                e.printStackTrace();
            }

        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.automation.core.Service#execute()
     */
    public Object execute() {
        return (this.serviceImplementation != null ? this.serviceImplementation.execute() : null);
    }

}