package org.eclipse.hyades.tests.logging.log4j;

import java.net.InetAddress;
import java.net.UnknownHostException;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;


/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Log4JLoggingTest.java,v 1.3 2005/02/16 22:20:54 qiyanli Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/** 
 * CLASS: Log4JLoggingTest.java
 * 
 * 
 * DESCRIPTION:  Sample class to demonstrate the usage of the Hyades support for logging Common Base Event log records using Apache Log4J logging APIs.
 * 
 * 
 * ASSUMPTIONS:  1) This class must be executed using JRE 1.4.0 or above.
 *               2) The required JAR files must be specified in the JVM's classpath system variable.
 *               3) JRE 1.4.0 or above must be specified in the PATH environment variable.
 *               4) When launching this application from the Profiling and Logging Perspective of the Workbench, logging is enabled in the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Enable logging) 
 *                  and the 'org.eclipse.hyades.tests.logging.logging.log4j.Log4JLoggingTest' logging agent is added to the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Logging Agents --> New agent --> Add Agent).
 *               5) When either launching this application from the Profiling and Logging Perspective of the Workbench or attaching to a logging agent using the Workbench, the Hyades Data Collection Engine application or service is configured and running.
 *               6) When launching this application from the Profiling and Logging Perspective of the Workbench, the application launch configuration or the Hyades Data Collection Engine application or service classpath contains Apache Log4J 1.2.8 or above (not provided) in the classpath.
 *               7) The 'log4j.configuration' environment variable is set to the Apache Log4J logging's configuration file's (Log4JLoggingTest.xml) absolute path and name in the form of a URL.  For example, <i>file:/&lt;plug-ins directory&gt;</i>/org.eclipse.hyades.logging.log4j/samples/org/eclipse/hyades/tests/logging/log4j/Log4JLoggingTest.xml. 
 *
 *                 
 * DEPENDENCIES: The following JAR files are required to compile and execute Log4JLoggingTest.java:
 * 
 * 		         hexr.jar				-		org.eclipse.hyades.execution.remote
 * 		         hllog4j.jar		    -		org.eclipse.hyades.logging.log4j
 * 		         hlcore.jar				-		org.eclipse.hyades.logging.core
 * 		         hlcb101.jar			-		org.eclipse.hyades.logging.core
 *               common.jar             -       org.eclipse.emf.common\runtime
 *               ecore.jar              -       org.eclipse.emf.ecore\runtime
 *               log4j-1.2.8.jar        -       org.apache.jakarta_log4j_logging
 * 
 * *Not provided.
 * <p>
 * 
 * 
 * @author	Paul E. Slauenwhite
 * @version	November 24, 2004
 * @since	July 20, 2004
 * @since   1.0.1
 * @see		org.apache.log4j.Logger
 * @see     org.apache.log4j.Level
 * @see		org.apache.log4j.Appender
 * @see		org.apache.log4j.Layout
 * @see		org.apache.log4j.spi.Filter
 * @see		org.eclipse.hyades.logging.log4j.LoggingAgentAppender
 * @see		org.eclipse.hyades.logging.log4j.SingleLoggingAgentAppender
 * @see		org.eclipse.hyades.logging.events.cbe.CommonBaseEvent 
 * @see		org.eclipse.hyades.logging.events.cbe.EventFactory
 * @see     org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext
 */
public class Log4JLoggingTest {

	//Amount of time (seconds) to wait to allow time to attach to the logging agent:
	private static final int WAITTIME = 30;

	//The name of the Log4J logger:
	private static final String LOGGERS_NAME = "org.eclipse.hyades.tests.logging.log4j.Log4JLoggingTest";

	/**
	 * IP address (IPv4) of the local host, otherwise "127.0.0.1". 
	 */
	private static String localHostIP = null;

	static {
		try {
			localHostIP = InetAddress.getLocalHost().getHostAddress();
		}
		catch (UnknownHostException u) {
			localHostIP = "127.0.0.1";
		}
	}

	public static void main(String[] args) {

		try {

			//OPTIONAL: Re-load the Log4J logging configuration XML file:
		    //DOMConfigurator.configure("Log4JLoggingTest.xml");

			//Create a logger named 'org.eclipse.hyades.tests.logging.log4j.Log4JLoggingTest':
			Logger logger = Logger.getLogger(LOGGERS_NAME);

			//NOTE: Alternatively, set the logger's level programmatically:
			//Set the logger to log warning or lower messages:
			//logger.setLevel(Level.WARN);

			//NOTE: Alternatively, create and set the logger's console appender programmatically:
			//Create a new instance of a console appender:
			//Appender consoleAppender = new ConsoleAppender();

			//NOTE: Alternatively, set the console appender's log record filter programmatically via the appender.addFilter() API:
			//Set the console appender's log record filter:
			//consoleAppender.addFilter(new CommonBaseEventFilter());

			//NOTE: Alternatively, set the logger's console appender programmatically via the logger.addAppender() API:
			//Add the console appender to the logger:
			//logger.addAppender(consoleAppender);

			//NOTE: Alternatively, create and set the logger's Logging Agent appender programmatically:
			//Create a new instance of a Logging Agent appender:
			//Appender loggingAgentAppender = new LoggingAgentAppender();

			//NOTE: Alternatively, set the Logging Agent appender's log record filter programmatically via the appender.addFilter() API:
			//Set the Logging Agent appender's log record filter:
			//loggingAgentAppender.addFilter(new CommonBaseEventFilter());

			//NOTE: Alternatively, set the logger's Logging Agent appender programmatically via the logger.addAppender() API:
			//Add the Logging Agent appender to the logger:
			//logger.addAppender(loggingAgentAppender);

			//NOTE: Alternatively, create and set the logger's Logging Agent appender programmatically using the Single Logging Agent appender:
			//Create a new instance of a Single Logging Agent appender:
			//Appender singleLoggingAgentAppender = new SingleLoggingAgentAppender();

			//NOTE: Alternatively, set the Single Logging Agent appender's log record filter programmatically via the appender.addFilter() API:
			//Set the Single Logging Agent appender's log record filter:
			//singleLoggingAgentAppender.addFilter(new CommonBaseEventFilter());

			//NOTE: Alternatively, set the logger's Single Logging Agent appender programmatically via the logger.addAppender() API:
			//Add the Single Logging Agent appender to the logger:
			//logger.addAppender(singleLoggingAgentAppender);

			//Retrieve the instance of the Event Factory:
			EventFactory eventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();

			//Create a new instance of a report situation:
			ReportSituation reportSituation = eventFactory.createReportSituation();
			reportSituation.setReasoningScope("INTERNAL");
			reportSituation.setReportCategory("LOG");

			//Create a new instance of a situation:
			Situation situation = eventFactory.createSituation();
			situation.setCategoryName("ReportSituation");
			situation.setSituationType(reportSituation);

			//Create a new instance of a Source Component:
			ComponentIdentification sourceComponentId = eventFactory.createComponentIdentification();
			sourceComponentId.setLocation(localHostIP);
			sourceComponentId.setLocationType("IPV4");
			sourceComponentId.setExecutionEnvironment("Java");
			sourceComponentId.setComponent("Log4JLoggingTest");
			sourceComponentId.setSubComponent("Log4JLoggingTest.main(String[] args)");
			sourceComponentId.setComponentIdType("Application");
			sourceComponentId.setComponentType("HyadesLoggingSamples");

			//Create a new instance of a Common Base Event:
			CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();
			
			//Set the event's globalInstanceId property with a new GUID:
	        commonBaseEvent.setGlobalInstanceId(Guid.generate());
			
	        commonBaseEvent.setSourceComponentId(sourceComponentId);
			commonBaseEvent.setSituation(situation);
			commonBaseEvent.setCreationTimeAsLong(System.currentTimeMillis());
			commonBaseEvent.setSeverity(((short) (60)));
			commonBaseEvent.setMsg("Apache Log4J Common Base Event log message");

			//Log the Common Base Event log record:
			if (logger.isEnabledFor(Priority.ERROR)){
				logger.error(commonBaseEvent);
			}

			//Attempt to log a message at a higher logging level:
			if (logger.isEnabledFor(Priority.DEBUG)){
				logger.debug("Apache Log4J invalid log message.");
			}

			//Wait for 2 seconds until all log messages are written to the console appender (i.e. standard error):
			try {
				Thread.sleep(2000);
			}
			catch (InterruptedException i) {
			}

			System.out.println("Please attach to the 'org.eclipse.hyades.tests.logging.log4j.Log4JLoggingTest' logging agent in the next " + WAITTIME + " second" + (WAITTIME == 1 ? "" : "s") + ":");
			System.out.println("NOTE: When attaching to a logging agent using the Workbench, please remember to enable logging in the Profiling and Logging preferences (Window --> Preferences --> Profiling and Logging --> Enable logging) and the Hyades Data Collection Engine application or service is running.");

			//Create the timer scale:
			System.out.print("I");

			for (int counter = 0; counter < WAITTIME; counter++){
				System.out.print("-");
			}

			System.out.print("I\nI");

			//Increment the timer:
			try {

				for (int counter = 0; counter < WAITTIME; counter++) {

					System.out.print("-");

					Thread.sleep(1000);
				}
			}
			catch (InterruptedException i) {
			}

			System.out.println("I");

			System.out.println("Log4JLoggingTest successfully ended!");
		}
		catch (Throwable t) {
			System.out.println("ERROR - Log4JLoggingTest unsuccessfully ended!");
			System.out.println("REASON: " + t.getMessage());
		}
	}
}
