
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.analysis.engine.IAnalysisEngine;
import org.eclipse.hyades.analysis.engine.IAnalysisMonitor;
import org.eclipse.hyades.analysis.engine.ILogAnalyzer;
import org.eclipse.hyades.analysis.engine.Incident;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.hyades.sdb.internal.SDbPlugin;
import org.eclipse.hyades.ui.internal.util.PreferencesUtil;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogAnalyzerImpl.java,v 1.5 2005/05/12 16:39:09 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Sample class to demonstrate the capability of creating a log analyzer.
 * <p>
 * A log analyzer uses an analysis engine to analyze log records within a log
 * file against a symptom database.
 * <p>
 * 
 * 
 * @author Nellie Chau
 * @author Paul E. Slauenwhite
 * @version September 15, 2004
 * @since September 15, 2004
 * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer
 */
public class LogAnalyzerImpl implements ILogAnalyzer {

    private String errorMessage = null;

    private ArrayList analysisEngines = null;

    /**
     * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#analyze(java.lang.Object,
     *      org.eclipse.hyades.analysis.engine.IAnalysisMonitor)
     */
    public String analyze(Object records, IAnalysisMonitor monitor) {

       errorMessage = null;

        if (analysisEngines != null) {

            try {

                if (records instanceof List) {

                    Iterator iterator = ((List) records).iterator();
                    final int TOTAL_RECORDS = ((List) records).size();
                    int recordCount = 0;
                    Object record = null;

                    while (iterator.hasNext()) {

                        record = iterator.next();

                        if (record instanceof CBECommonBaseEvent) {
                            analyze(((CBECommonBaseEvent) (record)), ((CBECommonBaseEvent) (record)).getMsg());
                        }

                        if (monitor != null) {

                            monitor.worked(1);
                            monitor.subTask("Record " + (++recordCount) + " of " + TOTAL_RECORDS);

                            if (monitor.isCanceled()) {
                                break;
                            }
                        }
                    }
                } else if (records instanceof CBECommonBaseEvent) {

                    analyze(((CBECommonBaseEvent) (records)), ((CBECommonBaseEvent) (records)).getMsg());

                    if (monitor != null) {

                        monitor.worked(1);
                        monitor.subTask("Record 1 of 1");
                    }
                }
            } catch (Exception e) {
                errorMessage = e.toString();
            }
        }

        return null;
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#loadDatabase()
     */
    public void loadDatabase() {

       errorMessage = null;

        try {
            
        String symptomDatabasePath = SDbPlugin.getDefault().getPreferenceStore().getString(PreferencesUtil.SYMPTOM_DB_PATH_KEY);

        if ((symptomDatabasePath != null) && (!symptomDatabasePath.trim().equals(""))) {

            StringTokenizer tokens = new StringTokenizer(symptomDatabasePath.trim(), ";");
            String token = null;
            IAnalysisEngine analysisEngine = null;

            while (tokens.hasMoreTokens()) {

                token = tokens.nextToken().trim();

                if (token.endsWith("1")) {
                    
                    IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(new Path(token.substring(0,token.lastIndexOf(','))));
                    
                    if ((resource != null) && (resource.exists())) {

                        if(analysisEngines == null){
                            analysisEngines = new ArrayList();
                        }

                        analysisEngine = new AnalysisEngineImpl();
                        analysisEngine.replaceSymptomDatabase(resource.getFullPath().toOSString());
                        
                        analysisEngines.add(analysisEngine);
                    }
                }
            }
        }
        } 
        catch (Exception e) {
            errorMessage = e.toString();
        }
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#unloadDatabase()
     */
    public void unloadDatabase() {

        errorMessage = null;

        analysisEngines = null;
    }

    /**
     * @see org.eclipse.hyades.analysis.engine.ILogAnalyzer#errorMsg()
     */
    public String errorMsg() {
        return errorMessage;
    }

    private void analyze(CBECommonBaseEvent commonBaseEvent, String symptomString) {

        commonBaseEvent.getSymptoms().clear();
        commonBaseEvent.setAnalyzed(true);

        Iterator iterator = analysisEngines.iterator();
        Object[] directives = null;

        while (iterator.hasNext()) {

            directives = ((IAnalysisEngine) (iterator.next())).analyze(new Incident(symptomString));

            if (directives != null) {
                commonBaseEvent.getSymptoms().addAll(Arrays.asList(directives));
            }
        }
    }
}