/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GenerateWizard.java,v 1.9 2005/03/16 13:13:46 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.java.internal.junit.wizard;

import java.lang.reflect.InvocationTargetException;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.TestCommonImages;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.codegen.Generator;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.hyades.test.java.internal.codegen.JUnitGenerator;
import org.eclipse.hyades.test.java.internal.util.TestJavaUtil;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.ui.internal.wizard.HyadesWizard;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;

public class GenerateWizard 
extends HyadesWizard implements IRunnableWithProgress
{
	public final static String PLUGIN_ID_JUNIT = "org.junit"; //$NON-NLS-1$
	public final static String PLUGIN_ID_EXECUTION_CORE = "org.eclipse.hyades.execution.core"; //$NON-NLS-1$
	public final static String PLUGIN_ID_EXECUTION_REMOTE = "org.eclipse.hyades.execution.remote"; //$NON-NLS-1$

	private ITestSuite testSuite;
	private AdjustSourceInfoPage adjustSourceInfoPage;
	private ProjectAndSourcePage projectAndSourcePage;

	/**
	 * Constructor for GenerateWizard
	 */
	public GenerateWizard()
	{
		super();

		setWindowTitle(TestJavaPlugin.getString("GEN_WTITLE")); //$NON-NLS-1$
		setDefaultPageImageDescriptor(TestCommonImages.INSTANCE.getImageDescriptor(TestCommonImages.IMG_WIZBAN_GENERATE));

		IDialogSettings dialogSettings = TestJavaPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("junit.GenerateWizard"); //$NON-NLS-1$
		if (settings == null)
			settings = dialogSettings.addNewSection("junit.GenerateWizard"); //$NON-NLS-1$
		setDialogSettings(settings);
	}

	public ITestSuite getTestSuite()
	{
		return testSuite;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
	 */
	protected void initPages()
	{
		IStructuredSelection structuredSelection = getSelection();
		if(structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof ITestSuite) {
				testSuite = (ITestSuite)selection;
			} else if (selection instanceof ITestSuiteProxyNode) {
				ITestSuiteProxyNode tsProxy = (ITestSuiteProxyNode) selection;
				//- load the test suite from the proxy
				testSuite = tsProxy.getTestSuite();
			}
		}
	
		IImplementor implementor = getTestSuite().getImplementor();
		
		IJavaProject[] javaProjects = TestJavaUtil.getJavaProjects();
		projectAndSourcePage = new ProjectAndSourcePage(implementor, javaProjects);
		
		boolean isOK = false;
		String pack = null;
		String cls = null;
		if(javaProjects.length > 0)
		{
			if(implementor.getResource() != null)
			{
				String resource = implementor.getResource();
				int index = resource.lastIndexOf('.');
				if(index > 0)
				{
					pack = resource.substring(0, index);
					isOK = JavaConventions.validatePackageName(pack).isOK();
				}
				
				if(index < (resource.length()-1))
				{
					cls = resource.substring(index+1);
					isOK = isOK && JavaConventions.validateIdentifier(cls).isOK();
				}
				
				isOK = isOK && (pack != null) && (cls != null);
			}
			
			if(!isOK)
				adjustSourceInfoPage = new AdjustSourceInfoPage(implementor);
		}
	}

	/**
	 * @see Wizard#addPages()
	 */
	public void addPages()
	{
		if((projectAndSourcePage.getJavaProjects().length > 0) && (adjustSourceInfoPage != null))
			addPage(adjustSourceInfoPage);
			
		addPage(projectAndSourcePage);
	}

	/**
	 * Asks the user whether overwriting an existing generated code is desired.
	 * @return
	 */
	protected boolean confirmOverwrite() {
		MessageDialog dialog = new MessageDialog(
				getShell(), TestJavaPlugin.getString("W_QUESTION"),  //$NON-NLS-1$
				null, TestJavaPlugin.getString("Q_OVR_FILE"),  //$NON-NLS-1$
				MessageDialog.QUESTION, 
				new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 1);
		return dialog.open() == IDialogConstants.OK_ID;
	}
	
	public void run(IProgressMonitor mon)
	throws InterruptedException, InvocationTargetException
	{
		getDialogSettings().put(ProjectAndSourcePage.SET_PROJECT, getJavaProject().getElementName());
		getDialogSettings().put(ProjectAndSourcePage.SET_SOURCE, getSourceFolder());

		if(getTestSuite() != null)
		{							
			Resource resource = ((EObject)getTestSuite()).eResource();
			boolean isTrackingModification = resource.isTrackingModification();
			resource.setTrackingModification(true);
			
			IContainer location = getJavaProject().getProject();
			if((getSourceFolder() != null) && (getSourceFolder().length() > 0))
				location = location.getFolder(new Path(getSourceFolder()));
			
			String sLocation = location.getFullPath().toString();
			if(!sLocation.equals(getTestSuite().getImplementor().getLocation()))
				getTestSuite().getImplementor().setLocation(sLocation);
			else
				resource.setModified(adjustSourceInfoPage != null);
			
			Generator generator = createGenerator();
			if(generator.getFileHandle(getTestSuite()).exists())
			{
				if(!confirmOverwrite())
				{
					mon.setCanceled(true);
					throw new InterruptedException();
				}
			}
				
			addRequiredLibraries(generator);		
			try
			{
				generator.generate(getTestSuite(), mon);
			}
			catch(Exception e)
			{
				TestJavaPlugin.logError(e);
				throw new InvocationTargetException(e);
			}
			
			if(resource.isModified())
			{
				try
				{
					EMFUtil.save(resource);
				}
				catch (Exception e)
				{
					TestJavaPlugin.logError(e);
				}
			}
			resource.setTrackingModification(isTrackingModification);
		}
	}

	public boolean performFinish()
	{
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(this);
		try
		{
			getContainer().run(false, true, op);
		}
		catch(InvocationTargetException e)
		{
			MessageDialog.openError(getShell(), TestJavaPlugin.getString("W_ERROR"), e.getTargetException().getMessage()); //$NON-NLS-1$
			return false;
		}
		catch (InterruptedException e)
		{
			return false;
		}

		return true;
	}

   /**
	* @return Java Project
	*/
	public IJavaProject getJavaProject()
	{
		return projectAndSourcePage.getJavaProject();
	}

   /**
	* @return source folder
	*/
	public String getSourceFolder()
	{
		return projectAndSourcePage.getSourceFolder();
	}
	
	protected Generator createGenerator()
	{
		return new JUnitGenerator();
	}
	
	protected void addRequiredLibraries(Generator generator)
	{
		JUnitGenerator junitGenerator = (JUnitGenerator)generator;
		junitGenerator.addAllLibraries(PLUGIN_ID_JUNIT);
		junitGenerator.addRequiredLibrary(TestCommonPlugin.getID(), "common.runner.jar"); //$NON-NLS-1$
		junitGenerator.addRequiredLibrary(TestJavaPlugin.getID(), "java.runner.jar"); //$NON-NLS-1$
	}
}